<?php

namespace App\Extension\defaultExt\modules\Opportunities\Process\Service\RecordActions;

use ApiPlatform\Core\Exception\InvalidArgumentException;
use App\Engine\LegacyHandler\LegacyHandler;
use App\Process\Entity\Process;
use App\Process\Service\ProcessHandlerInterface;

class LinkAccountAction extends LegacyHandler implements ProcessHandlerInterface
{
    protected const MSG_OPTIONS_NOT_FOUND = 'Process options is not defined';
    protected const PROCESS_TYPE = 'record-opportunity-link-account';

    /**
     * @inheritDoc
     */
    public function getProcessType(): string
    {
        return self::PROCESS_TYPE;
    }


    public function getHandlerKey(): string
    {
        return $this->getProcessType();
    }

    /**
     * @inheritDoc
     */
    public function requiredAuthRole(): string
    {
        return 'ROLE_USER';
    }

    /**
     * @inheritDoc
     */
    public function getRequiredACLs(Process $process): array
    {
        $options = $process->getOptions();
        $module = $options['module'] ?? '';

        $acls = [
            $module => [
                [
                    'action' => 'view',
                    'record' => $options['id'] ?? ''
                ]
            ]
        ];

        $modalRecord = $options['params']['modalRecord'] ?? [];
        $modalRecordModule = $modalRecord['module'] ?? '';
        $modalRecordId = $modalRecord['id'] ?? '';

        if ($modalRecordModule !== '') {
            $acls[$modalRecordModule] = [
                [
                    'action' => 'view',
                    'record' => $modalRecordId
                ]
            ];
        }

        return $acls;
    }

    /**
     * @inheritDoc
     */
    public function configure(Process $process): void
    {
        //This process is synchronous
        //We aren't going to store a record on db
        //thus we will use process type as the id
        $process->setId(self::PROCESS_TYPE);
        $process->setAsync(false);
    }

    /**
     * @inheritDoc
     *
     */
    public function validate(Process $process): void
    {
        if (empty($process->getOptions())) {
            throw new InvalidArgumentException(self::MSG_OPTIONS_NOT_FOUND);
        }

        $options = $process->getOptions();
        ['module' => $baseModule, 'id' => $id] = $options;

        if (empty($id) || $baseModule !== 'opportunities') {
            throw new InvalidArgumentException(self::MSG_OPTIONS_NOT_FOUND);
        }

        ['modalRecord' => $modalRecord] = $options['params'];
        ['module' => $modalModule, 'id' => $modalId] = $modalRecord;

        if (empty($modalId) || $modalModule !== 'accounts') {
            throw new InvalidArgumentException(self::MSG_OPTIONS_NOT_FOUND);
        }
    }

    /**
     * @inheritDoc
     */
    public function run(Process $process)
    {
        $options = $process->getOptions();
        ['id' => $opportunityId] = $options;
        ['modalRecord' => $modalRecord] = $options['params'];
        ['id' => $accountId] = $modalRecord;

        $result = $this->linkAccount($opportunityId, $accountId);

        $process->setStatus($result['status']);
        $process->setMessages($result['messages']);
        $process->setData($result['data']);
    }

    protected function linkAccount(string $opportunityId, string $accountId): array
    {
        // 1. initialize legacy handler
        $this->init();

        // 2. Retrieve the Account and the Opportunity
        /** @var \Opportunity $opportunity */
        $opportunity = \BeanFactory::getBean('Opportunities', $opportunityId);

        /** @var \Account $account */
        $account = \BeanFactory::getBean('Accounts', $accountId);

        if (empty($account)) {
            $this->close();

            return [
                'status' => 'error',
                'data' => [],
                'messages' => [
                    'LBL_RECORD_NOT_FOUND'
                ]
            ];
        }

        // 3. Link the account to opportunity and save
        $opportunity->account_id = $accountId;

        $opportunity->save();

        // 3. Prepare response and request frontend to reload
        $responseData = [

            'reload' => true,

            // OTHER AVAILABLE OPTIONS
            // 'reloadRecentlyViewed' => true,
            // 'reloadFavorites' => true,

            // 'handler' => 'export', // options: export, redirect, audit

            // Params vary according to the handler that is used
            // 'params' => [
            //     'route' => $options['module'],
            //     'queryParams' => []
            // ]
        ];

        $result = [
            'status' => 'success',
            'data' => $responseData,
            'messages' => [
                'LBL_LINK_RELATIONSHIP_SUCCESS'
            ]
        ];

        // Close legacy handler
        $this->close();

        return $result;
    }
}
