"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.hasNgModuleImport = hasNgModuleImport;
const schematics_1 = require("@angular-devkit/schematics");
const ts = require("typescript");
/**
 * Whether the Angular module in the given path imports the specified module class name.
 */
function hasNgModuleImport(tree, modulePath, className) {
    const moduleFileContent = tree.read(modulePath);
    if (!moduleFileContent) {
        throw new schematics_1.SchematicsException(`Could not read Angular module file: ${modulePath}`);
    }
    const parsedFile = ts.createSourceFile(modulePath, moduleFileContent.toString(), ts.ScriptTarget.Latest, true);
    const ngModuleMetadata = findNgModuleMetadata(parsedFile);
    if (!ngModuleMetadata) {
        throw new schematics_1.SchematicsException(`Could not find NgModule declaration inside: "${modulePath}"`);
    }
    for (let property of ngModuleMetadata.properties) {
        if (!ts.isPropertyAssignment(property) ||
            property.name.getText() !== 'imports' ||
            !ts.isArrayLiteralExpression(property.initializer)) {
            continue;
        }
        if (property.initializer.elements.some(element => element.getText() === className)) {
            return true;
        }
    }
    return false;
}
/**
 * Resolves the last identifier that is part of the given expression. This helps resolving
 * identifiers of nested property access expressions (e.g. myNamespace.core.NgModule).
 */
function resolveIdentifierOfExpression(expression) {
    if (ts.isIdentifier(expression)) {
        return expression;
    }
    else if (ts.isPropertyAccessExpression(expression) && ts.isIdentifier(expression.name)) {
        return expression.name;
    }
    return null;
}
/**
 * Finds a NgModule declaration within the specified TypeScript node and returns the
 * corresponding metadata for it. This function searches breadth first because
 * NgModule's are usually not nested within other expressions or declarations.
 */
function findNgModuleMetadata(rootNode) {
    // Add immediate child nodes of the root node to the queue.
    const nodeQueue = [...rootNode.getChildren()];
    while (nodeQueue.length) {
        const node = nodeQueue.shift();
        if (ts.isDecorator(node) &&
            ts.isCallExpression(node.expression) &&
            isNgModuleCallExpression(node.expression)) {
            return node.expression.arguments[0];
        }
        else {
            nodeQueue.push(...node.getChildren());
        }
    }
    return null;
}
/** Whether the specified call expression is referring to a NgModule definition. */
function isNgModuleCallExpression(callExpression) {
    if (!callExpression.arguments.length ||
        !ts.isObjectLiteralExpression(callExpression.arguments[0])) {
        return false;
    }
    // The `NgModule` call expression name is never referring to a `PrivateIdentifier`.
    const decoratorIdentifier = resolveIdentifierOfExpression(callExpression.expression);
    return decoratorIdentifier ? decoratorIdentifier.text === 'NgModule' : false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmctbW9kdWxlLWltcG9ydHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9zcmMvY2RrL3NjaGVtYXRpY3MvdXRpbHMvYXN0L25nLW1vZHVsZS1pbXBvcnRzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7O0dBTUc7O0FBUUgsOENBa0NDO0FBeENELDJEQUFxRTtBQUNyRSxpQ0FBaUM7QUFFakM7O0dBRUc7QUFDSCxTQUFnQixpQkFBaUIsQ0FBQyxJQUFVLEVBQUUsVUFBa0IsRUFBRSxTQUFpQjtJQUNqRixNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7SUFFaEQsSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUM7UUFDdkIsTUFBTSxJQUFJLGdDQUFtQixDQUFDLHVDQUF1QyxVQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFFRCxNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUMsZ0JBQWdCLENBQ3BDLFVBQVUsRUFDVixpQkFBaUIsQ0FBQyxRQUFRLEVBQUUsRUFDNUIsRUFBRSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQ3RCLElBQUksQ0FDTCxDQUFDO0lBQ0YsTUFBTSxnQkFBZ0IsR0FBRyxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUUxRCxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztRQUN0QixNQUFNLElBQUksZ0NBQW1CLENBQUMsZ0RBQWdELFVBQVUsR0FBRyxDQUFDLENBQUM7SUFDL0YsQ0FBQztJQUVELEtBQUssSUFBSSxRQUFRLElBQUksZ0JBQWlCLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDbEQsSUFDRSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLENBQUM7WUFDbEMsUUFBUSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsS0FBSyxTQUFTO1lBQ3JDLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsRUFDbEQsQ0FBQztZQUNELFNBQVM7UUFDWCxDQUFDO1FBRUQsSUFBSSxRQUFRLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEtBQUssU0FBUyxDQUFDLEVBQUUsQ0FBQztZQUNuRixPQUFPLElBQUksQ0FBQztRQUNkLENBQUM7SUFDSCxDQUFDO0lBRUQsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBRUQ7OztHQUdHO0FBQ0gsU0FBUyw2QkFBNkIsQ0FBQyxVQUF5QjtJQUM5RCxJQUFJLEVBQUUsQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQztRQUNoQyxPQUFPLFVBQVUsQ0FBQztJQUNwQixDQUFDO1NBQU0sSUFBSSxFQUFFLENBQUMsMEJBQTBCLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztRQUN6RixPQUFPLFVBQVUsQ0FBQyxJQUFJLENBQUM7SUFDekIsQ0FBQztJQUNELE9BQU8sSUFBSSxDQUFDO0FBQ2QsQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLG9CQUFvQixDQUFDLFFBQWlCO0lBQzdDLDJEQUEyRDtJQUMzRCxNQUFNLFNBQVMsR0FBYyxDQUFDLEdBQUcsUUFBUSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7SUFFekQsT0FBTyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDeEIsTUFBTSxJQUFJLEdBQUcsU0FBUyxDQUFDLEtBQUssRUFBRyxDQUFDO1FBRWhDLElBQ0UsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUM7WUFDcEIsRUFBRSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxVQUFVLENBQUM7WUFDcEMsd0JBQXdCLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUN6QyxDQUFDO1lBQ0QsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQStCLENBQUM7UUFDcEUsQ0FBQzthQUFNLENBQUM7WUFDTixTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFDeEMsQ0FBQztJQUNILENBQUM7SUFFRCxPQUFPLElBQUksQ0FBQztBQUNkLENBQUM7QUFFRCxtRkFBbUY7QUFDbkYsU0FBUyx3QkFBd0IsQ0FBQyxjQUFpQztJQUNqRSxJQUNFLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxNQUFNO1FBQ2hDLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFDMUQsQ0FBQztRQUNELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVELG1GQUFtRjtJQUNuRixNQUFNLG1CQUFtQixHQUFHLDZCQUE2QixDQUFDLGNBQWMsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNyRixPQUFPLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEtBQUssVUFBVSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7QUFDL0UsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQge1NjaGVtYXRpY3NFeGNlcHRpb24sIFRyZWV9IGZyb20gJ0Bhbmd1bGFyLWRldmtpdC9zY2hlbWF0aWNzJztcbmltcG9ydCAqIGFzIHRzIGZyb20gJ3R5cGVzY3JpcHQnO1xuXG4vKipcbiAqIFdoZXRoZXIgdGhlIEFuZ3VsYXIgbW9kdWxlIGluIHRoZSBnaXZlbiBwYXRoIGltcG9ydHMgdGhlIHNwZWNpZmllZCBtb2R1bGUgY2xhc3MgbmFtZS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGhhc05nTW9kdWxlSW1wb3J0KHRyZWU6IFRyZWUsIG1vZHVsZVBhdGg6IHN0cmluZywgY2xhc3NOYW1lOiBzdHJpbmcpOiBib29sZWFuIHtcbiAgY29uc3QgbW9kdWxlRmlsZUNvbnRlbnQgPSB0cmVlLnJlYWQobW9kdWxlUGF0aCk7XG5cbiAgaWYgKCFtb2R1bGVGaWxlQ29udGVudCkge1xuICAgIHRocm93IG5ldyBTY2hlbWF0aWNzRXhjZXB0aW9uKGBDb3VsZCBub3QgcmVhZCBBbmd1bGFyIG1vZHVsZSBmaWxlOiAke21vZHVsZVBhdGh9YCk7XG4gIH1cblxuICBjb25zdCBwYXJzZWRGaWxlID0gdHMuY3JlYXRlU291cmNlRmlsZShcbiAgICBtb2R1bGVQYXRoLFxuICAgIG1vZHVsZUZpbGVDb250ZW50LnRvU3RyaW5nKCksXG4gICAgdHMuU2NyaXB0VGFyZ2V0LkxhdGVzdCxcbiAgICB0cnVlLFxuICApO1xuICBjb25zdCBuZ01vZHVsZU1ldGFkYXRhID0gZmluZE5nTW9kdWxlTWV0YWRhdGEocGFyc2VkRmlsZSk7XG5cbiAgaWYgKCFuZ01vZHVsZU1ldGFkYXRhKSB7XG4gICAgdGhyb3cgbmV3IFNjaGVtYXRpY3NFeGNlcHRpb24oYENvdWxkIG5vdCBmaW5kIE5nTW9kdWxlIGRlY2xhcmF0aW9uIGluc2lkZTogXCIke21vZHVsZVBhdGh9XCJgKTtcbiAgfVxuXG4gIGZvciAobGV0IHByb3BlcnR5IG9mIG5nTW9kdWxlTWV0YWRhdGEhLnByb3BlcnRpZXMpIHtcbiAgICBpZiAoXG4gICAgICAhdHMuaXNQcm9wZXJ0eUFzc2lnbm1lbnQocHJvcGVydHkpIHx8XG4gICAgICBwcm9wZXJ0eS5uYW1lLmdldFRleHQoKSAhPT0gJ2ltcG9ydHMnIHx8XG4gICAgICAhdHMuaXNBcnJheUxpdGVyYWxFeHByZXNzaW9uKHByb3BlcnR5LmluaXRpYWxpemVyKVxuICAgICkge1xuICAgICAgY29udGludWU7XG4gICAgfVxuXG4gICAgaWYgKHByb3BlcnR5LmluaXRpYWxpemVyLmVsZW1lbnRzLnNvbWUoZWxlbWVudCA9PiBlbGVtZW50LmdldFRleHQoKSA9PT0gY2xhc3NOYW1lKSkge1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIGZhbHNlO1xufVxuXG4vKipcbiAqIFJlc29sdmVzIHRoZSBsYXN0IGlkZW50aWZpZXIgdGhhdCBpcyBwYXJ0IG9mIHRoZSBnaXZlbiBleHByZXNzaW9uLiBUaGlzIGhlbHBzIHJlc29sdmluZ1xuICogaWRlbnRpZmllcnMgb2YgbmVzdGVkIHByb3BlcnR5IGFjY2VzcyBleHByZXNzaW9ucyAoZS5nLiBteU5hbWVzcGFjZS5jb3JlLk5nTW9kdWxlKS5cbiAqL1xuZnVuY3Rpb24gcmVzb2x2ZUlkZW50aWZpZXJPZkV4cHJlc3Npb24oZXhwcmVzc2lvbjogdHMuRXhwcmVzc2lvbik6IHRzLklkZW50aWZpZXIgfCBudWxsIHtcbiAgaWYgKHRzLmlzSWRlbnRpZmllcihleHByZXNzaW9uKSkge1xuICAgIHJldHVybiBleHByZXNzaW9uO1xuICB9IGVsc2UgaWYgKHRzLmlzUHJvcGVydHlBY2Nlc3NFeHByZXNzaW9uKGV4cHJlc3Npb24pICYmIHRzLmlzSWRlbnRpZmllcihleHByZXNzaW9uLm5hbWUpKSB7XG4gICAgcmV0dXJuIGV4cHJlc3Npb24ubmFtZTtcbiAgfVxuICByZXR1cm4gbnVsbDtcbn1cblxuLyoqXG4gKiBGaW5kcyBhIE5nTW9kdWxlIGRlY2xhcmF0aW9uIHdpdGhpbiB0aGUgc3BlY2lmaWVkIFR5cGVTY3JpcHQgbm9kZSBhbmQgcmV0dXJucyB0aGVcbiAqIGNvcnJlc3BvbmRpbmcgbWV0YWRhdGEgZm9yIGl0LiBUaGlzIGZ1bmN0aW9uIHNlYXJjaGVzIGJyZWFkdGggZmlyc3QgYmVjYXVzZVxuICogTmdNb2R1bGUncyBhcmUgdXN1YWxseSBub3QgbmVzdGVkIHdpdGhpbiBvdGhlciBleHByZXNzaW9ucyBvciBkZWNsYXJhdGlvbnMuXG4gKi9cbmZ1bmN0aW9uIGZpbmROZ01vZHVsZU1ldGFkYXRhKHJvb3ROb2RlOiB0cy5Ob2RlKTogdHMuT2JqZWN0TGl0ZXJhbEV4cHJlc3Npb24gfCBudWxsIHtcbiAgLy8gQWRkIGltbWVkaWF0ZSBjaGlsZCBub2RlcyBvZiB0aGUgcm9vdCBub2RlIHRvIHRoZSBxdWV1ZS5cbiAgY29uc3Qgbm9kZVF1ZXVlOiB0cy5Ob2RlW10gPSBbLi4ucm9vdE5vZGUuZ2V0Q2hpbGRyZW4oKV07XG5cbiAgd2hpbGUgKG5vZGVRdWV1ZS5sZW5ndGgpIHtcbiAgICBjb25zdCBub2RlID0gbm9kZVF1ZXVlLnNoaWZ0KCkhO1xuXG4gICAgaWYgKFxuICAgICAgdHMuaXNEZWNvcmF0b3Iobm9kZSkgJiZcbiAgICAgIHRzLmlzQ2FsbEV4cHJlc3Npb24obm9kZS5leHByZXNzaW9uKSAmJlxuICAgICAgaXNOZ01vZHVsZUNhbGxFeHByZXNzaW9uKG5vZGUuZXhwcmVzc2lvbilcbiAgICApIHtcbiAgICAgIHJldHVybiBub2RlLmV4cHJlc3Npb24uYXJndW1lbnRzWzBdIGFzIHRzLk9iamVjdExpdGVyYWxFeHByZXNzaW9uO1xuICAgIH0gZWxzZSB7XG4gICAgICBub2RlUXVldWUucHVzaCguLi5ub2RlLmdldENoaWxkcmVuKCkpO1xuICAgIH1cbiAgfVxuXG4gIHJldHVybiBudWxsO1xufVxuXG4vKiogV2hldGhlciB0aGUgc3BlY2lmaWVkIGNhbGwgZXhwcmVzc2lvbiBpcyByZWZlcnJpbmcgdG8gYSBOZ01vZHVsZSBkZWZpbml0aW9uLiAqL1xuZnVuY3Rpb24gaXNOZ01vZHVsZUNhbGxFeHByZXNzaW9uKGNhbGxFeHByZXNzaW9uOiB0cy5DYWxsRXhwcmVzc2lvbik6IGJvb2xlYW4ge1xuICBpZiAoXG4gICAgIWNhbGxFeHByZXNzaW9uLmFyZ3VtZW50cy5sZW5ndGggfHxcbiAgICAhdHMuaXNPYmplY3RMaXRlcmFsRXhwcmVzc2lvbihjYWxsRXhwcmVzc2lvbi5hcmd1bWVudHNbMF0pXG4gICkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIC8vIFRoZSBgTmdNb2R1bGVgIGNhbGwgZXhwcmVzc2lvbiBuYW1lIGlzIG5ldmVyIHJlZmVycmluZyB0byBhIGBQcml2YXRlSWRlbnRpZmllcmAuXG4gIGNvbnN0IGRlY29yYXRvcklkZW50aWZpZXIgPSByZXNvbHZlSWRlbnRpZmllck9mRXhwcmVzc2lvbihjYWxsRXhwcmVzc2lvbi5leHByZXNzaW9uKTtcbiAgcmV0dXJuIGRlY29yYXRvcklkZW50aWZpZXIgPyBkZWNvcmF0b3JJZGVudGlmaWVyLnRleHQgPT09ICdOZ01vZHVsZScgOiBmYWxzZTtcbn1cbiJdfQ==