"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultComponentOptions = getDefaultComponentOptions;
exports.isStandaloneSchematic = isStandaloneSchematic;
const core_1 = require("@angular-devkit/core");
const schema_1 = require("@schematics/angular/component/schema");
const ng_ast_utils_1 = require("@schematics/angular/utility/ng-ast-utils");
const project_main_file_1 = require("./project-main-file");
const workspace_1 = require("@schematics/angular/utility/workspace");
const get_project_1 = require("./get-project");
/**
 * Returns the default options for the `@schematics/angular:component` schematic which would
 * have been specified at project initialization (ng new or ng init).
 *
 * This is necessary because the Angular CLI only exposes the default values for the "--style",
 * "--inlineStyle", "--skipTests" and "--inlineTemplate" options to the "component" schematic.
 */
function getDefaultComponentOptions(project) {
    // Note: Not all options which are available when running "ng new" will be stored in the
    // workspace config. List of options which will be available in the configuration:
    // angular/angular-cli/blob/main/packages/schematics/angular/application/index.ts#L109-L131
    let skipTests = getDefaultComponentOption(project, ['skipTests'], null);
    // In case "skipTests" is not set explicitly, also look for the "spec" option. The "spec"
    // option has been deprecated but can be still used in older Angular CLI projects.
    // See: https://github.com/angular/angular-cli/commit/a12a4e02a4689b5bdbc6e740c0d9865afb55671a
    if (skipTests === null) {
        skipTests = !getDefaultComponentOption(project, ['spec'], true);
    }
    return {
        style: getDefaultComponentOption(project, ['style', 'styleext'], schema_1.Style.Css),
        inlineStyle: getDefaultComponentOption(project, ['inlineStyle'], false),
        inlineTemplate: getDefaultComponentOption(project, ['inlineTemplate'], false),
        skipTests: skipTests,
    };
}
/** Determines whether the schematic is configured to be standalone. */
async function isStandaloneSchematic(host, options) {
    if (options.standalone != null) {
        return options.standalone;
    }
    // If the `--standalone` flag isn't passed and there isn't a default, infer based on the project.
    const workspace = await (0, workspace_1.getWorkspace)(host);
    const project = (0, get_project_1.getProjectFromWorkspace)(workspace, options.project);
    // Legacy projects might not have a `build` target, but they're likely
    // not on an Angular version that supports standalone either.
    if (!project.targets?.has('build')) {
        return false;
    }
    return (0, ng_ast_utils_1.isStandaloneApp)(host, (0, project_main_file_1.getProjectMainFile)(project));
}
/**
 * Gets the default value for the specified option. The default options will be determined
 * by looking at the stored schematic options for `@schematics/angular:component` in the
 * CLI workspace configuration.
 */
function getDefaultComponentOption(project, optionNames, fallbackValue) {
    const schematicOptions = (0, core_1.isJsonObject)(project.extensions['schematics'] || null)
        ? project.extensions['schematics']
        : null;
    const defaultSchematic = schematicOptions
        ? schematicOptions['@schematics/angular:component']
        : null;
    for (const optionName of optionNames) {
        if (defaultSchematic && defaultSchematic[optionName] != null) {
            return defaultSchematic[optionName];
        }
    }
    return fallbackValue;
}
//# sourceMappingURL=data:application/json;base64,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