/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import * as core from '../../core';
import { DEFAULT_INTERPOLATION_CONFIG } from '../../ml_parser/defaults';
import * as o from '../../output/output_ast';
import { ParseLocation, ParseSourceFile, ParseSourceSpan } from '../../parse_util';
import { RecursiveVisitor, visitAll } from '../r3_ast';
import { Identifiers as R3 } from '../r3_identifiers';
import { generateForwardRef } from '../util';
import { R3TemplateDependencyKind, } from '../view/api';
import { createComponentType } from '../view/compiler';
import { DefinitionMap } from '../view/util';
import { createDirectiveDefinitionMap } from './directive';
import { toOptionalLiteralArray } from './util';
/**
 * Compile a component declaration defined by the `R3ComponentMetadata`.
 */
export function compileDeclareComponentFromMetadata(meta, template, additionalTemplateInfo) {
    const definitionMap = createComponentDefinitionMap(meta, template, additionalTemplateInfo);
    const expression = o.importExpr(R3.declareComponent).callFn([definitionMap.toLiteralMap()]);
    const type = createComponentType(meta);
    return { expression, type, statements: [] };
}
/**
 * Gathers the declaration fields for a component into a `DefinitionMap`.
 */
export function createComponentDefinitionMap(meta, template, templateInfo) {
    const definitionMap = createDirectiveDefinitionMap(meta);
    const blockVisitor = new BlockPresenceVisitor();
    visitAll(blockVisitor, template.nodes);
    definitionMap.set('template', getTemplateExpression(template, templateInfo));
    if (templateInfo.isInline) {
        definitionMap.set('isInline', o.literal(true));
    }
    // Set the minVersion to 17.0.0 if the component is using at least one block in its template.
    // We don't do this for templates without blocks, in order to preserve backwards compatibility.
    if (blockVisitor.hasBlocks) {
        definitionMap.set('minVersion', o.literal('17.0.0'));
    }
    definitionMap.set('styles', toOptionalLiteralArray(meta.styles, o.literal));
    definitionMap.set('dependencies', compileUsedDependenciesMetadata(meta));
    definitionMap.set('viewProviders', meta.viewProviders);
    definitionMap.set('animations', meta.animations);
    if (meta.changeDetection !== null) {
        if (typeof meta.changeDetection === 'object') {
            throw new Error('Impossible state! Change detection flag is not resolved!');
        }
        definitionMap.set('changeDetection', o
            .importExpr(R3.ChangeDetectionStrategy)
            .prop(core.ChangeDetectionStrategy[meta.changeDetection]));
    }
    if (meta.encapsulation !== core.ViewEncapsulation.Emulated) {
        definitionMap.set('encapsulation', o.importExpr(R3.ViewEncapsulation).prop(core.ViewEncapsulation[meta.encapsulation]));
    }
    if (meta.interpolation !== DEFAULT_INTERPOLATION_CONFIG) {
        definitionMap.set('interpolation', o.literalArr([o.literal(meta.interpolation.start), o.literal(meta.interpolation.end)]));
    }
    if (template.preserveWhitespaces === true) {
        definitionMap.set('preserveWhitespaces', o.literal(true));
    }
    if (meta.defer.mode === 0 /* DeferBlockDepsEmitMode.PerBlock */) {
        const resolvers = [];
        let hasResolvers = false;
        for (const deps of meta.defer.blocks.values()) {
            // Note: we need to push a `null` even if there are no dependencies, because matching of
            // defer resolver functions to defer blocks happens by index and not adding an array
            // entry for a block can throw off the blocks coming after it.
            if (deps === null) {
                resolvers.push(o.literal(null));
            }
            else {
                resolvers.push(deps);
                hasResolvers = true;
            }
        }
        // If *all* the resolvers are null, we can skip the field.
        if (hasResolvers) {
            definitionMap.set('deferBlockDependencies', o.literalArr(resolvers));
        }
    }
    else {
        throw new Error('Unsupported defer function emit mode in partial compilation');
    }
    return definitionMap;
}
function getTemplateExpression(template, templateInfo) {
    // If the template has been defined using a direct literal, we use that expression directly
    // without any modifications. This is ensures proper source mapping from the partially
    // compiled code to the source file declaring the template. Note that this does not capture
    // template literals referenced indirectly through an identifier.
    if (templateInfo.inlineTemplateLiteralExpression !== null) {
        return templateInfo.inlineTemplateLiteralExpression;
    }
    // If the template is defined inline but not through a literal, the template has been resolved
    // through static interpretation. We create a literal but cannot provide any source span. Note
    // that we cannot use the expression defining the template because the linker expects the template
    // to be defined as a literal in the declaration.
    if (templateInfo.isInline) {
        return o.literal(templateInfo.content, null, null);
    }
    // The template is external so we must synthesize an expression node with
    // the appropriate source-span.
    const contents = templateInfo.content;
    const file = new ParseSourceFile(contents, templateInfo.sourceUrl);
    const start = new ParseLocation(file, 0, 0, 0);
    const end = computeEndLocation(file, contents);
    const span = new ParseSourceSpan(start, end);
    return o.literal(contents, null, span);
}
function computeEndLocation(file, contents) {
    const length = contents.length;
    let lineStart = 0;
    let lastLineStart = 0;
    let line = 0;
    do {
        lineStart = contents.indexOf('\n', lastLineStart);
        if (lineStart !== -1) {
            lastLineStart = lineStart + 1;
            line++;
        }
    } while (lineStart !== -1);
    return new ParseLocation(file, length, line, length - lastLineStart);
}
function compileUsedDependenciesMetadata(meta) {
    const wrapType = meta.declarationListEmitMode !== 0 /* DeclarationListEmitMode.Direct */
        ? generateForwardRef
        : (expr) => expr;
    if (meta.declarationListEmitMode === 3 /* DeclarationListEmitMode.RuntimeResolved */) {
        throw new Error(`Unsupported emit mode`);
    }
    return toOptionalLiteralArray(meta.declarations, (decl) => {
        switch (decl.kind) {
            case R3TemplateDependencyKind.Directive:
                const dirMeta = new DefinitionMap();
                dirMeta.set('kind', o.literal(decl.isComponent ? 'component' : 'directive'));
                dirMeta.set('type', wrapType(decl.type));
                dirMeta.set('selector', o.literal(decl.selector));
                dirMeta.set('inputs', toOptionalLiteralArray(decl.inputs, o.literal));
                dirMeta.set('outputs', toOptionalLiteralArray(decl.outputs, o.literal));
                dirMeta.set('exportAs', toOptionalLiteralArray(decl.exportAs, o.literal));
                return dirMeta.toLiteralMap();
            case R3TemplateDependencyKind.Pipe:
                const pipeMeta = new DefinitionMap();
                pipeMeta.set('kind', o.literal('pipe'));
                pipeMeta.set('type', wrapType(decl.type));
                pipeMeta.set('name', o.literal(decl.name));
                return pipeMeta.toLiteralMap();
            case R3TemplateDependencyKind.NgModule:
                const ngModuleMeta = new DefinitionMap();
                ngModuleMeta.set('kind', o.literal('ngmodule'));
                ngModuleMeta.set('type', wrapType(decl.type));
                return ngModuleMeta.toLiteralMap();
        }
    });
}
class BlockPresenceVisitor extends RecursiveVisitor {
    constructor() {
        super(...arguments);
        this.hasBlocks = false;
    }
    visitDeferredBlock() {
        this.hasBlocks = true;
    }
    visitDeferredBlockPlaceholder() {
        this.hasBlocks = true;
    }
    visitDeferredBlockLoading() {
        this.hasBlocks = true;
    }
    visitDeferredBlockError() {
        this.hasBlocks = true;
    }
    visitIfBlock() {
        this.hasBlocks = true;
    }
    visitIfBlockBranch() {
        this.hasBlocks = true;
    }
    visitForLoopBlock() {
        this.hasBlocks = true;
    }
    visitForLoopBlockEmpty() {
        this.hasBlocks = true;
    }
    visitSwitchBlock() {
        this.hasBlocks = true;
    }
    visitSwitchBlockCase() {
        this.hasBlocks = true;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcG9uZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvY29tcGlsZXIvc3JjL3JlbmRlcjMvcGFydGlhbC9jb21wb25lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBQ0gsT0FBTyxLQUFLLElBQUksTUFBTSxZQUFZLENBQUM7QUFDbkMsT0FBTyxFQUFDLDRCQUE0QixFQUFDLE1BQU0sMEJBQTBCLENBQUM7QUFDdEUsT0FBTyxLQUFLLENBQUMsTUFBTSx5QkFBeUIsQ0FBQztBQUM3QyxPQUFPLEVBQUMsYUFBYSxFQUFFLGVBQWUsRUFBRSxlQUFlLEVBQUMsTUFBTSxrQkFBa0IsQ0FBQztBQUNqRixPQUFPLEVBQUMsZ0JBQWdCLEVBQUUsUUFBUSxFQUFDLE1BQU0sV0FBVyxDQUFDO0FBQ3JELE9BQU8sRUFBQyxXQUFXLElBQUksRUFBRSxFQUFDLE1BQU0sbUJBQW1CLENBQUM7QUFDcEQsT0FBTyxFQUFDLGtCQUFrQixFQUF1QixNQUFNLFNBQVMsQ0FBQztBQUNqRSxPQUFPLEVBSUwsd0JBQXdCLEdBRXpCLE1BQU0sYUFBYSxDQUFDO0FBQ3JCLE9BQU8sRUFBQyxtQkFBbUIsRUFBQyxNQUFNLGtCQUFrQixDQUFDO0FBRXJELE9BQU8sRUFBQyxhQUFhLEVBQUMsTUFBTSxjQUFjLENBQUM7QUFRM0MsT0FBTyxFQUFDLDRCQUE0QixFQUFDLE1BQU0sYUFBYSxDQUFDO0FBQ3pELE9BQU8sRUFBQyxzQkFBc0IsRUFBQyxNQUFNLFFBQVEsQ0FBQztBQStCOUM7O0dBRUc7QUFDSCxNQUFNLFVBQVUsbUNBQW1DLENBQ2pELElBQXVELEVBQ3ZELFFBQXdCLEVBQ3hCLHNCQUFvRDtJQUVwRCxNQUFNLGFBQWEsR0FBRyw0QkFBNEIsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFLHNCQUFzQixDQUFDLENBQUM7SUFFM0YsTUFBTSxVQUFVLEdBQUcsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxhQUFhLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQzVGLE1BQU0sSUFBSSxHQUFHLG1CQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBRXZDLE9BQU8sRUFBQyxVQUFVLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxFQUFFLEVBQUMsQ0FBQztBQUM1QyxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxNQUFNLFVBQVUsNEJBQTRCLENBQzFDLElBQXVELEVBQ3ZELFFBQXdCLEVBQ3hCLFlBQTBDO0lBRTFDLE1BQU0sYUFBYSxHQUNqQiw0QkFBNEIsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNyQyxNQUFNLFlBQVksR0FBRyxJQUFJLG9CQUFvQixFQUFFLENBQUM7SUFDaEQsUUFBUSxDQUFDLFlBQVksRUFBRSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFdkMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxVQUFVLEVBQUUscUJBQXFCLENBQUMsUUFBUSxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUM7SUFFN0UsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDMUIsYUFBYSxDQUFDLEdBQUcsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRCw2RkFBNkY7SUFDN0YsK0ZBQStGO0lBQy9GLElBQUksWUFBWSxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQzNCLGFBQWEsQ0FBQyxHQUFHLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsc0JBQXNCLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUM1RSxhQUFhLENBQUMsR0FBRyxDQUFDLGNBQWMsRUFBRSwrQkFBK0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ3pFLGFBQWEsQ0FBQyxHQUFHLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUN2RCxhQUFhLENBQUMsR0FBRyxDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7SUFFakQsSUFBSSxJQUFJLENBQUMsZUFBZSxLQUFLLElBQUksRUFBRSxDQUFDO1FBQ2xDLElBQUksT0FBTyxJQUFJLENBQUMsZUFBZSxLQUFLLFFBQVEsRUFBRSxDQUFDO1lBQzdDLE1BQU0sSUFBSSxLQUFLLENBQUMsMERBQTBELENBQUMsQ0FBQztRQUM5RSxDQUFDO1FBRUQsYUFBYSxDQUFDLEdBQUcsQ0FDZixpQkFBaUIsRUFDakIsQ0FBQzthQUNFLFVBQVUsQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUM7YUFDdEMsSUFBSSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FDNUQsQ0FBQztJQUNKLENBQUM7SUFDRCxJQUFJLElBQUksQ0FBQyxhQUFhLEtBQUssSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzNELGFBQWEsQ0FBQyxHQUFHLENBQ2YsZUFBZSxFQUNmLENBQUMsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FDcEYsQ0FBQztJQUNKLENBQUM7SUFDRCxJQUFJLElBQUksQ0FBQyxhQUFhLEtBQUssNEJBQTRCLEVBQUUsQ0FBQztRQUN4RCxhQUFhLENBQUMsR0FBRyxDQUNmLGVBQWUsRUFDZixDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQ3ZGLENBQUM7SUFDSixDQUFDO0lBRUQsSUFBSSxRQUFRLENBQUMsbUJBQW1CLEtBQUssSUFBSSxFQUFFLENBQUM7UUFDMUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQUVELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLDRDQUFvQyxFQUFFLENBQUM7UUFDeEQsTUFBTSxTQUFTLEdBQW1CLEVBQUUsQ0FBQztRQUNyQyxJQUFJLFlBQVksR0FBRyxLQUFLLENBQUM7UUFFekIsS0FBSyxNQUFNLElBQUksSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDO1lBQzlDLHdGQUF3RjtZQUN4RixvRkFBb0Y7WUFDcEYsOERBQThEO1lBQzlELElBQUksSUFBSSxLQUFLLElBQUksRUFBRSxDQUFDO2dCQUNsQixTQUFTLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztZQUNsQyxDQUFDO2lCQUFNLENBQUM7Z0JBQ04sU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDckIsWUFBWSxHQUFHLElBQUksQ0FBQztZQUN0QixDQUFDO1FBQ0gsQ0FBQztRQUNELDBEQUEwRDtRQUMxRCxJQUFJLFlBQVksRUFBRSxDQUFDO1lBQ2pCLGFBQWEsQ0FBQyxHQUFHLENBQUMsd0JBQXdCLEVBQUUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLENBQUM7SUFDSCxDQUFDO1NBQU0sQ0FBQztRQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQsT0FBTyxhQUFhLENBQUM7QUFDdkIsQ0FBQztBQUVELFNBQVMscUJBQXFCLENBQzVCLFFBQXdCLEVBQ3hCLFlBQTBDO0lBRTFDLDJGQUEyRjtJQUMzRixzRkFBc0Y7SUFDdEYsMkZBQTJGO0lBQzNGLGlFQUFpRTtJQUNqRSxJQUFJLFlBQVksQ0FBQywrQkFBK0IsS0FBSyxJQUFJLEVBQUUsQ0FBQztRQUMxRCxPQUFPLFlBQVksQ0FBQywrQkFBK0IsQ0FBQztJQUN0RCxDQUFDO0lBRUQsOEZBQThGO0lBQzlGLDhGQUE4RjtJQUM5RixrR0FBa0c7SUFDbEcsaURBQWlEO0lBQ2pELElBQUksWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzFCLE9BQU8sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQseUVBQXlFO0lBQ3pFLCtCQUErQjtJQUMvQixNQUFNLFFBQVEsR0FBRyxZQUFZLENBQUMsT0FBTyxDQUFDO0lBQ3RDLE1BQU0sSUFBSSxHQUFHLElBQUksZUFBZSxDQUFDLFFBQVEsRUFBRSxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbkUsTUFBTSxLQUFLLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDL0MsTUFBTSxHQUFHLEdBQUcsa0JBQWtCLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQy9DLE1BQU0sSUFBSSxHQUFHLElBQUksZUFBZSxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztJQUM3QyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztBQUN6QyxDQUFDO0FBRUQsU0FBUyxrQkFBa0IsQ0FBQyxJQUFxQixFQUFFLFFBQWdCO0lBQ2pFLE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUM7SUFDL0IsSUFBSSxTQUFTLEdBQUcsQ0FBQyxDQUFDO0lBQ2xCLElBQUksYUFBYSxHQUFHLENBQUMsQ0FBQztJQUN0QixJQUFJLElBQUksR0FBRyxDQUFDLENBQUM7SUFDYixHQUFHLENBQUM7UUFDRixTQUFTLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDbEQsSUFBSSxTQUFTLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUNyQixhQUFhLEdBQUcsU0FBUyxHQUFHLENBQUMsQ0FBQztZQUM5QixJQUFJLEVBQUUsQ0FBQztRQUNULENBQUM7SUFDSCxDQUFDLFFBQVEsU0FBUyxLQUFLLENBQUMsQ0FBQyxFQUFFO0lBRTNCLE9BQU8sSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsTUFBTSxHQUFHLGFBQWEsQ0FBQyxDQUFDO0FBQ3ZFLENBQUM7QUFFRCxTQUFTLCtCQUErQixDQUN0QyxJQUF1RDtJQUV2RCxNQUFNLFFBQVEsR0FDWixJQUFJLENBQUMsdUJBQXVCLDJDQUFtQztRQUM3RCxDQUFDLENBQUMsa0JBQWtCO1FBQ3BCLENBQUMsQ0FBQyxDQUFDLElBQWtCLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQztJQUVuQyxJQUFJLElBQUksQ0FBQyx1QkFBdUIsb0RBQTRDLEVBQUUsQ0FBQztRQUM3RSxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVELE9BQU8sc0JBQXNCLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ3hELFFBQVEsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2xCLEtBQUssd0JBQXdCLENBQUMsU0FBUztnQkFDckMsTUFBTSxPQUFPLEdBQUcsSUFBSSxhQUFhLEVBQXdDLENBQUM7Z0JBQzFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO2dCQUM3RSxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7Z0JBQ3pDLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7Z0JBQ2xELE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLHNCQUFzQixDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7Z0JBQ3RFLE9BQU8sQ0FBQyxHQUFHLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7Z0JBQ3hFLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxFQUFFLHNCQUFzQixDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7Z0JBQzFFLE9BQU8sT0FBTyxDQUFDLFlBQVksRUFBRSxDQUFDO1lBQ2hDLEtBQUssd0JBQXdCLENBQUMsSUFBSTtnQkFDaEMsTUFBTSxRQUFRLEdBQUcsSUFBSSxhQUFhLEVBQW1DLENBQUM7Z0JBQ3RFLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztnQkFDeEMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO2dCQUMxQyxRQUFRLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO2dCQUMzQyxPQUFPLFFBQVEsQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUNqQyxLQUFLLHdCQUF3QixDQUFDLFFBQVE7Z0JBQ3BDLE1BQU0sWUFBWSxHQUFHLElBQUksYUFBYSxFQUF1QyxDQUFDO2dCQUM5RSxZQUFZLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7Z0JBQ2hELFlBQVksQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztnQkFDOUMsT0FBTyxZQUFZLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDdkMsQ0FBQztJQUNILENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELE1BQU0sb0JBQXFCLFNBQVEsZ0JBQWdCO0lBQW5EOztRQUNFLGNBQVMsR0FBRyxLQUFLLENBQUM7SUF5Q3BCLENBQUM7SUF2Q1Usa0JBQWtCO1FBQ3pCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDO0lBQ3hCLENBQUM7SUFFUSw2QkFBNkI7UUFDcEMsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7SUFDeEIsQ0FBQztJQUVRLHlCQUF5QjtRQUNoQyxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQztJQUN4QixDQUFDO0lBRVEsdUJBQXVCO1FBQzlCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDO0lBQ3hCLENBQUM7SUFFUSxZQUFZO1FBQ25CLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDO0lBQ3hCLENBQUM7SUFFUSxrQkFBa0I7UUFDekIsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7SUFDeEIsQ0FBQztJQUVRLGlCQUFpQjtRQUN4QixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQztJQUN4QixDQUFDO0lBRVEsc0JBQXNCO1FBQzdCLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDO0lBQ3hCLENBQUM7SUFFUSxnQkFBZ0I7UUFDdkIsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7SUFDeEIsQ0FBQztJQUVRLG9CQUFvQjtRQUMzQixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQztJQUN4QixDQUFDO0NBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIExMQyBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5kZXYvbGljZW5zZVxuICovXG5pbXBvcnQgKiBhcyBjb3JlIGZyb20gJy4uLy4uL2NvcmUnO1xuaW1wb3J0IHtERUZBVUxUX0lOVEVSUE9MQVRJT05fQ09ORklHfSBmcm9tICcuLi8uLi9tbF9wYXJzZXIvZGVmYXVsdHMnO1xuaW1wb3J0ICogYXMgbyBmcm9tICcuLi8uLi9vdXRwdXQvb3V0cHV0X2FzdCc7XG5pbXBvcnQge1BhcnNlTG9jYXRpb24sIFBhcnNlU291cmNlRmlsZSwgUGFyc2VTb3VyY2VTcGFufSBmcm9tICcuLi8uLi9wYXJzZV91dGlsJztcbmltcG9ydCB7UmVjdXJzaXZlVmlzaXRvciwgdmlzaXRBbGx9IGZyb20gJy4uL3IzX2FzdCc7XG5pbXBvcnQge0lkZW50aWZpZXJzIGFzIFIzfSBmcm9tICcuLi9yM19pZGVudGlmaWVycyc7XG5pbXBvcnQge2dlbmVyYXRlRm9yd2FyZFJlZiwgUjNDb21waWxlZEV4cHJlc3Npb259IGZyb20gJy4uL3V0aWwnO1xuaW1wb3J0IHtcbiAgRGVjbGFyYXRpb25MaXN0RW1pdE1vZGUsXG4gIERlZmVyQmxvY2tEZXBzRW1pdE1vZGUsXG4gIFIzQ29tcG9uZW50TWV0YWRhdGEsXG4gIFIzVGVtcGxhdGVEZXBlbmRlbmN5S2luZCxcbiAgUjNUZW1wbGF0ZURlcGVuZGVuY3lNZXRhZGF0YSxcbn0gZnJvbSAnLi4vdmlldy9hcGknO1xuaW1wb3J0IHtjcmVhdGVDb21wb25lbnRUeXBlfSBmcm9tICcuLi92aWV3L2NvbXBpbGVyJztcbmltcG9ydCB7UGFyc2VkVGVtcGxhdGV9IGZyb20gJy4uL3ZpZXcvdGVtcGxhdGUnO1xuaW1wb3J0IHtEZWZpbml0aW9uTWFwfSBmcm9tICcuLi92aWV3L3V0aWwnO1xuXG5pbXBvcnQge1xuICBSM0RlY2xhcmVDb21wb25lbnRNZXRhZGF0YSxcbiAgUjNEZWNsYXJlRGlyZWN0aXZlRGVwZW5kZW5jeU1ldGFkYXRhLFxuICBSM0RlY2xhcmVOZ01vZHVsZURlcGVuZGVuY3lNZXRhZGF0YSxcbiAgUjNEZWNsYXJlUGlwZURlcGVuZGVuY3lNZXRhZGF0YSxcbn0gZnJvbSAnLi9hcGknO1xuaW1wb3J0IHtjcmVhdGVEaXJlY3RpdmVEZWZpbml0aW9uTWFwfSBmcm9tICcuL2RpcmVjdGl2ZSc7XG5pbXBvcnQge3RvT3B0aW9uYWxMaXRlcmFsQXJyYXl9IGZyb20gJy4vdXRpbCc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRGVjbGFyZUNvbXBvbmVudFRlbXBsYXRlSW5mbyB7XG4gIC8qKlxuICAgKiBUaGUgc3RyaW5nIGNvbnRlbnRzIG9mIHRoZSB0ZW1wbGF0ZS5cbiAgICpcbiAgICogVGhpcyBpcyB0aGUgXCJsb2dpY2FsXCIgdGVtcGxhdGUgc3RyaW5nLCBhZnRlciBleHBhbnNpb24gb2YgYW55IGVzY2FwZWQgY2hhcmFjdGVycyAoZm9yIGlubGluZVxuICAgKiB0ZW1wbGF0ZXMpLiBUaGlzIG1heSBkaWZmZXIgZnJvbSB0aGUgYWN0dWFsIHRlbXBsYXRlIGJ5dGVzIGFzIHRoZXkgYXBwZWFyIGluIHRoZSAudHMgZmlsZS5cbiAgICovXG4gIGNvbnRlbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogQSBmdWxsIHBhdGggdG8gdGhlIGZpbGUgd2hpY2ggY29udGFpbnMgdGhlIHRlbXBsYXRlLlxuICAgKlxuICAgKiBUaGlzIGNhbiBiZSBlaXRoZXIgdGhlIG9yaWdpbmFsIC50cyBmaWxlIGlmIHRoZSB0ZW1wbGF0ZSBpcyBpbmxpbmUsIG9yIHRoZSAuaHRtbCBmaWxlIGlmIGFuXG4gICAqIGV4dGVybmFsIGZpbGUgd2FzIHVzZWQuXG4gICAqL1xuICBzb3VyY2VVcmw6IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgdGVtcGxhdGUgd2FzIGlubGluZSAodXNpbmcgYHRlbXBsYXRlYCkgb3IgZXh0ZXJuYWwgKHVzaW5nIGB0ZW1wbGF0ZVVybGApLlxuICAgKi9cbiAgaXNJbmxpbmU6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIElmIHRoZSB0ZW1wbGF0ZSB3YXMgZGVmaW5lZCBpbmxpbmUgYnkgYSBkaXJlY3Qgc3RyaW5nIGxpdGVyYWwsIHRoZW4gdGhpcyBpcyB0aGF0IGxpdGVyYWxcbiAgICogZXhwcmVzc2lvbi4gT3RoZXJ3aXNlIGBudWxsYCwgaWYgdGhlIHRlbXBsYXRlIHdhcyBub3QgZGVmaW5lZCBpbmxpbmUgb3Igd2FzIG5vdCBhIGxpdGVyYWwuXG4gICAqL1xuICBpbmxpbmVUZW1wbGF0ZUxpdGVyYWxFeHByZXNzaW9uOiBvLkV4cHJlc3Npb24gfCBudWxsO1xufVxuXG4vKipcbiAqIENvbXBpbGUgYSBjb21wb25lbnQgZGVjbGFyYXRpb24gZGVmaW5lZCBieSB0aGUgYFIzQ29tcG9uZW50TWV0YWRhdGFgLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY29tcGlsZURlY2xhcmVDb21wb25lbnRGcm9tTWV0YWRhdGEoXG4gIG1ldGE6IFIzQ29tcG9uZW50TWV0YWRhdGE8UjNUZW1wbGF0ZURlcGVuZGVuY3lNZXRhZGF0YT4sXG4gIHRlbXBsYXRlOiBQYXJzZWRUZW1wbGF0ZSxcbiAgYWRkaXRpb25hbFRlbXBsYXRlSW5mbzogRGVjbGFyZUNvbXBvbmVudFRlbXBsYXRlSW5mbyxcbik6IFIzQ29tcGlsZWRFeHByZXNzaW9uIHtcbiAgY29uc3QgZGVmaW5pdGlvbk1hcCA9IGNyZWF0ZUNvbXBvbmVudERlZmluaXRpb25NYXAobWV0YSwgdGVtcGxhdGUsIGFkZGl0aW9uYWxUZW1wbGF0ZUluZm8pO1xuXG4gIGNvbnN0IGV4cHJlc3Npb24gPSBvLmltcG9ydEV4cHIoUjMuZGVjbGFyZUNvbXBvbmVudCkuY2FsbEZuKFtkZWZpbml0aW9uTWFwLnRvTGl0ZXJhbE1hcCgpXSk7XG4gIGNvbnN0IHR5cGUgPSBjcmVhdGVDb21wb25lbnRUeXBlKG1ldGEpO1xuXG4gIHJldHVybiB7ZXhwcmVzc2lvbiwgdHlwZSwgc3RhdGVtZW50czogW119O1xufVxuXG4vKipcbiAqIEdhdGhlcnMgdGhlIGRlY2xhcmF0aW9uIGZpZWxkcyBmb3IgYSBjb21wb25lbnQgaW50byBhIGBEZWZpbml0aW9uTWFwYC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUNvbXBvbmVudERlZmluaXRpb25NYXAoXG4gIG1ldGE6IFIzQ29tcG9uZW50TWV0YWRhdGE8UjNUZW1wbGF0ZURlcGVuZGVuY3lNZXRhZGF0YT4sXG4gIHRlbXBsYXRlOiBQYXJzZWRUZW1wbGF0ZSxcbiAgdGVtcGxhdGVJbmZvOiBEZWNsYXJlQ29tcG9uZW50VGVtcGxhdGVJbmZvLFxuKTogRGVmaW5pdGlvbk1hcDxSM0RlY2xhcmVDb21wb25lbnRNZXRhZGF0YT4ge1xuICBjb25zdCBkZWZpbml0aW9uTWFwOiBEZWZpbml0aW9uTWFwPFIzRGVjbGFyZUNvbXBvbmVudE1ldGFkYXRhPiA9XG4gICAgY3JlYXRlRGlyZWN0aXZlRGVmaW5pdGlvbk1hcChtZXRhKTtcbiAgY29uc3QgYmxvY2tWaXNpdG9yID0gbmV3IEJsb2NrUHJlc2VuY2VWaXNpdG9yKCk7XG4gIHZpc2l0QWxsKGJsb2NrVmlzaXRvciwgdGVtcGxhdGUubm9kZXMpO1xuXG4gIGRlZmluaXRpb25NYXAuc2V0KCd0ZW1wbGF0ZScsIGdldFRlbXBsYXRlRXhwcmVzc2lvbih0ZW1wbGF0ZSwgdGVtcGxhdGVJbmZvKSk7XG5cbiAgaWYgKHRlbXBsYXRlSW5mby5pc0lubGluZSkge1xuICAgIGRlZmluaXRpb25NYXAuc2V0KCdpc0lubGluZScsIG8ubGl0ZXJhbCh0cnVlKSk7XG4gIH1cblxuICAvLyBTZXQgdGhlIG1pblZlcnNpb24gdG8gMTcuMC4wIGlmIHRoZSBjb21wb25lbnQgaXMgdXNpbmcgYXQgbGVhc3Qgb25lIGJsb2NrIGluIGl0cyB0ZW1wbGF0ZS5cbiAgLy8gV2UgZG9uJ3QgZG8gdGhpcyBmb3IgdGVtcGxhdGVzIHdpdGhvdXQgYmxvY2tzLCBpbiBvcmRlciB0byBwcmVzZXJ2ZSBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eS5cbiAgaWYgKGJsb2NrVmlzaXRvci5oYXNCbG9ja3MpIHtcbiAgICBkZWZpbml0aW9uTWFwLnNldCgnbWluVmVyc2lvbicsIG8ubGl0ZXJhbCgnMTcuMC4wJykpO1xuICB9XG5cbiAgZGVmaW5pdGlvbk1hcC5zZXQoJ3N0eWxlcycsIHRvT3B0aW9uYWxMaXRlcmFsQXJyYXkobWV0YS5zdHlsZXMsIG8ubGl0ZXJhbCkpO1xuICBkZWZpbml0aW9uTWFwLnNldCgnZGVwZW5kZW5jaWVzJywgY29tcGlsZVVzZWREZXBlbmRlbmNpZXNNZXRhZGF0YShtZXRhKSk7XG4gIGRlZmluaXRpb25NYXAuc2V0KCd2aWV3UHJvdmlkZXJzJywgbWV0YS52aWV3UHJvdmlkZXJzKTtcbiAgZGVmaW5pdGlvbk1hcC5zZXQoJ2FuaW1hdGlvbnMnLCBtZXRhLmFuaW1hdGlvbnMpO1xuXG4gIGlmIChtZXRhLmNoYW5nZURldGVjdGlvbiAhPT0gbnVsbCkge1xuICAgIGlmICh0eXBlb2YgbWV0YS5jaGFuZ2VEZXRlY3Rpb24gPT09ICdvYmplY3QnKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ltcG9zc2libGUgc3RhdGUhIENoYW5nZSBkZXRlY3Rpb24gZmxhZyBpcyBub3QgcmVzb2x2ZWQhJyk7XG4gICAgfVxuXG4gICAgZGVmaW5pdGlvbk1hcC5zZXQoXG4gICAgICAnY2hhbmdlRGV0ZWN0aW9uJyxcbiAgICAgIG9cbiAgICAgICAgLmltcG9ydEV4cHIoUjMuQ2hhbmdlRGV0ZWN0aW9uU3RyYXRlZ3kpXG4gICAgICAgIC5wcm9wKGNvcmUuQ2hhbmdlRGV0ZWN0aW9uU3RyYXRlZ3lbbWV0YS5jaGFuZ2VEZXRlY3Rpb25dKSxcbiAgICApO1xuICB9XG4gIGlmIChtZXRhLmVuY2Fwc3VsYXRpb24gIT09IGNvcmUuVmlld0VuY2Fwc3VsYXRpb24uRW11bGF0ZWQpIHtcbiAgICBkZWZpbml0aW9uTWFwLnNldChcbiAgICAgICdlbmNhcHN1bGF0aW9uJyxcbiAgICAgIG8uaW1wb3J0RXhwcihSMy5WaWV3RW5jYXBzdWxhdGlvbikucHJvcChjb3JlLlZpZXdFbmNhcHN1bGF0aW9uW21ldGEuZW5jYXBzdWxhdGlvbl0pLFxuICAgICk7XG4gIH1cbiAgaWYgKG1ldGEuaW50ZXJwb2xhdGlvbiAhPT0gREVGQVVMVF9JTlRFUlBPTEFUSU9OX0NPTkZJRykge1xuICAgIGRlZmluaXRpb25NYXAuc2V0KFxuICAgICAgJ2ludGVycG9sYXRpb24nLFxuICAgICAgby5saXRlcmFsQXJyKFtvLmxpdGVyYWwobWV0YS5pbnRlcnBvbGF0aW9uLnN0YXJ0KSwgby5saXRlcmFsKG1ldGEuaW50ZXJwb2xhdGlvbi5lbmQpXSksXG4gICAgKTtcbiAgfVxuXG4gIGlmICh0ZW1wbGF0ZS5wcmVzZXJ2ZVdoaXRlc3BhY2VzID09PSB0cnVlKSB7XG4gICAgZGVmaW5pdGlvbk1hcC5zZXQoJ3ByZXNlcnZlV2hpdGVzcGFjZXMnLCBvLmxpdGVyYWwodHJ1ZSkpO1xuICB9XG5cbiAgaWYgKG1ldGEuZGVmZXIubW9kZSA9PT0gRGVmZXJCbG9ja0RlcHNFbWl0TW9kZS5QZXJCbG9jaykge1xuICAgIGNvbnN0IHJlc29sdmVyczogby5FeHByZXNzaW9uW10gPSBbXTtcbiAgICBsZXQgaGFzUmVzb2x2ZXJzID0gZmFsc2U7XG5cbiAgICBmb3IgKGNvbnN0IGRlcHMgb2YgbWV0YS5kZWZlci5ibG9ja3MudmFsdWVzKCkpIHtcbiAgICAgIC8vIE5vdGU6IHdlIG5lZWQgdG8gcHVzaCBhIGBudWxsYCBldmVuIGlmIHRoZXJlIGFyZSBubyBkZXBlbmRlbmNpZXMsIGJlY2F1c2UgbWF0Y2hpbmcgb2ZcbiAgICAgIC8vIGRlZmVyIHJlc29sdmVyIGZ1bmN0aW9ucyB0byBkZWZlciBibG9ja3MgaGFwcGVucyBieSBpbmRleCBhbmQgbm90IGFkZGluZyBhbiBhcnJheVxuICAgICAgLy8gZW50cnkgZm9yIGEgYmxvY2sgY2FuIHRocm93IG9mZiB0aGUgYmxvY2tzIGNvbWluZyBhZnRlciBpdC5cbiAgICAgIGlmIChkZXBzID09PSBudWxsKSB7XG4gICAgICAgIHJlc29sdmVycy5wdXNoKG8ubGl0ZXJhbChudWxsKSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXNvbHZlcnMucHVzaChkZXBzKTtcbiAgICAgICAgaGFzUmVzb2x2ZXJzID0gdHJ1ZTtcbiAgICAgIH1cbiAgICB9XG4gICAgLy8gSWYgKmFsbCogdGhlIHJlc29sdmVycyBhcmUgbnVsbCwgd2UgY2FuIHNraXAgdGhlIGZpZWxkLlxuICAgIGlmIChoYXNSZXNvbHZlcnMpIHtcbiAgICAgIGRlZmluaXRpb25NYXAuc2V0KCdkZWZlckJsb2NrRGVwZW5kZW5jaWVzJywgby5saXRlcmFsQXJyKHJlc29sdmVycykpO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ1Vuc3VwcG9ydGVkIGRlZmVyIGZ1bmN0aW9uIGVtaXQgbW9kZSBpbiBwYXJ0aWFsIGNvbXBpbGF0aW9uJyk7XG4gIH1cblxuICByZXR1cm4gZGVmaW5pdGlvbk1hcDtcbn1cblxuZnVuY3Rpb24gZ2V0VGVtcGxhdGVFeHByZXNzaW9uKFxuICB0ZW1wbGF0ZTogUGFyc2VkVGVtcGxhdGUsXG4gIHRlbXBsYXRlSW5mbzogRGVjbGFyZUNvbXBvbmVudFRlbXBsYXRlSW5mbyxcbik6IG8uRXhwcmVzc2lvbiB7XG4gIC8vIElmIHRoZSB0ZW1wbGF0ZSBoYXMgYmVlbiBkZWZpbmVkIHVzaW5nIGEgZGlyZWN0IGxpdGVyYWwsIHdlIHVzZSB0aGF0IGV4cHJlc3Npb24gZGlyZWN0bHlcbiAgLy8gd2l0aG91dCBhbnkgbW9kaWZpY2F0aW9ucy4gVGhpcyBpcyBlbnN1cmVzIHByb3BlciBzb3VyY2UgbWFwcGluZyBmcm9tIHRoZSBwYXJ0aWFsbHlcbiAgLy8gY29tcGlsZWQgY29kZSB0byB0aGUgc291cmNlIGZpbGUgZGVjbGFyaW5nIHRoZSB0ZW1wbGF0ZS4gTm90ZSB0aGF0IHRoaXMgZG9lcyBub3QgY2FwdHVyZVxuICAvLyB0ZW1wbGF0ZSBsaXRlcmFscyByZWZlcmVuY2VkIGluZGlyZWN0bHkgdGhyb3VnaCBhbiBpZGVudGlmaWVyLlxuICBpZiAodGVtcGxhdGVJbmZvLmlubGluZVRlbXBsYXRlTGl0ZXJhbEV4cHJlc3Npb24gIT09IG51bGwpIHtcbiAgICByZXR1cm4gdGVtcGxhdGVJbmZvLmlubGluZVRlbXBsYXRlTGl0ZXJhbEV4cHJlc3Npb247XG4gIH1cblxuICAvLyBJZiB0aGUgdGVtcGxhdGUgaXMgZGVmaW5lZCBpbmxpbmUgYnV0IG5vdCB0aHJvdWdoIGEgbGl0ZXJhbCwgdGhlIHRlbXBsYXRlIGhhcyBiZWVuIHJlc29sdmVkXG4gIC8vIHRocm91Z2ggc3RhdGljIGludGVycHJldGF0aW9uLiBXZSBjcmVhdGUgYSBsaXRlcmFsIGJ1dCBjYW5ub3QgcHJvdmlkZSBhbnkgc291cmNlIHNwYW4uIE5vdGVcbiAgLy8gdGhhdCB3ZSBjYW5ub3QgdXNlIHRoZSBleHByZXNzaW9uIGRlZmluaW5nIHRoZSB0ZW1wbGF0ZSBiZWNhdXNlIHRoZSBsaW5rZXIgZXhwZWN0cyB0aGUgdGVtcGxhdGVcbiAgLy8gdG8gYmUgZGVmaW5lZCBhcyBhIGxpdGVyYWwgaW4gdGhlIGRlY2xhcmF0aW9uLlxuICBpZiAodGVtcGxhdGVJbmZvLmlzSW5saW5lKSB7XG4gICAgcmV0dXJuIG8ubGl0ZXJhbCh0ZW1wbGF0ZUluZm8uY29udGVudCwgbnVsbCwgbnVsbCk7XG4gIH1cblxuICAvLyBUaGUgdGVtcGxhdGUgaXMgZXh0ZXJuYWwgc28gd2UgbXVzdCBzeW50aGVzaXplIGFuIGV4cHJlc3Npb24gbm9kZSB3aXRoXG4gIC8vIHRoZSBhcHByb3ByaWF0ZSBzb3VyY2Utc3Bhbi5cbiAgY29uc3QgY29udGVudHMgPSB0ZW1wbGF0ZUluZm8uY29udGVudDtcbiAgY29uc3QgZmlsZSA9IG5ldyBQYXJzZVNvdXJjZUZpbGUoY29udGVudHMsIHRlbXBsYXRlSW5mby5zb3VyY2VVcmwpO1xuICBjb25zdCBzdGFydCA9IG5ldyBQYXJzZUxvY2F0aW9uKGZpbGUsIDAsIDAsIDApO1xuICBjb25zdCBlbmQgPSBjb21wdXRlRW5kTG9jYXRpb24oZmlsZSwgY29udGVudHMpO1xuICBjb25zdCBzcGFuID0gbmV3IFBhcnNlU291cmNlU3BhbihzdGFydCwgZW5kKTtcbiAgcmV0dXJuIG8ubGl0ZXJhbChjb250ZW50cywgbnVsbCwgc3Bhbik7XG59XG5cbmZ1bmN0aW9uIGNvbXB1dGVFbmRMb2NhdGlvbihmaWxlOiBQYXJzZVNvdXJjZUZpbGUsIGNvbnRlbnRzOiBzdHJpbmcpOiBQYXJzZUxvY2F0aW9uIHtcbiAgY29uc3QgbGVuZ3RoID0gY29udGVudHMubGVuZ3RoO1xuICBsZXQgbGluZVN0YXJ0ID0gMDtcbiAgbGV0IGxhc3RMaW5lU3RhcnQgPSAwO1xuICBsZXQgbGluZSA9IDA7XG4gIGRvIHtcbiAgICBsaW5lU3RhcnQgPSBjb250ZW50cy5pbmRleE9mKCdcXG4nLCBsYXN0TGluZVN0YXJ0KTtcbiAgICBpZiAobGluZVN0YXJ0ICE9PSAtMSkge1xuICAgICAgbGFzdExpbmVTdGFydCA9IGxpbmVTdGFydCArIDE7XG4gICAgICBsaW5lKys7XG4gICAgfVxuICB9IHdoaWxlIChsaW5lU3RhcnQgIT09IC0xKTtcblxuICByZXR1cm4gbmV3IFBhcnNlTG9jYXRpb24oZmlsZSwgbGVuZ3RoLCBsaW5lLCBsZW5ndGggLSBsYXN0TGluZVN0YXJ0KTtcbn1cblxuZnVuY3Rpb24gY29tcGlsZVVzZWREZXBlbmRlbmNpZXNNZXRhZGF0YShcbiAgbWV0YTogUjNDb21wb25lbnRNZXRhZGF0YTxSM1RlbXBsYXRlRGVwZW5kZW5jeU1ldGFkYXRhPixcbik6IG8uTGl0ZXJhbEFycmF5RXhwciB8IG51bGwge1xuICBjb25zdCB3cmFwVHlwZSA9XG4gICAgbWV0YS5kZWNsYXJhdGlvbkxpc3RFbWl0TW9kZSAhPT0gRGVjbGFyYXRpb25MaXN0RW1pdE1vZGUuRGlyZWN0XG4gICAgICA/IGdlbmVyYXRlRm9yd2FyZFJlZlxuICAgICAgOiAoZXhwcjogby5FeHByZXNzaW9uKSA9PiBleHByO1xuXG4gIGlmIChtZXRhLmRlY2xhcmF0aW9uTGlzdEVtaXRNb2RlID09PSBEZWNsYXJhdGlvbkxpc3RFbWl0TW9kZS5SdW50aW1lUmVzb2x2ZWQpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIGVtaXQgbW9kZWApO1xuICB9XG5cbiAgcmV0dXJuIHRvT3B0aW9uYWxMaXRlcmFsQXJyYXkobWV0YS5kZWNsYXJhdGlvbnMsIChkZWNsKSA9PiB7XG4gICAgc3dpdGNoIChkZWNsLmtpbmQpIHtcbiAgICAgIGNhc2UgUjNUZW1wbGF0ZURlcGVuZGVuY3lLaW5kLkRpcmVjdGl2ZTpcbiAgICAgICAgY29uc3QgZGlyTWV0YSA9IG5ldyBEZWZpbml0aW9uTWFwPFIzRGVjbGFyZURpcmVjdGl2ZURlcGVuZGVuY3lNZXRhZGF0YT4oKTtcbiAgICAgICAgZGlyTWV0YS5zZXQoJ2tpbmQnLCBvLmxpdGVyYWwoZGVjbC5pc0NvbXBvbmVudCA/ICdjb21wb25lbnQnIDogJ2RpcmVjdGl2ZScpKTtcbiAgICAgICAgZGlyTWV0YS5zZXQoJ3R5cGUnLCB3cmFwVHlwZShkZWNsLnR5cGUpKTtcbiAgICAgICAgZGlyTWV0YS5zZXQoJ3NlbGVjdG9yJywgby5saXRlcmFsKGRlY2wuc2VsZWN0b3IpKTtcbiAgICAgICAgZGlyTWV0YS5zZXQoJ2lucHV0cycsIHRvT3B0aW9uYWxMaXRlcmFsQXJyYXkoZGVjbC5pbnB1dHMsIG8ubGl0ZXJhbCkpO1xuICAgICAgICBkaXJNZXRhLnNldCgnb3V0cHV0cycsIHRvT3B0aW9uYWxMaXRlcmFsQXJyYXkoZGVjbC5vdXRwdXRzLCBvLmxpdGVyYWwpKTtcbiAgICAgICAgZGlyTWV0YS5zZXQoJ2V4cG9ydEFzJywgdG9PcHRpb25hbExpdGVyYWxBcnJheShkZWNsLmV4cG9ydEFzLCBvLmxpdGVyYWwpKTtcbiAgICAgICAgcmV0dXJuIGRpck1ldGEudG9MaXRlcmFsTWFwKCk7XG4gICAgICBjYXNlIFIzVGVtcGxhdGVEZXBlbmRlbmN5S2luZC5QaXBlOlxuICAgICAgICBjb25zdCBwaXBlTWV0YSA9IG5ldyBEZWZpbml0aW9uTWFwPFIzRGVjbGFyZVBpcGVEZXBlbmRlbmN5TWV0YWRhdGE+KCk7XG4gICAgICAgIHBpcGVNZXRhLnNldCgna2luZCcsIG8ubGl0ZXJhbCgncGlwZScpKTtcbiAgICAgICAgcGlwZU1ldGEuc2V0KCd0eXBlJywgd3JhcFR5cGUoZGVjbC50eXBlKSk7XG4gICAgICAgIHBpcGVNZXRhLnNldCgnbmFtZScsIG8ubGl0ZXJhbChkZWNsLm5hbWUpKTtcbiAgICAgICAgcmV0dXJuIHBpcGVNZXRhLnRvTGl0ZXJhbE1hcCgpO1xuICAgICAgY2FzZSBSM1RlbXBsYXRlRGVwZW5kZW5jeUtpbmQuTmdNb2R1bGU6XG4gICAgICAgIGNvbnN0IG5nTW9kdWxlTWV0YSA9IG5ldyBEZWZpbml0aW9uTWFwPFIzRGVjbGFyZU5nTW9kdWxlRGVwZW5kZW5jeU1ldGFkYXRhPigpO1xuICAgICAgICBuZ01vZHVsZU1ldGEuc2V0KCdraW5kJywgby5saXRlcmFsKCduZ21vZHVsZScpKTtcbiAgICAgICAgbmdNb2R1bGVNZXRhLnNldCgndHlwZScsIHdyYXBUeXBlKGRlY2wudHlwZSkpO1xuICAgICAgICByZXR1cm4gbmdNb2R1bGVNZXRhLnRvTGl0ZXJhbE1hcCgpO1xuICAgIH1cbiAgfSk7XG59XG5cbmNsYXNzIEJsb2NrUHJlc2VuY2VWaXNpdG9yIGV4dGVuZHMgUmVjdXJzaXZlVmlzaXRvciB7XG4gIGhhc0Jsb2NrcyA9IGZhbHNlO1xuXG4gIG92ZXJyaWRlIHZpc2l0RGVmZXJyZWRCbG9jaygpOiB2b2lkIHtcbiAgICB0aGlzLmhhc0Jsb2NrcyA9IHRydWU7XG4gIH1cblxuICBvdmVycmlkZSB2aXNpdERlZmVycmVkQmxvY2tQbGFjZWhvbGRlcigpOiB2b2lkIHtcbiAgICB0aGlzLmhhc0Jsb2NrcyA9IHRydWU7XG4gIH1cblxuICBvdmVycmlkZSB2aXNpdERlZmVycmVkQmxvY2tMb2FkaW5nKCk6IHZvaWQge1xuICAgIHRoaXMuaGFzQmxvY2tzID0gdHJ1ZTtcbiAgfVxuXG4gIG92ZXJyaWRlIHZpc2l0RGVmZXJyZWRCbG9ja0Vycm9yKCk6IHZvaWQge1xuICAgIHRoaXMuaGFzQmxvY2tzID0gdHJ1ZTtcbiAgfVxuXG4gIG92ZXJyaWRlIHZpc2l0SWZCbG9jaygpOiB2b2lkIHtcbiAgICB0aGlzLmhhc0Jsb2NrcyA9IHRydWU7XG4gIH1cblxuICBvdmVycmlkZSB2aXNpdElmQmxvY2tCcmFuY2goKTogdm9pZCB7XG4gICAgdGhpcy5oYXNCbG9ja3MgPSB0cnVlO1xuICB9XG5cbiAgb3ZlcnJpZGUgdmlzaXRGb3JMb29wQmxvY2soKTogdm9pZCB7XG4gICAgdGhpcy5oYXNCbG9ja3MgPSB0cnVlO1xuICB9XG5cbiAgb3ZlcnJpZGUgdmlzaXRGb3JMb29wQmxvY2tFbXB0eSgpOiB2b2lkIHtcbiAgICB0aGlzLmhhc0Jsb2NrcyA9IHRydWU7XG4gIH1cblxuICBvdmVycmlkZSB2aXNpdFN3aXRjaEJsb2NrKCk6IHZvaWQge1xuICAgIHRoaXMuaGFzQmxvY2tzID0gdHJ1ZTtcbiAgfVxuXG4gIG92ZXJyaWRlIHZpc2l0U3dpdGNoQmxvY2tDYXNlKCk6IHZvaWQge1xuICAgIHRoaXMuaGFzQmxvY2tzID0gdHJ1ZTtcbiAgfVxufVxuIl19