/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { I18nContextKind, Namespace, OpKind, } from '../enums';
import { SlotHandle } from '../handle';
import { OpList } from '../operations';
import { TRAIT_CONSUMES_SLOT, TRAIT_CONSUMES_VARS, } from '../traits';
import { NEW_OP } from './shared';
/**
 * The set of OpKinds that represent the creation of an element or container
 */
const elementContainerOpKinds = new Set([
    OpKind.Element,
    OpKind.ElementStart,
    OpKind.Container,
    OpKind.ContainerStart,
    OpKind.Template,
    OpKind.RepeaterCreate,
]);
/**
 * Checks whether the given operation represents the creation of an element or container.
 */
export function isElementOrContainerOp(op) {
    return elementContainerOpKinds.has(op.kind);
}
/**
 * Create an `ElementStartOp`.
 */
export function createElementStartOp(tag, xref, namespace, i18nPlaceholder, startSourceSpan, wholeSourceSpan) {
    return {
        kind: OpKind.ElementStart,
        xref,
        tag,
        handle: new SlotHandle(),
        attributes: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        startSourceSpan,
        wholeSourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `TemplateOp`.
 */
export function createTemplateOp(xref, templateKind, tag, functionNameSuffix, namespace, i18nPlaceholder, startSourceSpan, wholeSourceSpan) {
    return {
        kind: OpKind.Template,
        xref,
        templateKind,
        attributes: null,
        tag,
        handle: new SlotHandle(),
        functionNameSuffix,
        decls: null,
        vars: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        startSourceSpan,
        wholeSourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
export function createRepeaterCreateOp(primaryView, emptyView, tag, track, varNames, emptyTag, i18nPlaceholder, emptyI18nPlaceholder, startSourceSpan, wholeSourceSpan) {
    return {
        kind: OpKind.RepeaterCreate,
        attributes: null,
        xref: primaryView,
        handle: new SlotHandle(),
        emptyView,
        track,
        trackByFn: null,
        tag,
        emptyTag,
        emptyAttributes: null,
        functionNameSuffix: 'For',
        namespace: Namespace.HTML,
        nonBindable: false,
        localRefs: [],
        decls: null,
        vars: null,
        varNames,
        usesComponentInstance: false,
        i18nPlaceholder,
        emptyI18nPlaceholder,
        startSourceSpan,
        wholeSourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
        ...TRAIT_CONSUMES_VARS,
        numSlotsUsed: emptyView === null ? 2 : 3,
    };
}
/**
 * Create an `ElementEndOp`.
 */
export function createElementEndOp(xref, sourceSpan) {
    return {
        kind: OpKind.ElementEnd,
        xref,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createDisableBindingsOp(xref) {
    return {
        kind: OpKind.DisableBindings,
        xref,
        ...NEW_OP,
    };
}
export function createEnableBindingsOp(xref) {
    return {
        kind: OpKind.EnableBindings,
        xref,
        ...NEW_OP,
    };
}
/**
 * Create a `TextOp`.
 */
export function createTextOp(xref, initialValue, icuPlaceholder, sourceSpan) {
    return {
        kind: OpKind.Text,
        xref,
        handle: new SlotHandle(),
        initialValue,
        icuPlaceholder,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `ListenerOp`. Host bindings reuse all the listener logic.
 */
export function createListenerOp(target, targetSlot, name, tag, handlerOps, animationPhase, eventTarget, hostListener, sourceSpan) {
    const handlerList = new OpList();
    handlerList.push(handlerOps);
    return {
        kind: OpKind.Listener,
        target,
        targetSlot,
        tag,
        hostListener,
        name,
        handlerOps: handlerList,
        handlerFnName: null,
        consumesDollarEvent: false,
        isAnimationListener: animationPhase !== null,
        animationPhase,
        eventTarget,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a `TwoWayListenerOp`.
 */
export function createTwoWayListenerOp(target, targetSlot, name, tag, handlerOps, sourceSpan) {
    const handlerList = new OpList();
    handlerList.push(handlerOps);
    return {
        kind: OpKind.TwoWayListener,
        target,
        targetSlot,
        tag,
        name,
        handlerOps: handlerList,
        handlerFnName: null,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createPipeOp(xref, slot, name) {
    return {
        kind: OpKind.Pipe,
        xref,
        handle: slot,
        name,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
export function createNamespaceOp(namespace) {
    return {
        kind: OpKind.Namespace,
        active: namespace,
        ...NEW_OP,
    };
}
export function createProjectionDefOp(def) {
    return {
        kind: OpKind.ProjectionDef,
        def,
        ...NEW_OP,
    };
}
export function createProjectionOp(xref, selector, i18nPlaceholder, fallbackView, sourceSpan) {
    return {
        kind: OpKind.Projection,
        xref,
        handle: new SlotHandle(),
        selector,
        i18nPlaceholder,
        fallbackView,
        projectionSlotIndex: 0,
        attributes: null,
        localRefs: [],
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
        numSlotsUsed: fallbackView === null ? 1 : 2,
    };
}
/**
 * Create an `ExtractedAttributeOp`.
 */
export function createExtractedAttributeOp(target, bindingKind, namespace, name, expression, i18nContext, i18nMessage, securityContext) {
    return {
        kind: OpKind.ExtractedAttribute,
        target,
        bindingKind,
        namespace,
        name,
        expression,
        i18nContext,
        i18nMessage,
        securityContext,
        trustedValueFn: null,
        ...NEW_OP,
    };
}
export function createDeferOp(xref, main, mainSlot, ownResolverFn, resolverFn, sourceSpan) {
    return {
        kind: OpKind.Defer,
        xref,
        handle: new SlotHandle(),
        mainView: main,
        mainSlot,
        loadingView: null,
        loadingSlot: null,
        loadingConfig: null,
        loadingMinimumTime: null,
        loadingAfterTime: null,
        placeholderView: null,
        placeholderSlot: null,
        placeholderConfig: null,
        placeholderMinimumTime: null,
        errorView: null,
        errorSlot: null,
        ownResolverFn,
        resolverFn,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
        numSlotsUsed: 2,
    };
}
export function createDeferOnOp(defer, trigger, prefetch, sourceSpan) {
    return {
        kind: OpKind.DeferOn,
        defer,
        trigger,
        prefetch,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates a `DeclareLetOp`.
 */
export function createDeclareLetOp(xref, declaredName, sourceSpan) {
    return {
        kind: OpKind.DeclareLet,
        xref,
        declaredName,
        sourceSpan,
        handle: new SlotHandle(),
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create an `ExtractedMessageOp`.
 */
export function createI18nMessageOp(xref, i18nContext, i18nBlock, message, messagePlaceholder, params, postprocessingParams, needsPostprocessing) {
    return {
        kind: OpKind.I18nMessage,
        xref,
        i18nContext,
        i18nBlock,
        message,
        messagePlaceholder,
        params,
        postprocessingParams,
        needsPostprocessing,
        subMessages: [],
        ...NEW_OP,
    };
}
/**
 * Create an `I18nStartOp`.
 */
export function createI18nStartOp(xref, message, root, sourceSpan) {
    return {
        kind: OpKind.I18nStart,
        xref,
        handle: new SlotHandle(),
        root: root ?? xref,
        message,
        messageIndex: null,
        subTemplateIndex: null,
        context: null,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
/**
 * Create an `I18nEndOp`.
 */
export function createI18nEndOp(xref, sourceSpan) {
    return {
        kind: OpKind.I18nEnd,
        xref,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU start op.
 */
export function createIcuStartOp(xref, message, messagePlaceholder, sourceSpan) {
    return {
        kind: OpKind.IcuStart,
        xref,
        message,
        messagePlaceholder,
        context: null,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU end op.
 */
export function createIcuEndOp(xref) {
    return {
        kind: OpKind.IcuEnd,
        xref,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU placeholder op.
 */
export function createIcuPlaceholderOp(xref, name, strings) {
    return {
        kind: OpKind.IcuPlaceholder,
        xref,
        name,
        strings,
        expressionPlaceholders: [],
        ...NEW_OP,
    };
}
export function createI18nContextOp(contextKind, xref, i18nBlock, message, sourceSpan) {
    if (i18nBlock === null && contextKind !== I18nContextKind.Attr) {
        throw new Error('AssertionError: i18nBlock must be provided for non-attribute contexts.');
    }
    return {
        kind: OpKind.I18nContext,
        contextKind,
        xref,
        i18nBlock,
        message,
        sourceSpan,
        params: new Map(),
        postprocessingParams: new Map(),
        ...NEW_OP,
    };
}
export function createI18nAttributesOp(xref, handle, target) {
    return {
        kind: OpKind.I18nAttributes,
        xref,
        handle,
        target,
        i18nAttributesConfig: null,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvY29tcGlsZXIvc3JjL3RlbXBsYXRlL3BpcGVsaW5lL2lyL3NyYy9vcHMvY3JlYXRlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQU1ILE9BQU8sRUFHTCxlQUFlLEVBRWYsU0FBUyxFQUNULE1BQU0sR0FFUCxNQUFNLFVBQVUsQ0FBQztBQUNsQixPQUFPLEVBQUMsVUFBVSxFQUFDLE1BQU0sV0FBVyxDQUFDO0FBQ3JDLE9BQU8sRUFBSyxNQUFNLEVBQVMsTUFBTSxlQUFlLENBQUM7QUFDakQsT0FBTyxFQUdMLG1CQUFtQixFQUNuQixtQkFBbUIsR0FDcEIsTUFBTSxXQUFXLENBQUM7QUFFbkIsT0FBTyxFQUFZLE1BQU0sRUFBMEIsTUFBTSxVQUFVLENBQUM7QUFxRHBFOztHQUVHO0FBQ0gsTUFBTSx1QkFBdUIsR0FBRyxJQUFJLEdBQUcsQ0FBQztJQUN0QyxNQUFNLENBQUMsT0FBTztJQUNkLE1BQU0sQ0FBQyxZQUFZO0lBQ25CLE1BQU0sQ0FBQyxTQUFTO0lBQ2hCLE1BQU0sQ0FBQyxjQUFjO0lBQ3JCLE1BQU0sQ0FBQyxRQUFRO0lBQ2YsTUFBTSxDQUFDLGNBQWM7Q0FDdEIsQ0FBQyxDQUFDO0FBRUg7O0dBRUc7QUFDSCxNQUFNLFVBQVUsc0JBQXNCLENBQUMsRUFBWTtJQUNqRCxPQUFPLHVCQUF1QixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDOUMsQ0FBQztBQTBGRDs7R0FFRztBQUNILE1BQU0sVUFBVSxvQkFBb0IsQ0FDbEMsR0FBVyxFQUNYLElBQVksRUFDWixTQUFvQixFQUNwQixlQUFnRCxFQUNoRCxlQUFnQyxFQUNoQyxlQUFnQztJQUVoQyxPQUFPO1FBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxZQUFZO1FBQ3pCLElBQUk7UUFDSixHQUFHO1FBQ0gsTUFBTSxFQUFFLElBQUksVUFBVSxFQUFFO1FBQ3hCLFVBQVUsRUFBRSxJQUFJO1FBQ2hCLFNBQVMsRUFBRSxFQUFFO1FBQ2IsV0FBVyxFQUFFLEtBQUs7UUFDbEIsU0FBUztRQUNULGVBQWU7UUFDZixlQUFlO1FBQ2YsZUFBZTtRQUNmLEdBQUcsbUJBQW1CO1FBQ3RCLEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBNkNEOztHQUVHO0FBQ0gsTUFBTSxVQUFVLGdCQUFnQixDQUM5QixJQUFZLEVBQ1osWUFBMEIsRUFDMUIsR0FBa0IsRUFDbEIsa0JBQTBCLEVBQzFCLFNBQW9CLEVBQ3BCLGVBQXdFLEVBQ3hFLGVBQWdDLEVBQ2hDLGVBQWdDO0lBRWhDLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLFFBQVE7UUFDckIsSUFBSTtRQUNKLFlBQVk7UUFDWixVQUFVLEVBQUUsSUFBSTtRQUNoQixHQUFHO1FBQ0gsTUFBTSxFQUFFLElBQUksVUFBVSxFQUFFO1FBQ3hCLGtCQUFrQjtRQUNsQixLQUFLLEVBQUUsSUFBSTtRQUNYLElBQUksRUFBRSxJQUFJO1FBQ1YsU0FBUyxFQUFFLEVBQUU7UUFDYixXQUFXLEVBQUUsS0FBSztRQUNsQixTQUFTO1FBQ1QsZUFBZTtRQUNmLGVBQWU7UUFDZixlQUFlO1FBQ2YsR0FBRyxtQkFBbUI7UUFDdEIsR0FBRyxNQUFNO0tBQ1YsQ0FBQztBQUNKLENBQUM7QUErRUQsTUFBTSxVQUFVLHNCQUFzQixDQUNwQyxXQUFtQixFQUNuQixTQUF3QixFQUN4QixHQUFrQixFQUNsQixLQUFtQixFQUNuQixRQUEwQixFQUMxQixRQUF1QixFQUN2QixlQUFrRCxFQUNsRCxvQkFBdUQsRUFDdkQsZUFBZ0MsRUFDaEMsZUFBZ0M7SUFFaEMsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsY0FBYztRQUMzQixVQUFVLEVBQUUsSUFBSTtRQUNoQixJQUFJLEVBQUUsV0FBVztRQUNqQixNQUFNLEVBQUUsSUFBSSxVQUFVLEVBQUU7UUFDeEIsU0FBUztRQUNULEtBQUs7UUFDTCxTQUFTLEVBQUUsSUFBSTtRQUNmLEdBQUc7UUFDSCxRQUFRO1FBQ1IsZUFBZSxFQUFFLElBQUk7UUFDckIsa0JBQWtCLEVBQUUsS0FBSztRQUN6QixTQUFTLEVBQUUsU0FBUyxDQUFDLElBQUk7UUFDekIsV0FBVyxFQUFFLEtBQUs7UUFDbEIsU0FBUyxFQUFFLEVBQUU7UUFDYixLQUFLLEVBQUUsSUFBSTtRQUNYLElBQUksRUFBRSxJQUFJO1FBQ1YsUUFBUTtRQUNSLHFCQUFxQixFQUFFLEtBQUs7UUFDNUIsZUFBZTtRQUNmLG9CQUFvQjtRQUNwQixlQUFlO1FBQ2YsZUFBZTtRQUNmLEdBQUcsbUJBQW1CO1FBQ3RCLEdBQUcsTUFBTTtRQUNULEdBQUcsbUJBQW1CO1FBQ3RCLFlBQVksRUFBRSxTQUFTLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDekMsQ0FBQztBQUNKLENBQUM7QUFrQkQ7O0dBRUc7QUFDSCxNQUFNLFVBQVUsa0JBQWtCLENBQUMsSUFBWSxFQUFFLFVBQWtDO0lBQ2pGLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLFVBQVU7UUFDdkIsSUFBSTtRQUNKLFVBQVU7UUFDVixHQUFHLE1BQU07S0FDVixDQUFDO0FBQ0osQ0FBQztBQTRDRCxNQUFNLFVBQVUsdUJBQXVCLENBQUMsSUFBWTtJQUNsRCxPQUFPO1FBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlO1FBQzVCLElBQUk7UUFDSixHQUFHLE1BQU07S0FDVixDQUFDO0FBQ0osQ0FBQztBQWVELE1BQU0sVUFBVSxzQkFBc0IsQ0FBQyxJQUFZO0lBQ2pELE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLGNBQWM7UUFDM0IsSUFBSTtRQUNKLEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBMkJEOztHQUVHO0FBQ0gsTUFBTSxVQUFVLFlBQVksQ0FDMUIsSUFBWSxFQUNaLFlBQW9CLEVBQ3BCLGNBQTZCLEVBQzdCLFVBQWtDO0lBRWxDLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUk7UUFDakIsSUFBSTtRQUNKLE1BQU0sRUFBRSxJQUFJLFVBQVUsRUFBRTtRQUN4QixZQUFZO1FBQ1osY0FBYztRQUNkLFVBQVU7UUFDVixHQUFHLG1CQUFtQjtRQUN0QixHQUFHLE1BQU07S0FDVixDQUFDO0FBQ0osQ0FBQztBQTRERDs7R0FFRztBQUNILE1BQU0sVUFBVSxnQkFBZ0IsQ0FDOUIsTUFBYyxFQUNkLFVBQXNCLEVBQ3RCLElBQVksRUFDWixHQUFrQixFQUNsQixVQUEyQixFQUMzQixjQUE2QixFQUM3QixXQUEwQixFQUMxQixZQUFxQixFQUNyQixVQUEyQjtJQUUzQixNQUFNLFdBQVcsR0FBRyxJQUFJLE1BQU0sRUFBWSxDQUFDO0lBQzNDLFdBQVcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDN0IsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsUUFBUTtRQUNyQixNQUFNO1FBQ04sVUFBVTtRQUNWLEdBQUc7UUFDSCxZQUFZO1FBQ1osSUFBSTtRQUNKLFVBQVUsRUFBRSxXQUFXO1FBQ3ZCLGFBQWEsRUFBRSxJQUFJO1FBQ25CLG1CQUFtQixFQUFFLEtBQUs7UUFDMUIsbUJBQW1CLEVBQUUsY0FBYyxLQUFLLElBQUk7UUFDNUMsY0FBYztRQUNkLFdBQVc7UUFDWCxVQUFVO1FBQ1YsR0FBRyxNQUFNO0tBQ1YsQ0FBQztBQUNKLENBQUM7QUFtQ0Q7O0dBRUc7QUFDSCxNQUFNLFVBQVUsc0JBQXNCLENBQ3BDLE1BQWMsRUFDZCxVQUFzQixFQUN0QixJQUFZLEVBQ1osR0FBa0IsRUFDbEIsVUFBMkIsRUFDM0IsVUFBMkI7SUFFM0IsTUFBTSxXQUFXLEdBQUcsSUFBSSxNQUFNLEVBQVksQ0FBQztJQUMzQyxXQUFXLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzdCLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLGNBQWM7UUFDM0IsTUFBTTtRQUNOLFVBQVU7UUFDVixHQUFHO1FBQ0gsSUFBSTtRQUNKLFVBQVUsRUFBRSxXQUFXO1FBQ3ZCLGFBQWEsRUFBRSxJQUFJO1FBQ25CLFVBQVU7UUFDVixHQUFHLE1BQU07S0FDVixDQUFDO0FBQ0osQ0FBQztBQVFELE1BQU0sVUFBVSxZQUFZLENBQUMsSUFBWSxFQUFFLElBQWdCLEVBQUUsSUFBWTtJQUN2RSxPQUFPO1FBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJO1FBQ2pCLElBQUk7UUFDSixNQUFNLEVBQUUsSUFBSTtRQUNaLElBQUk7UUFDSixHQUFHLE1BQU07UUFDVCxHQUFHLG1CQUFtQjtLQUN2QixDQUFDO0FBQ0osQ0FBQztBQVVELE1BQU0sVUFBVSxpQkFBaUIsQ0FBQyxTQUFvQjtJQUNwRCxPQUFPO1FBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxTQUFTO1FBQ3RCLE1BQU0sRUFBRSxTQUFTO1FBQ2pCLEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBWUQsTUFBTSxVQUFVLHFCQUFxQixDQUFDLEdBQXdCO0lBQzVELE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLGFBQWE7UUFDMUIsR0FBRztRQUNILEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBeUJELE1BQU0sVUFBVSxrQkFBa0IsQ0FDaEMsSUFBWSxFQUNaLFFBQWdCLEVBQ2hCLGVBQWdELEVBQ2hELFlBQTJCLEVBQzNCLFVBQTJCO0lBRTNCLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLFVBQVU7UUFDdkIsSUFBSTtRQUNKLE1BQU0sRUFBRSxJQUFJLFVBQVUsRUFBRTtRQUN4QixRQUFRO1FBQ1IsZUFBZTtRQUNmLFlBQVk7UUFDWixtQkFBbUIsRUFBRSxDQUFDO1FBQ3RCLFVBQVUsRUFBRSxJQUFJO1FBQ2hCLFNBQVMsRUFBRSxFQUFFO1FBQ2IsVUFBVTtRQUNWLEdBQUcsTUFBTTtRQUNULEdBQUcsbUJBQW1CO1FBQ3RCLFlBQVksRUFBRSxZQUFZLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDNUMsQ0FBQztBQUNKLENBQUM7QUFvREQ7O0dBRUc7QUFDSCxNQUFNLFVBQVUsMEJBQTBCLENBQ3hDLE1BQWMsRUFDZCxXQUF3QixFQUN4QixTQUF3QixFQUN4QixJQUFZLEVBQ1osVUFBK0IsRUFDL0IsV0FBMEIsRUFDMUIsV0FBZ0MsRUFDaEMsZUFBb0Q7SUFFcEQsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsa0JBQWtCO1FBQy9CLE1BQU07UUFDTixXQUFXO1FBQ1gsU0FBUztRQUNULElBQUk7UUFDSixVQUFVO1FBQ1YsV0FBVztRQUNYLFdBQVc7UUFDWCxlQUFlO1FBQ2YsY0FBYyxFQUFFLElBQUk7UUFDcEIsR0FBRyxNQUFNO0tBQ1YsQ0FBQztBQUNKLENBQUM7QUE2REQsTUFBTSxVQUFVLGFBQWEsQ0FDM0IsSUFBWSxFQUNaLElBQVksRUFDWixRQUFvQixFQUNwQixhQUFrQyxFQUNsQyxVQUErQixFQUMvQixVQUEyQjtJQUUzQixPQUFPO1FBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxLQUFLO1FBQ2xCLElBQUk7UUFDSixNQUFNLEVBQUUsSUFBSSxVQUFVLEVBQUU7UUFDeEIsUUFBUSxFQUFFLElBQUk7UUFDZCxRQUFRO1FBQ1IsV0FBVyxFQUFFLElBQUk7UUFDakIsV0FBVyxFQUFFLElBQUk7UUFDakIsYUFBYSxFQUFFLElBQUk7UUFDbkIsa0JBQWtCLEVBQUUsSUFBSTtRQUN4QixnQkFBZ0IsRUFBRSxJQUFJO1FBQ3RCLGVBQWUsRUFBRSxJQUFJO1FBQ3JCLGVBQWUsRUFBRSxJQUFJO1FBQ3JCLGlCQUFpQixFQUFFLElBQUk7UUFDdkIsc0JBQXNCLEVBQUUsSUFBSTtRQUM1QixTQUFTLEVBQUUsSUFBSTtRQUNmLFNBQVMsRUFBRSxJQUFJO1FBQ2YsYUFBYTtRQUNiLFVBQVU7UUFDVixVQUFVO1FBQ1YsR0FBRyxNQUFNO1FBQ1QsR0FBRyxtQkFBbUI7UUFDdEIsWUFBWSxFQUFFLENBQUM7S0FDaEIsQ0FBQztBQUNKLENBQUM7QUFrRkQsTUFBTSxVQUFVLGVBQWUsQ0FDN0IsS0FBYSxFQUNiLE9BQXFCLEVBQ3JCLFFBQWlCLEVBQ2pCLFVBQTJCO0lBRTNCLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLE9BQU87UUFDcEIsS0FBSztRQUNMLE9BQU87UUFDUCxRQUFRO1FBQ1IsVUFBVTtRQUNWLEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBWUQ7O0dBRUc7QUFDSCxNQUFNLFVBQVUsa0JBQWtCLENBQ2hDLElBQVksRUFDWixZQUFvQixFQUNwQixVQUEyQjtJQUUzQixPQUFPO1FBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxVQUFVO1FBQ3ZCLElBQUk7UUFDSixZQUFZO1FBQ1osVUFBVTtRQUNWLE1BQU0sRUFBRSxJQUFJLFVBQVUsRUFBRTtRQUN4QixHQUFHLG1CQUFtQjtRQUN0QixHQUFHLE1BQU07S0FDVixDQUFDO0FBQ0osQ0FBQztBQWlGRDs7R0FFRztBQUNILE1BQU0sVUFBVSxtQkFBbUIsQ0FDakMsSUFBWSxFQUNaLFdBQW1CLEVBQ25CLFNBQXdCLEVBQ3hCLE9BQXFCLEVBQ3JCLGtCQUFpQyxFQUNqQyxNQUFpQyxFQUNqQyxvQkFBK0MsRUFDL0MsbUJBQTRCO0lBRTVCLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLFdBQVc7UUFDeEIsSUFBSTtRQUNKLFdBQVc7UUFDWCxTQUFTO1FBQ1QsT0FBTztRQUNQLGtCQUFrQjtRQUNsQixNQUFNO1FBQ04sb0JBQW9CO1FBQ3BCLG1CQUFtQjtRQUNuQixXQUFXLEVBQUUsRUFBRTtRQUNmLEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBcUREOztHQUVHO0FBQ0gsTUFBTSxVQUFVLGlCQUFpQixDQUMvQixJQUFZLEVBQ1osT0FBcUIsRUFDckIsSUFBd0IsRUFDeEIsVUFBa0M7SUFFbEMsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsU0FBUztRQUN0QixJQUFJO1FBQ0osTUFBTSxFQUFFLElBQUksVUFBVSxFQUFFO1FBQ3hCLElBQUksRUFBRSxJQUFJLElBQUksSUFBSTtRQUNsQixPQUFPO1FBQ1AsWUFBWSxFQUFFLElBQUk7UUFDbEIsZ0JBQWdCLEVBQUUsSUFBSTtRQUN0QixPQUFPLEVBQUUsSUFBSTtRQUNiLFVBQVU7UUFDVixHQUFHLE1BQU07UUFDVCxHQUFHLG1CQUFtQjtLQUN2QixDQUFDO0FBQ0osQ0FBQztBQWdCRDs7R0FFRztBQUNILE1BQU0sVUFBVSxlQUFlLENBQUMsSUFBWSxFQUFFLFVBQWtDO0lBQzlFLE9BQU87UUFDTCxJQUFJLEVBQUUsTUFBTSxDQUFDLE9BQU87UUFDcEIsSUFBSTtRQUNKLFVBQVU7UUFDVixHQUFHLE1BQU07S0FDVixDQUFDO0FBQ0osQ0FBQztBQStCRDs7R0FFRztBQUNILE1BQU0sVUFBVSxnQkFBZ0IsQ0FDOUIsSUFBWSxFQUNaLE9BQXFCLEVBQ3JCLGtCQUEwQixFQUMxQixVQUEyQjtJQUUzQixPQUFPO1FBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1FBQ3JCLElBQUk7UUFDSixPQUFPO1FBQ1Asa0JBQWtCO1FBQ2xCLE9BQU8sRUFBRSxJQUFJO1FBQ2IsVUFBVTtRQUNWLEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBY0Q7O0dBRUc7QUFDSCxNQUFNLFVBQVUsY0FBYyxDQUFDLElBQVk7SUFDekMsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsTUFBTTtRQUNuQixJQUFJO1FBQ0osR0FBRyxNQUFNO0tBQ1YsQ0FBQztBQUNKLENBQUM7QUFnQ0Q7O0dBRUc7QUFDSCxNQUFNLFVBQVUsc0JBQXNCLENBQ3BDLElBQVksRUFDWixJQUFZLEVBQ1osT0FBaUI7SUFFakIsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsY0FBYztRQUMzQixJQUFJO1FBQ0osSUFBSTtRQUNKLE9BQU87UUFDUCxzQkFBc0IsRUFBRSxFQUFFO1FBQzFCLEdBQUcsTUFBTTtLQUNWLENBQUM7QUFDSixDQUFDO0FBd0RELE1BQU0sVUFBVSxtQkFBbUIsQ0FDakMsV0FBNEIsRUFDNUIsSUFBWSxFQUNaLFNBQXdCLEVBQ3hCLE9BQXFCLEVBQ3JCLFVBQTJCO0lBRTNCLElBQUksU0FBUyxLQUFLLElBQUksSUFBSSxXQUFXLEtBQUssZUFBZSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQy9ELE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQztJQUM1RixDQUFDO0lBRUQsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsV0FBVztRQUN4QixXQUFXO1FBQ1gsSUFBSTtRQUNKLFNBQVM7UUFDVCxPQUFPO1FBQ1AsVUFBVTtRQUNWLE1BQU0sRUFBRSxJQUFJLEdBQUcsRUFBRTtRQUNqQixvQkFBb0IsRUFBRSxJQUFJLEdBQUcsRUFBRTtRQUMvQixHQUFHLE1BQU07S0FDVixDQUFDO0FBQ0osQ0FBQztBQWdCRCxNQUFNLFVBQVUsc0JBQXNCLENBQ3BDLElBQVksRUFDWixNQUFrQixFQUNsQixNQUFjO0lBRWQsT0FBTztRQUNMLElBQUksRUFBRSxNQUFNLENBQUMsY0FBYztRQUMzQixJQUFJO1FBQ0osTUFBTTtRQUNOLE1BQU07UUFDTixvQkFBb0IsRUFBRSxJQUFJO1FBQzFCLEdBQUcsTUFBTTtRQUNULEdBQUcsbUJBQW1CO0tBQ3ZCLENBQUM7QUFDSixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuZGV2L2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQge1NlY3VyaXR5Q29udGV4dH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vY29yZSc7XG5pbXBvcnQgKiBhcyBpMThuIGZyb20gJy4uLy4uLy4uLy4uLy4uL2kxOG4vaTE4bl9hc3QnO1xuaW1wb3J0ICogYXMgbyBmcm9tICcuLi8uLi8uLi8uLi8uLi9vdXRwdXQvb3V0cHV0X2FzdCc7XG5pbXBvcnQge1BhcnNlU291cmNlU3Bhbn0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vcGFyc2VfdXRpbCc7XG5pbXBvcnQge1xuICBCaW5kaW5nS2luZCxcbiAgRGVmZXJUcmlnZ2VyS2luZCxcbiAgSTE4bkNvbnRleHRLaW5kLFxuICBJMThuUGFyYW1WYWx1ZUZsYWdzLFxuICBOYW1lc3BhY2UsXG4gIE9wS2luZCxcbiAgVGVtcGxhdGVLaW5kLFxufSBmcm9tICcuLi9lbnVtcyc7XG5pbXBvcnQge1Nsb3RIYW5kbGV9IGZyb20gJy4uL2hhbmRsZSc7XG5pbXBvcnQge09wLCBPcExpc3QsIFhyZWZJZH0gZnJvbSAnLi4vb3BlcmF0aW9ucyc7XG5pbXBvcnQge1xuICBDb25zdW1lc1Nsb3RPcFRyYWl0LFxuICBDb25zdW1lc1ZhcnNUcmFpdCxcbiAgVFJBSVRfQ09OU1VNRVNfU0xPVCxcbiAgVFJBSVRfQ09OU1VNRVNfVkFSUyxcbn0gZnJvbSAnLi4vdHJhaXRzJztcblxuaW1wb3J0IHtMaXN0RW5kT3AsIE5FV19PUCwgU3RhdGVtZW50T3AsIFZhcmlhYmxlT3B9IGZyb20gJy4vc2hhcmVkJztcblxuaW1wb3J0IHR5cGUge1VwZGF0ZU9wfSBmcm9tICcuL3VwZGF0ZSc7XG5cbi8qKlxuICogQW4gb3BlcmF0aW9uIHVzYWJsZSBvbiB0aGUgY3JlYXRpb24gc2lkZSBvZiB0aGUgSVIuXG4gKi9cbmV4cG9ydCB0eXBlIENyZWF0ZU9wID1cbiAgfCBMaXN0RW5kT3A8Q3JlYXRlT3A+XG4gIHwgU3RhdGVtZW50T3A8Q3JlYXRlT3A+XG4gIHwgRWxlbWVudE9wXG4gIHwgRWxlbWVudFN0YXJ0T3BcbiAgfCBFbGVtZW50RW5kT3BcbiAgfCBDb250YWluZXJPcFxuICB8IENvbnRhaW5lclN0YXJ0T3BcbiAgfCBDb250YWluZXJFbmRPcFxuICB8IFRlbXBsYXRlT3BcbiAgfCBFbmFibGVCaW5kaW5nc09wXG4gIHwgRGlzYWJsZUJpbmRpbmdzT3BcbiAgfCBUZXh0T3BcbiAgfCBMaXN0ZW5lck9wXG4gIHwgVHdvV2F5TGlzdGVuZXJPcFxuICB8IFBpcGVPcFxuICB8IFZhcmlhYmxlT3A8Q3JlYXRlT3A+XG4gIHwgTmFtZXNwYWNlT3BcbiAgfCBQcm9qZWN0aW9uRGVmT3BcbiAgfCBQcm9qZWN0aW9uT3BcbiAgfCBFeHRyYWN0ZWRBdHRyaWJ1dGVPcFxuICB8IERlZmVyT3BcbiAgfCBEZWZlck9uT3BcbiAgfCBSZXBlYXRlckNyZWF0ZU9wXG4gIHwgSTE4bk1lc3NhZ2VPcFxuICB8IEkxOG5PcFxuICB8IEkxOG5TdGFydE9wXG4gIHwgSTE4bkVuZE9wXG4gIHwgSWN1U3RhcnRPcFxuICB8IEljdUVuZE9wXG4gIHwgSWN1UGxhY2Vob2xkZXJPcFxuICB8IEkxOG5Db250ZXh0T3BcbiAgfCBJMThuQXR0cmlidXRlc09wXG4gIHwgRGVjbGFyZUxldE9wO1xuXG4vKipcbiAqIEFuIG9wZXJhdGlvbiByZXByZXNlbnRpbmcgdGhlIGNyZWF0aW9uIG9mIGFuIGVsZW1lbnQgb3IgY29udGFpbmVyLlxuICovXG5leHBvcnQgdHlwZSBFbGVtZW50T3JDb250YWluZXJPcHMgPVxuICB8IEVsZW1lbnRPcFxuICB8IEVsZW1lbnRTdGFydE9wXG4gIHwgQ29udGFpbmVyT3BcbiAgfCBDb250YWluZXJTdGFydE9wXG4gIHwgVGVtcGxhdGVPcFxuICB8IFJlcGVhdGVyQ3JlYXRlT3A7XG5cbi8qKlxuICogVGhlIHNldCBvZiBPcEtpbmRzIHRoYXQgcmVwcmVzZW50IHRoZSBjcmVhdGlvbiBvZiBhbiBlbGVtZW50IG9yIGNvbnRhaW5lclxuICovXG5jb25zdCBlbGVtZW50Q29udGFpbmVyT3BLaW5kcyA9IG5ldyBTZXQoW1xuICBPcEtpbmQuRWxlbWVudCxcbiAgT3BLaW5kLkVsZW1lbnRTdGFydCxcbiAgT3BLaW5kLkNvbnRhaW5lcixcbiAgT3BLaW5kLkNvbnRhaW5lclN0YXJ0LFxuICBPcEtpbmQuVGVtcGxhdGUsXG4gIE9wS2luZC5SZXBlYXRlckNyZWF0ZSxcbl0pO1xuXG4vKipcbiAqIENoZWNrcyB3aGV0aGVyIHRoZSBnaXZlbiBvcGVyYXRpb24gcmVwcmVzZW50cyB0aGUgY3JlYXRpb24gb2YgYW4gZWxlbWVudCBvciBjb250YWluZXIuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc0VsZW1lbnRPckNvbnRhaW5lck9wKG9wOiBDcmVhdGVPcCk6IG9wIGlzIEVsZW1lbnRPckNvbnRhaW5lck9wcyB7XG4gIHJldHVybiBlbGVtZW50Q29udGFpbmVyT3BLaW5kcy5oYXMob3Aua2luZCk7XG59XG5cbi8qKlxuICogUmVwcmVzZW50YXRpb24gb2YgYSBsb2NhbCByZWZlcmVuY2Ugb24gYW4gZWxlbWVudC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2NhbFJlZiB7XG4gIC8qKlxuICAgKiBVc2VyLWRlZmluZWQgbmFtZSBvZiB0aGUgbG9jYWwgcmVmIHZhcmlhYmxlLlxuICAgKi9cbiAgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUYXJnZXQgb2YgdGhlIGxvY2FsIHJlZmVyZW5jZSB2YXJpYWJsZSAob2Z0ZW4gYCcnYCkuXG4gICAqL1xuICB0YXJnZXQ6IHN0cmluZztcbn1cblxuLyoqXG4gKiBCYXNlIGludGVyZmFjZSBmb3IgYEVsZW1lbnRgLCBgRWxlbWVudFN0YXJ0YCwgYW5kIGBUZW1wbGF0ZWAgb3BlcmF0aW9ucywgY29udGFpbmluZyBjb21tb24gZmllbGRzXG4gKiB1c2VkIHRvIHJlcHJlc2VudCB0aGVpciBlbGVtZW50LWxpa2UgbmF0dXJlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVsZW1lbnRPckNvbnRhaW5lck9wQmFzZSBleHRlbmRzIE9wPENyZWF0ZU9wPiwgQ29uc3VtZXNTbG90T3BUcmFpdCB7XG4gIGtpbmQ6IEVsZW1lbnRPckNvbnRhaW5lck9wc1sna2luZCddO1xuXG4gIC8qKlxuICAgKiBgWHJlZklkYCBhbGxvY2F0ZWQgZm9yIHRoaXMgZWxlbWVudC5cbiAgICpcbiAgICogVGhpcyBJRCBpcyB1c2VkIHRvIHJlZmVyZW5jZSB0aGlzIGVsZW1lbnQgZnJvbSBvdGhlciBJUiBzdHJ1Y3R1cmVzLlxuICAgKi9cbiAgeHJlZjogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBBdHRyaWJ1dGVzIG9mIHZhcmlvdXMga2luZHMgb24gdGhpcyBlbGVtZW50LiBSZXByZXNlbnRlZCBhcyBhIGBDb25zdEluZGV4YCBwb2ludGVyIGludG8gdGhlXG4gICAqIHNoYXJlZCBgY29uc3RzYCBhcnJheSBvZiB0aGUgY29tcG9uZW50IGNvbXBpbGF0aW9uLlxuICAgKi9cbiAgYXR0cmlidXRlczogQ29uc3RJbmRleCB8IG51bGw7XG5cbiAgLyoqXG4gICAqIExvY2FsIHJlZmVyZW5jZXMgdG8gdGhpcyBlbGVtZW50LlxuICAgKlxuICAgKiBCZWZvcmUgbG9jYWwgcmVmIHByb2Nlc3NpbmcsIHRoaXMgaXMgYW4gYXJyYXkgb2YgYExvY2FsUmVmYCBkZWNsYXJhdGlvbnMuXG4gICAqXG4gICAqIEFmdGVyIHByb2Nlc3NpbmcsIGl0J3MgYSBgQ29uc3RJbmRleGAgcG9pbnRlciBpbnRvIHRoZSBzaGFyZWQgYGNvbnN0c2AgYXJyYXkgb2YgdGhlIGNvbXBvbmVudFxuICAgKiBjb21waWxhdGlvbi5cbiAgICovXG4gIGxvY2FsUmVmczogTG9jYWxSZWZbXSB8IENvbnN0SW5kZXggfCBudWxsO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgY29udGFpbmVyIGlzIG1hcmtlZCBgbmdOb25CaW5kYWJsZWAsIHdoaWNoIGRpc2FibGVkIEFuZ3VsYXIgYmluZGluZyBmb3IgaXRzZWxmIGFuZFxuICAgKiBhbGwgZGVzY2VuZGFudHMuXG4gICAqL1xuICBub25CaW5kYWJsZTogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIHNwYW4gb2YgdGhlIGVsZW1lbnQncyBzdGFydCB0YWcuXG4gICAqL1xuICBzdGFydFNvdXJjZVNwYW46IFBhcnNlU291cmNlU3BhbjtcblxuICAvKipcbiAgICogVGhlIHdob2xlIHNvdXJjZSBzcGFuIG9mIHRoZSBlbGVtZW50LCBpbmNsdWRpbmcgY2hpbGRyZW4uXG4gICAqL1xuICB3aG9sZVNvdXJjZVNwYW46IFBhcnNlU291cmNlU3Bhbjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBFbGVtZW50T3BCYXNlIGV4dGVuZHMgRWxlbWVudE9yQ29udGFpbmVyT3BCYXNlIHtcbiAga2luZDogT3BLaW5kLkVsZW1lbnQgfCBPcEtpbmQuRWxlbWVudFN0YXJ0IHwgT3BLaW5kLlRlbXBsYXRlIHwgT3BLaW5kLlJlcGVhdGVyQ3JlYXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgSFRNTCB0YWcgbmFtZSBmb3IgdGhpcyBlbGVtZW50LlxuICAgKi9cbiAgdGFnOiBzdHJpbmcgfCBudWxsO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZXNwYWNlIG9mIHRoaXMgZWxlbWVudCwgd2hpY2ggY29udHJvbHMgdGhlIHByZWNlZGluZyBuYW1lc3BhY2UgaW5zdHJ1Y3Rpb24uXG4gICAqL1xuICBuYW1lc3BhY2U6IE5hbWVzcGFjZTtcbn1cblxuLyoqXG4gKiBMb2dpY2FsIG9wZXJhdGlvbiByZXByZXNlbnRpbmcgdGhlIHN0YXJ0IG9mIGFuIGVsZW1lbnQgaW4gdGhlIGNyZWF0aW9uIElSLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVsZW1lbnRTdGFydE9wIGV4dGVuZHMgRWxlbWVudE9wQmFzZSB7XG4gIGtpbmQ6IE9wS2luZC5FbGVtZW50U3RhcnQ7XG5cbiAgLyoqXG4gICAqIFRoZSBpMThuIHBsYWNlaG9sZGVyIGRhdGEgYXNzb2NpYXRlZCB3aXRoIHRoaXMgZWxlbWVudC5cbiAgICovXG4gIGkxOG5QbGFjZWhvbGRlcj86IGkxOG4uVGFnUGxhY2Vob2xkZXI7XG59XG5cbi8qKlxuICogQ3JlYXRlIGFuIGBFbGVtZW50U3RhcnRPcGAuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVFbGVtZW50U3RhcnRPcChcbiAgdGFnOiBzdHJpbmcsXG4gIHhyZWY6IFhyZWZJZCxcbiAgbmFtZXNwYWNlOiBOYW1lc3BhY2UsXG4gIGkxOG5QbGFjZWhvbGRlcjogaTE4bi5UYWdQbGFjZWhvbGRlciB8IHVuZGVmaW5lZCxcbiAgc3RhcnRTb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW4sXG4gIHdob2xlU291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogRWxlbWVudFN0YXJ0T3Age1xuICByZXR1cm4ge1xuICAgIGtpbmQ6IE9wS2luZC5FbGVtZW50U3RhcnQsXG4gICAgeHJlZixcbiAgICB0YWcsXG4gICAgaGFuZGxlOiBuZXcgU2xvdEhhbmRsZSgpLFxuICAgIGF0dHJpYnV0ZXM6IG51bGwsXG4gICAgbG9jYWxSZWZzOiBbXSxcbiAgICBub25CaW5kYWJsZTogZmFsc2UsXG4gICAgbmFtZXNwYWNlLFxuICAgIGkxOG5QbGFjZWhvbGRlcixcbiAgICBzdGFydFNvdXJjZVNwYW4sXG4gICAgd2hvbGVTb3VyY2VTcGFuLFxuICAgIC4uLlRSQUlUX0NPTlNVTUVTX1NMT1QsXG4gICAgLi4uTkVXX09QLFxuICB9O1xufVxuXG4vKipcbiAqIExvZ2ljYWwgb3BlcmF0aW9uIHJlcHJlc2VudGluZyBhbiBlbGVtZW50IHdpdGggbm8gY2hpbGRyZW4gaW4gdGhlIGNyZWF0aW9uIElSLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVsZW1lbnRPcCBleHRlbmRzIEVsZW1lbnRPcEJhc2Uge1xuICBraW5kOiBPcEtpbmQuRWxlbWVudDtcblxuICAvKipcbiAgICogVGhlIGkxOG4gcGxhY2Vob2xkZXIgZGF0YSBhc3NvY2lhdGVkIHdpdGggdGhpcyBlbGVtZW50LlxuICAgKi9cbiAgaTE4blBsYWNlaG9sZGVyPzogaTE4bi5UYWdQbGFjZWhvbGRlcjtcbn1cblxuLyoqXG4gKiBMb2dpY2FsIG9wZXJhdGlvbiByZXByZXNlbnRpbmcgYW4gZW1iZWRkZWQgdmlldyBkZWNsYXJhdGlvbiBpbiB0aGUgY3JlYXRpb24gSVIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVGVtcGxhdGVPcCBleHRlbmRzIEVsZW1lbnRPcEJhc2Uge1xuICBraW5kOiBPcEtpbmQuVGVtcGxhdGU7XG5cbiAgdGVtcGxhdGVLaW5kOiBUZW1wbGF0ZUtpbmQ7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgZGVjbGFyYXRpb24gc2xvdHMgdXNlZCBieSB0aGlzIHRlbXBsYXRlLCBvciBgbnVsbGAgaWYgc2xvdHMgaGF2ZSBub3QgeWV0IGJlZW5cbiAgICogYXNzaWduZWQuXG4gICAqL1xuICBkZWNsczogbnVtYmVyIHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBiaW5kaW5nIHZhcmlhYmxlIHNsb3RzIHVzZWQgYnkgdGhpcyB0ZW1wbGF0ZSwgb3IgYG51bGxgIGlmIGJpbmRpbmcgdmFyaWFibGVzIGhhdmVcbiAgICogbm90IHlldCBiZWVuIGNvdW50ZWQuXG4gICAqL1xuICB2YXJzOiBudW1iZXIgfCBudWxsO1xuXG4gIC8qKlxuICAgKiBTdWZmaXggdG8gYWRkIHRvIHRoZSBuYW1lIG9mIHRoZSBnZW5lcmF0ZWQgdGVtcGxhdGUgZnVuY3Rpb24uXG4gICAqL1xuICBmdW5jdGlvbk5hbWVTdWZmaXg6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGkxOG4gcGxhY2Vob2xkZXIgZGF0YSBhc3NvY2lhdGVkIHdpdGggdGhpcyB0ZW1wbGF0ZS5cbiAgICovXG4gIGkxOG5QbGFjZWhvbGRlcj86IGkxOG4uVGFnUGxhY2Vob2xkZXIgfCBpMThuLkJsb2NrUGxhY2Vob2xkZXI7XG59XG5cbi8qKlxuICogQ3JlYXRlIGEgYFRlbXBsYXRlT3BgLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlVGVtcGxhdGVPcChcbiAgeHJlZjogWHJlZklkLFxuICB0ZW1wbGF0ZUtpbmQ6IFRlbXBsYXRlS2luZCxcbiAgdGFnOiBzdHJpbmcgfCBudWxsLFxuICBmdW5jdGlvbk5hbWVTdWZmaXg6IHN0cmluZyxcbiAgbmFtZXNwYWNlOiBOYW1lc3BhY2UsXG4gIGkxOG5QbGFjZWhvbGRlcjogaTE4bi5UYWdQbGFjZWhvbGRlciB8IGkxOG4uQmxvY2tQbGFjZWhvbGRlciB8IHVuZGVmaW5lZCxcbiAgc3RhcnRTb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW4sXG4gIHdob2xlU291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogVGVtcGxhdGVPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLlRlbXBsYXRlLFxuICAgIHhyZWYsXG4gICAgdGVtcGxhdGVLaW5kLFxuICAgIGF0dHJpYnV0ZXM6IG51bGwsXG4gICAgdGFnLFxuICAgIGhhbmRsZTogbmV3IFNsb3RIYW5kbGUoKSxcbiAgICBmdW5jdGlvbk5hbWVTdWZmaXgsXG4gICAgZGVjbHM6IG51bGwsXG4gICAgdmFyczogbnVsbCxcbiAgICBsb2NhbFJlZnM6IFtdLFxuICAgIG5vbkJpbmRhYmxlOiBmYWxzZSxcbiAgICBuYW1lc3BhY2UsXG4gICAgaTE4blBsYWNlaG9sZGVyLFxuICAgIHN0YXJ0U291cmNlU3BhbixcbiAgICB3aG9sZVNvdXJjZVNwYW4sXG4gICAgLi4uVFJBSVRfQ09OU1VNRVNfU0xPVCxcbiAgICAuLi5ORVdfT1AsXG4gIH07XG59XG5cbi8qKlxuICogQW4gb3AgdGhhdCBjcmVhdGVzIGEgcmVwZWF0ZXIgKGUuZy4gYSBmb3IgbG9vcCkuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVwZWF0ZXJDcmVhdGVPcCBleHRlbmRzIEVsZW1lbnRPcEJhc2UsIENvbnN1bWVzVmFyc1RyYWl0IHtcbiAga2luZDogT3BLaW5kLlJlcGVhdGVyQ3JlYXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGRlY2xhcmF0aW9uIHNsb3RzIHVzZWQgYnkgdGhpcyByZXBlYXRlcidzIHRlbXBsYXRlLCBvciBgbnVsbGAgaWYgc2xvdHMgaGF2ZSBub3RcbiAgICogeWV0IGJlZW4gYXNzaWduZWQuXG4gICAqL1xuICBkZWNsczogbnVtYmVyIHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBiaW5kaW5nIHZhcmlhYmxlIHNsb3RzIHVzZWQgYnkgdGhpcyByZXBlYXRlcidzLCBvciBgbnVsbGAgaWYgYmluZGluZyB2YXJpYWJsZXNcbiAgICogaGF2ZSBub3QgeWV0IGJlZW4gY291bnRlZC5cbiAgICovXG4gIHZhcnM6IG51bWJlciB8IG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBYcmVmIG9mIHRoZSBlbXB0eSB2aWV3IGZ1bmN0aW9uLiAoRm9yIHRoZSBwcmltYXJ5IHZpZXcgZnVuY3Rpb24sIHVzZSB0aGUgYHhyZWZgIHByb3BlcnR5KS5cbiAgICovXG4gIGVtcHR5VmlldzogWHJlZklkIHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIHRyYWNrIGV4cHJlc3Npb24gdG8gdXNlIHdoaWxlIGl0ZXJhdGluZy5cbiAgICovXG4gIHRyYWNrOiBvLkV4cHJlc3Npb247XG5cbiAgLyoqXG4gICAqIGBudWxsYCBpbml0aWFsbHksIHRoZW4gYW4gYG8uRXhwcmVzc2lvbmAuIE1pZ2h0IGJlIGEgdHJhY2sgZXhwcmVzc2lvbiwgb3IgbWlnaHQgYmUgYSByZWZlcmVuY2VcbiAgICogaW50byB0aGUgY29uc3RhbnQgcG9vbC5cbiAgICovXG4gIHRyYWNrQnlGbjogby5FeHByZXNzaW9uIHwgbnVsbDtcblxuICAvKipcbiAgICogQ29udGV4dCB2YXJpYWJsZXMgYXZhaWFsYWJsZSBpbiB0aGlzIGJsb2NrLlxuICAgKi9cbiAgdmFyTmFtZXM6IFJlcGVhdGVyVmFyTmFtZXM7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIHJlcGVhdGVyIHRyYWNrIGZ1bmN0aW9uIHJlbGllcyBvbiB0aGUgY29tcG9uZW50IGluc3RhbmNlLlxuICAgKi9cbiAgdXNlc0NvbXBvbmVudEluc3RhbmNlOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBTdWZmaXggdG8gYWRkIHRvIHRoZSBuYW1lIG9mIHRoZSBnZW5lcmF0ZWQgdGVtcGxhdGUgZnVuY3Rpb24uXG4gICAqL1xuICBmdW5jdGlvbk5hbWVTdWZmaXg6IHN0cmluZztcblxuICAvKipcbiAgICogVGFnIG5hbWUgZm9yIHRoZSBlbXB0eSBibG9jay5cbiAgICovXG4gIGVtcHR5VGFnOiBzdHJpbmcgfCBudWxsO1xuXG4gIC8qKlxuICAgKiBBdHRyaWJ1dGVzIG9mIHZhcmlvdXMga2luZHMgb24gdGhlIGVtcHR5IGJsb2NrLiBSZXByZXNlbnRlZCBhcyBhIGBDb25zdEluZGV4YCBwb2ludGVyIGludG8gdGhlXG4gICAqIHNoYXJlZCBgY29uc3RzYCBhcnJheSBvZiB0aGUgY29tcG9uZW50IGNvbXBpbGF0aW9uLlxuICAgKi9cbiAgZW1wdHlBdHRyaWJ1dGVzOiBDb25zdEluZGV4IHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIGkxOG4gcGxhY2Vob2xkZXIgZm9yIHRoZSByZXBlYXRlZCBpdGVtIHRlbXBsYXRlLlxuICAgKi9cbiAgaTE4blBsYWNlaG9sZGVyOiBpMThuLkJsb2NrUGxhY2Vob2xkZXIgfCB1bmRlZmluZWQ7XG5cbiAgLyoqXG4gICAqIFRoZSBpMThuIHBsYWNlaG9sZGVyIGZvciB0aGUgZW1wdHkgdGVtcGxhdGUuXG4gICAqL1xuICBlbXB0eUkxOG5QbGFjZWhvbGRlcjogaTE4bi5CbG9ja1BsYWNlaG9sZGVyIHwgdW5kZWZpbmVkO1xufVxuXG4vLyBUT0RPOiBhZGQgc291cmNlIHNwYW5zP1xuZXhwb3J0IGludGVyZmFjZSBSZXBlYXRlclZhck5hbWVzIHtcbiAgJGluZGV4OiBTZXQ8c3RyaW5nPjtcbiAgJGltcGxpY2l0OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVSZXBlYXRlckNyZWF0ZU9wKFxuICBwcmltYXJ5VmlldzogWHJlZklkLFxuICBlbXB0eVZpZXc6IFhyZWZJZCB8IG51bGwsXG4gIHRhZzogc3RyaW5nIHwgbnVsbCxcbiAgdHJhY2s6IG8uRXhwcmVzc2lvbixcbiAgdmFyTmFtZXM6IFJlcGVhdGVyVmFyTmFtZXMsXG4gIGVtcHR5VGFnOiBzdHJpbmcgfCBudWxsLFxuICBpMThuUGxhY2Vob2xkZXI6IGkxOG4uQmxvY2tQbGFjZWhvbGRlciB8IHVuZGVmaW5lZCxcbiAgZW1wdHlJMThuUGxhY2Vob2xkZXI6IGkxOG4uQmxvY2tQbGFjZWhvbGRlciB8IHVuZGVmaW5lZCxcbiAgc3RhcnRTb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW4sXG4gIHdob2xlU291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogUmVwZWF0ZXJDcmVhdGVPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLlJlcGVhdGVyQ3JlYXRlLFxuICAgIGF0dHJpYnV0ZXM6IG51bGwsXG4gICAgeHJlZjogcHJpbWFyeVZpZXcsXG4gICAgaGFuZGxlOiBuZXcgU2xvdEhhbmRsZSgpLFxuICAgIGVtcHR5VmlldyxcbiAgICB0cmFjayxcbiAgICB0cmFja0J5Rm46IG51bGwsXG4gICAgdGFnLFxuICAgIGVtcHR5VGFnLFxuICAgIGVtcHR5QXR0cmlidXRlczogbnVsbCxcbiAgICBmdW5jdGlvbk5hbWVTdWZmaXg6ICdGb3InLFxuICAgIG5hbWVzcGFjZTogTmFtZXNwYWNlLkhUTUwsXG4gICAgbm9uQmluZGFibGU6IGZhbHNlLFxuICAgIGxvY2FsUmVmczogW10sXG4gICAgZGVjbHM6IG51bGwsXG4gICAgdmFyczogbnVsbCxcbiAgICB2YXJOYW1lcyxcbiAgICB1c2VzQ29tcG9uZW50SW5zdGFuY2U6IGZhbHNlLFxuICAgIGkxOG5QbGFjZWhvbGRlcixcbiAgICBlbXB0eUkxOG5QbGFjZWhvbGRlcixcbiAgICBzdGFydFNvdXJjZVNwYW4sXG4gICAgd2hvbGVTb3VyY2VTcGFuLFxuICAgIC4uLlRSQUlUX0NPTlNVTUVTX1NMT1QsXG4gICAgLi4uTkVXX09QLFxuICAgIC4uLlRSQUlUX0NPTlNVTUVTX1ZBUlMsXG4gICAgbnVtU2xvdHNVc2VkOiBlbXB0eVZpZXcgPT09IG51bGwgPyAyIDogMyxcbiAgfTtcbn1cblxuLyoqXG4gKiBMb2dpY2FsIG9wZXJhdGlvbiByZXByZXNlbnRpbmcgdGhlIGVuZCBvZiBhbiBlbGVtZW50IHN0cnVjdHVyZSBpbiB0aGUgY3JlYXRpb24gSVIuXG4gKlxuICogUGFpcnMgd2l0aCBhbiBgRWxlbWVudFN0YXJ0YCBvcGVyYXRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRWxlbWVudEVuZE9wIGV4dGVuZHMgT3A8Q3JlYXRlT3A+IHtcbiAga2luZDogT3BLaW5kLkVsZW1lbnRFbmQ7XG5cbiAgLyoqXG4gICAqIFRoZSBgWHJlZklkYCBvZiB0aGUgZWxlbWVudCBkZWNsYXJlZCB2aWEgYEVsZW1lbnRTdGFydGAuXG4gICAqL1xuICB4cmVmOiBYcmVmSWQ7XG5cbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuIHwgbnVsbDtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYW4gYEVsZW1lbnRFbmRPcGAuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVFbGVtZW50RW5kT3AoeHJlZjogWHJlZklkLCBzb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW4gfCBudWxsKTogRWxlbWVudEVuZE9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuRWxlbWVudEVuZCxcbiAgICB4cmVmLFxuICAgIHNvdXJjZVNwYW4sXG4gICAgLi4uTkVXX09QLFxuICB9O1xufVxuXG4vKipcbiAqIExvZ2ljYWwgb3BlcmF0aW9uIHJlcHJlc2VudGluZyB0aGUgc3RhcnQgb2YgYSBjb250YWluZXIgaW4gdGhlIGNyZWF0aW9uIElSLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lclN0YXJ0T3AgZXh0ZW5kcyBFbGVtZW50T3JDb250YWluZXJPcEJhc2Uge1xuICBraW5kOiBPcEtpbmQuQ29udGFpbmVyU3RhcnQ7XG59XG5cbi8qKlxuICogTG9naWNhbCBvcGVyYXRpb24gcmVwcmVzZW50aW5nIGFuIGVtcHR5IGNvbnRhaW5lciBpbiB0aGUgY3JlYXRpb24gSVIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29udGFpbmVyT3AgZXh0ZW5kcyBFbGVtZW50T3JDb250YWluZXJPcEJhc2Uge1xuICBraW5kOiBPcEtpbmQuQ29udGFpbmVyO1xufVxuXG4vKipcbiAqIExvZ2ljYWwgb3BlcmF0aW9uIHJlcHJlc2VudGluZyB0aGUgZW5kIG9mIGEgY29udGFpbmVyIHN0cnVjdHVyZSBpbiB0aGUgY3JlYXRpb24gSVIuXG4gKlxuICogUGFpcnMgd2l0aCBhbiBgQ29udGFpbmVyU3RhcnRgIG9wZXJhdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb250YWluZXJFbmRPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiB7XG4gIGtpbmQ6IE9wS2luZC5Db250YWluZXJFbmQ7XG5cbiAgLyoqXG4gICAqIFRoZSBgWHJlZklkYCBvZiB0aGUgZWxlbWVudCBkZWNsYXJlZCB2aWEgYENvbnRhaW5lclN0YXJ0YC5cbiAgICovXG4gIHhyZWY6IFhyZWZJZDtcblxuICBzb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW47XG59XG5cbi8qKlxuICogTG9naWNhbCBvcGVyYXRpb24gY2F1c2luZyBiaW5kaW5nIHRvIGJlIGRpc2FibGVkIGluIGRlc2NlbmRlbnRzIG9mIGEgbm9uLWJpbmRhYmxlIGNvbnRhaW5lci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEaXNhYmxlQmluZGluZ3NPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiB7XG4gIGtpbmQ6IE9wS2luZC5EaXNhYmxlQmluZGluZ3M7XG5cbiAgLyoqXG4gICAqIGBYcmVmSWRgIG9mIHRoZSBlbGVtZW50IHRoYXQgd2FzIG1hcmtlZCBub24tYmluZGFibGUuXG4gICAqL1xuICB4cmVmOiBYcmVmSWQ7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVEaXNhYmxlQmluZGluZ3NPcCh4cmVmOiBYcmVmSWQpOiBEaXNhYmxlQmluZGluZ3NPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLkRpc2FibGVCaW5kaW5ncyxcbiAgICB4cmVmLFxuICAgIC4uLk5FV19PUCxcbiAgfTtcbn1cblxuLyoqXG4gKiBMb2dpY2FsIG9wZXJhdGlvbiBjYXVzaW5nIGJpbmRpbmcgdG8gYmUgcmUtZW5hYmxlZCBhZnRlciB2aXNpdGluZyBkZXNjZW5kYW50cyBvZiBhXG4gKiBub24tYmluZGFibGUgY29udGFpbmVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVuYWJsZUJpbmRpbmdzT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4ge1xuICBraW5kOiBPcEtpbmQuRW5hYmxlQmluZGluZ3M7XG5cbiAgLyoqXG4gICAqIGBYcmVmSWRgIG9mIHRoZSBlbGVtZW50IHRoYXQgd2FzIG1hcmtlZCBub24tYmluZGFibGUuXG4gICAqL1xuICB4cmVmOiBYcmVmSWQ7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVFbmFibGVCaW5kaW5nc09wKHhyZWY6IFhyZWZJZCk6IEVuYWJsZUJpbmRpbmdzT3Age1xuICByZXR1cm4ge1xuICAgIGtpbmQ6IE9wS2luZC5FbmFibGVCaW5kaW5ncyxcbiAgICB4cmVmLFxuICAgIC4uLk5FV19PUCxcbiAgfTtcbn1cblxuLyoqXG4gKiBMb2dpY2FsIG9wZXJhdGlvbiByZXByZXNlbnRpbmcgYSB0ZXh0IG5vZGUgaW4gdGhlIGNyZWF0aW9uIElSLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRleHRPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiwgQ29uc3VtZXNTbG90T3BUcmFpdCB7XG4gIGtpbmQ6IE9wS2luZC5UZXh0O1xuXG4gIC8qKlxuICAgKiBgWHJlZklkYCB1c2VkIHRvIHJlZmVyZW5jZSB0aGlzIHRleHQgbm9kZSBpbiBvdGhlciBJUiBzdHJ1Y3R1cmVzLlxuICAgKi9cbiAgeHJlZjogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBUaGUgc3RhdGljIGluaXRpYWwgdmFsdWUgb2YgdGhlIHRleHQgbm9kZS5cbiAgICovXG4gIGluaXRpYWxWYWx1ZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGxhY2Vob2xkZXIgZm9yIHRoaXMgdGV4dCBpbiBpdHMgcGFyZW50IElDVS4gSWYgdGhpcyB0ZXh0IGlzIG5vdCBwYXJ0IG9mIGFuIElDVSwgdGhlXG4gICAqIHBsYWNlaG9sZGVyIGlzIG51bGwuXG4gICAqL1xuICBpY3VQbGFjZWhvbGRlcjogc3RyaW5nIHwgbnVsbDtcblxuICBzb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW4gfCBudWxsO1xufVxuXG4vKipcbiAqIENyZWF0ZSBhIGBUZXh0T3BgLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlVGV4dE9wKFxuICB4cmVmOiBYcmVmSWQsXG4gIGluaXRpYWxWYWx1ZTogc3RyaW5nLFxuICBpY3VQbGFjZWhvbGRlcjogc3RyaW5nIHwgbnVsbCxcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuIHwgbnVsbCxcbik6IFRleHRPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLlRleHQsXG4gICAgeHJlZixcbiAgICBoYW5kbGU6IG5ldyBTbG90SGFuZGxlKCksXG4gICAgaW5pdGlhbFZhbHVlLFxuICAgIGljdVBsYWNlaG9sZGVyLFxuICAgIHNvdXJjZVNwYW4sXG4gICAgLi4uVFJBSVRfQ09OU1VNRVNfU0xPVCxcbiAgICAuLi5ORVdfT1AsXG4gIH07XG59XG5cbi8qKlxuICogTG9naWNhbCBvcGVyYXRpb24gcmVwcmVzZW50aW5nIGFuIGV2ZW50IGxpc3RlbmVyIG9uIGFuIGVsZW1lbnQgaW4gdGhlIGNyZWF0aW9uIElSLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExpc3RlbmVyT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4ge1xuICBraW5kOiBPcEtpbmQuTGlzdGVuZXI7XG5cbiAgdGFyZ2V0OiBYcmVmSWQ7XG4gIHRhcmdldFNsb3Q6IFNsb3RIYW5kbGU7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhpcyBsaXN0ZW5lciBpcyBmcm9tIGEgaG9zdCBiaW5kaW5nLlxuICAgKi9cbiAgaG9zdExpc3RlbmVyOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBldmVudCB3aGljaCBpcyBiZWluZyBsaXN0ZW5lZCB0by5cbiAgICovXG4gIG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGFnIG5hbWUgb2YgdGhlIGVsZW1lbnQgb24gd2hpY2ggdGhpcyBsaXN0ZW5lciBpcyBwbGFjZWQuIE1pZ2h0IGJlIG51bGwsIGlmIHRoaXMgbGlzdGVuZXJcbiAgICogYmVsb25ncyB0byBhIGhvc3QgYmluZGluZy5cbiAgICovXG4gIHRhZzogc3RyaW5nIHwgbnVsbDtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIGBVcGRhdGVPcGBzIHJlcHJlc2VudGluZyB0aGUgYm9keSBvZiB0aGUgZXZlbnQgbGlzdGVuZXIuXG4gICAqL1xuICBoYW5kbGVyT3BzOiBPcExpc3Q8VXBkYXRlT3A+O1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBmdW5jdGlvblxuICAgKi9cbiAgaGFuZGxlckZuTmFtZTogc3RyaW5nIHwgbnVsbDtcblxuICAvKipcbiAgICogV2hldGhlciB0aGlzIGxpc3RlbmVyIGlzIGtub3duIHRvIGNvbnN1bWUgYCRldmVudGAgaW4gaXRzIGJvZHkuXG4gICAqL1xuICBjb25zdW1lc0RvbGxhckV2ZW50OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBsaXN0ZW5lciBpcyBsaXN0ZW5pbmcgZm9yIGFuIGFuaW1hdGlvbiBldmVudC5cbiAgICovXG4gIGlzQW5pbWF0aW9uTGlzdGVuZXI6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBhbmltYXRpb24gcGhhc2Ugb2YgdGhlIGxpc3RlbmVyLlxuICAgKi9cbiAgYW5pbWF0aW9uUGhhc2U6IHN0cmluZyB8IG51bGw7XG5cbiAgLyoqXG4gICAqIFNvbWUgZXZlbnQgbGlzdGVuZXJzIGNhbiBoYXZlIGEgdGFyZ2V0LCBlLmcuIGluIGBkb2N1bWVudDpkcmFnb3ZlcmAuXG4gICAqL1xuICBldmVudFRhcmdldDogc3RyaW5nIHwgbnVsbDtcblxuICBzb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW47XG59XG5cbi8qKlxuICogQ3JlYXRlIGEgYExpc3RlbmVyT3BgLiBIb3N0IGJpbmRpbmdzIHJldXNlIGFsbCB0aGUgbGlzdGVuZXIgbG9naWMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVMaXN0ZW5lck9wKFxuICB0YXJnZXQ6IFhyZWZJZCxcbiAgdGFyZ2V0U2xvdDogU2xvdEhhbmRsZSxcbiAgbmFtZTogc3RyaW5nLFxuICB0YWc6IHN0cmluZyB8IG51bGwsXG4gIGhhbmRsZXJPcHM6IEFycmF5PFVwZGF0ZU9wPixcbiAgYW5pbWF0aW9uUGhhc2U6IHN0cmluZyB8IG51bGwsXG4gIGV2ZW50VGFyZ2V0OiBzdHJpbmcgfCBudWxsLFxuICBob3N0TGlzdGVuZXI6IGJvb2xlYW4sXG4gIHNvdXJjZVNwYW46IFBhcnNlU291cmNlU3Bhbixcbik6IExpc3RlbmVyT3Age1xuICBjb25zdCBoYW5kbGVyTGlzdCA9IG5ldyBPcExpc3Q8VXBkYXRlT3A+KCk7XG4gIGhhbmRsZXJMaXN0LnB1c2goaGFuZGxlck9wcyk7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLkxpc3RlbmVyLFxuICAgIHRhcmdldCxcbiAgICB0YXJnZXRTbG90LFxuICAgIHRhZyxcbiAgICBob3N0TGlzdGVuZXIsXG4gICAgbmFtZSxcbiAgICBoYW5kbGVyT3BzOiBoYW5kbGVyTGlzdCxcbiAgICBoYW5kbGVyRm5OYW1lOiBudWxsLFxuICAgIGNvbnN1bWVzRG9sbGFyRXZlbnQ6IGZhbHNlLFxuICAgIGlzQW5pbWF0aW9uTGlzdGVuZXI6IGFuaW1hdGlvblBoYXNlICE9PSBudWxsLFxuICAgIGFuaW1hdGlvblBoYXNlLFxuICAgIGV2ZW50VGFyZ2V0LFxuICAgIHNvdXJjZVNwYW4sXG4gICAgLi4uTkVXX09QLFxuICB9O1xufVxuXG4vKipcbiAqIExvZ2ljYWwgb3BlcmF0aW9uIHJlcHJlc2VudGluZyB0aGUgZXZlbnQgc2lkZSBvZiBhIHR3by13YXkgYmluZGluZyBvbiBhbiBlbGVtZW50XG4gKiBpbiB0aGUgY3JlYXRpb24gSVIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVHdvV2F5TGlzdGVuZXJPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiB7XG4gIGtpbmQ6IE9wS2luZC5Ud29XYXlMaXN0ZW5lcjtcblxuICB0YXJnZXQ6IFhyZWZJZDtcbiAgdGFyZ2V0U2xvdDogU2xvdEhhbmRsZTtcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGUgZXZlbnQgd2hpY2ggaXMgYmVpbmcgbGlzdGVuZWQgdG8uXG4gICAqL1xuICBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRhZyBuYW1lIG9mIHRoZSBlbGVtZW50IG9uIHdoaWNoIHRoaXMgbGlzdGVuZXIgaXMgcGxhY2VkLlxuICAgKi9cbiAgdGFnOiBzdHJpbmcgfCBudWxsO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgYFVwZGF0ZU9wYHMgcmVwcmVzZW50aW5nIHRoZSBib2R5IG9mIHRoZSBldmVudCBsaXN0ZW5lci5cbiAgICovXG4gIGhhbmRsZXJPcHM6IE9wTGlzdDxVcGRhdGVPcD47XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGZ1bmN0aW9uXG4gICAqL1xuICBoYW5kbGVyRm5OYW1lOiBzdHJpbmcgfCBudWxsO1xuXG4gIHNvdXJjZVNwYW46IFBhcnNlU291cmNlU3Bhbjtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBgVHdvV2F5TGlzdGVuZXJPcGAuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVUd29XYXlMaXN0ZW5lck9wKFxuICB0YXJnZXQ6IFhyZWZJZCxcbiAgdGFyZ2V0U2xvdDogU2xvdEhhbmRsZSxcbiAgbmFtZTogc3RyaW5nLFxuICB0YWc6IHN0cmluZyB8IG51bGwsXG4gIGhhbmRsZXJPcHM6IEFycmF5PFVwZGF0ZU9wPixcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogVHdvV2F5TGlzdGVuZXJPcCB7XG4gIGNvbnN0IGhhbmRsZXJMaXN0ID0gbmV3IE9wTGlzdDxVcGRhdGVPcD4oKTtcbiAgaGFuZGxlckxpc3QucHVzaChoYW5kbGVyT3BzKTtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuVHdvV2F5TGlzdGVuZXIsXG4gICAgdGFyZ2V0LFxuICAgIHRhcmdldFNsb3QsXG4gICAgdGFnLFxuICAgIG5hbWUsXG4gICAgaGFuZGxlck9wczogaGFuZGxlckxpc3QsXG4gICAgaGFuZGxlckZuTmFtZTogbnVsbCxcbiAgICBzb3VyY2VTcGFuLFxuICAgIC4uLk5FV19PUCxcbiAgfTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQaXBlT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4sIENvbnN1bWVzU2xvdE9wVHJhaXQge1xuICBraW5kOiBPcEtpbmQuUGlwZTtcbiAgeHJlZjogWHJlZklkO1xuICBuYW1lOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVQaXBlT3AoeHJlZjogWHJlZklkLCBzbG90OiBTbG90SGFuZGxlLCBuYW1lOiBzdHJpbmcpOiBQaXBlT3Age1xuICByZXR1cm4ge1xuICAgIGtpbmQ6IE9wS2luZC5QaXBlLFxuICAgIHhyZWYsXG4gICAgaGFuZGxlOiBzbG90LFxuICAgIG5hbWUsXG4gICAgLi4uTkVXX09QLFxuICAgIC4uLlRSQUlUX0NPTlNVTUVTX1NMT1QsXG4gIH07XG59XG5cbi8qKlxuICogQW4gb3AgY29ycmVzcG9uZGluZyB0byBhIG5hbWVzcGFjZSBpbnN0cnVjdGlvbiwgZm9yIHN3aXRjaGluZyBiZXR3ZWVuIEhUTUwsIFNWRywgYW5kIE1hdGhNTC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOYW1lc3BhY2VPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiB7XG4gIGtpbmQ6IE9wS2luZC5OYW1lc3BhY2U7XG4gIGFjdGl2ZTogTmFtZXNwYWNlO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlTmFtZXNwYWNlT3AobmFtZXNwYWNlOiBOYW1lc3BhY2UpOiBOYW1lc3BhY2VPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLk5hbWVzcGFjZSxcbiAgICBhY3RpdmU6IG5hbWVzcGFjZSxcbiAgICAuLi5ORVdfT1AsXG4gIH07XG59XG5cbi8qKlxuICogQW4gb3AgdGhhdCBjcmVhdGVzIGEgY29udGVudCBwcm9qZWN0aW9uIHNsb3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUHJvamVjdGlvbkRlZk9wIGV4dGVuZHMgT3A8Q3JlYXRlT3A+IHtcbiAga2luZDogT3BLaW5kLlByb2plY3Rpb25EZWY7XG5cbiAgLy8gVGhlIHBhcnNlZCBzZWxlY3RvciBpbmZvcm1hdGlvbiBmb3IgdGhpcyBwcm9qZWN0aW9uIGRlZi5cbiAgZGVmOiBvLkV4cHJlc3Npb24gfCBudWxsO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlUHJvamVjdGlvbkRlZk9wKGRlZjogby5FeHByZXNzaW9uIHwgbnVsbCk6IFByb2plY3Rpb25EZWZPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLlByb2plY3Rpb25EZWYsXG4gICAgZGVmLFxuICAgIC4uLk5FV19PUCxcbiAgfTtcbn1cblxuLyoqXG4gKiBBbiBvcCB0aGF0IGNyZWF0ZXMgYSBjb250ZW50IHByb2plY3Rpb24gc2xvdC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQcm9qZWN0aW9uT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4sIENvbnN1bWVzU2xvdE9wVHJhaXQge1xuICBraW5kOiBPcEtpbmQuUHJvamVjdGlvbjtcblxuICB4cmVmOiBYcmVmSWQ7XG5cbiAgcHJvamVjdGlvblNsb3RJbmRleDogbnVtYmVyO1xuXG4gIGF0dHJpYnV0ZXM6IG51bGwgfCBvLkxpdGVyYWxBcnJheUV4cHI7XG5cbiAgbG9jYWxSZWZzOiBzdHJpbmdbXTtcblxuICBzZWxlY3Rvcjogc3RyaW5nO1xuXG4gIGkxOG5QbGFjZWhvbGRlcj86IGkxOG4uVGFnUGxhY2Vob2xkZXI7XG5cbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuO1xuXG4gIGZhbGxiYWNrVmlldzogWHJlZklkIHwgbnVsbDtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZVByb2plY3Rpb25PcChcbiAgeHJlZjogWHJlZklkLFxuICBzZWxlY3Rvcjogc3RyaW5nLFxuICBpMThuUGxhY2Vob2xkZXI6IGkxOG4uVGFnUGxhY2Vob2xkZXIgfCB1bmRlZmluZWQsXG4gIGZhbGxiYWNrVmlldzogWHJlZklkIHwgbnVsbCxcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogUHJvamVjdGlvbk9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuUHJvamVjdGlvbixcbiAgICB4cmVmLFxuICAgIGhhbmRsZTogbmV3IFNsb3RIYW5kbGUoKSxcbiAgICBzZWxlY3RvcixcbiAgICBpMThuUGxhY2Vob2xkZXIsXG4gICAgZmFsbGJhY2tWaWV3LFxuICAgIHByb2plY3Rpb25TbG90SW5kZXg6IDAsXG4gICAgYXR0cmlidXRlczogbnVsbCxcbiAgICBsb2NhbFJlZnM6IFtdLFxuICAgIHNvdXJjZVNwYW4sXG4gICAgLi4uTkVXX09QLFxuICAgIC4uLlRSQUlUX0NPTlNVTUVTX1NMT1QsXG4gICAgbnVtU2xvdHNVc2VkOiBmYWxsYmFja1ZpZXcgPT09IG51bGwgPyAxIDogMixcbiAgfTtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGFuIGF0dHJpYnV0ZSB0aGF0IGhhcyBiZWVuIGV4dHJhY3RlZCBmb3IgaW5jbHVzaW9uIGluIHRoZSBjb25zdHMgYXJyYXkuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXh0cmFjdGVkQXR0cmlidXRlT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4ge1xuICBraW5kOiBPcEtpbmQuRXh0cmFjdGVkQXR0cmlidXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgYFhyZWZJZGAgb2YgdGhlIHRlbXBsYXRlLWxpa2UgZWxlbWVudCB0aGUgZXh0cmFjdGVkIGF0dHJpYnV0ZSB3aWxsIGJlbG9uZyB0by5cbiAgICovXG4gIHRhcmdldDogWHJlZklkO1xuXG4gIC8qKlxuICAgKiAgVGhlIGtpbmQgb2YgYmluZGluZyByZXByZXNlbnRlZCBieSB0aGlzIGV4dHJhY3RlZCBhdHRyaWJ1dGUuXG4gICAqL1xuICBiaW5kaW5nS2luZDogQmluZGluZ0tpbmQ7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lc3BhY2Ugb2YgdGhlIGF0dHJpYnV0ZSAob3IgbnVsbCBpZiBub25lKS5cbiAgICovXG4gIG5hbWVzcGFjZTogc3RyaW5nIHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGV4dHJhY3RlZCBhdHRyaWJ1dGUuXG4gICAqL1xuICBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSB2YWx1ZSBleHByZXNzaW9uIG9mIHRoZSBleHRyYWN0ZWQgYXR0cmlidXRlLlxuICAgKi9cbiAgZXhwcmVzc2lvbjogby5FeHByZXNzaW9uIHwgbnVsbDtcblxuICAvKipcbiAgICogSWYgdGhpcyBhdHRyaWJ1dGUgaGFzIGEgY29ycmVzcG9uZGluZyBpMThuIGF0dHJpYnV0ZSAoZS5nLiBgaTE4bi1mb289XCJtOmRcImApLCB0aGVuIHRoaXMgaXMgdGhlXG4gICAqIGkxOG4gY29udGV4dCBmb3IgaXQuXG4gICAqL1xuICBpMThuQ29udGV4dDogWHJlZklkIHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGNvbnRleHQgb2YgdGhlIGJpbmRpbmcuXG4gICAqL1xuICBzZWN1cml0eUNvbnRleHQ6IFNlY3VyaXR5Q29udGV4dCB8IFNlY3VyaXR5Q29udGV4dFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgdHJ1c3RlZCB2YWx1ZSBmdW5jdGlvbiBmb3IgdGhpcyBwcm9wZXJ0eS5cbiAgICovXG4gIHRydXN0ZWRWYWx1ZUZuOiBvLkV4cHJlc3Npb24gfCBudWxsO1xuXG4gIGkxOG5NZXNzYWdlOiBpMThuLk1lc3NhZ2UgfCBudWxsO1xufVxuXG4vKipcbiAqIENyZWF0ZSBhbiBgRXh0cmFjdGVkQXR0cmlidXRlT3BgLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlRXh0cmFjdGVkQXR0cmlidXRlT3AoXG4gIHRhcmdldDogWHJlZklkLFxuICBiaW5kaW5nS2luZDogQmluZGluZ0tpbmQsXG4gIG5hbWVzcGFjZTogc3RyaW5nIHwgbnVsbCxcbiAgbmFtZTogc3RyaW5nLFxuICBleHByZXNzaW9uOiBvLkV4cHJlc3Npb24gfCBudWxsLFxuICBpMThuQ29udGV4dDogWHJlZklkIHwgbnVsbCxcbiAgaTE4bk1lc3NhZ2U6IGkxOG4uTWVzc2FnZSB8IG51bGwsXG4gIHNlY3VyaXR5Q29udGV4dDogU2VjdXJpdHlDb250ZXh0IHwgU2VjdXJpdHlDb250ZXh0W10sXG4pOiBFeHRyYWN0ZWRBdHRyaWJ1dGVPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLkV4dHJhY3RlZEF0dHJpYnV0ZSxcbiAgICB0YXJnZXQsXG4gICAgYmluZGluZ0tpbmQsXG4gICAgbmFtZXNwYWNlLFxuICAgIG5hbWUsXG4gICAgZXhwcmVzc2lvbixcbiAgICBpMThuQ29udGV4dCxcbiAgICBpMThuTWVzc2FnZSxcbiAgICBzZWN1cml0eUNvbnRleHQsXG4gICAgdHJ1c3RlZFZhbHVlRm46IG51bGwsXG4gICAgLi4uTkVXX09QLFxuICB9O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIERlZmVyT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4sIENvbnN1bWVzU2xvdE9wVHJhaXQge1xuICBraW5kOiBPcEtpbmQuRGVmZXI7XG5cbiAgLyoqXG4gICAqIFRoZSB4cmVmIG9mIHRoaXMgZGVmZXIgb3AuXG4gICAqL1xuICB4cmVmOiBYcmVmSWQ7XG5cbiAgLyoqXG4gICAqIFRoZSB4cmVmIG9mIHRoZSBtYWluIHZpZXcuXG4gICAqL1xuICBtYWluVmlldzogWHJlZklkO1xuXG4gIG1haW5TbG90OiBTbG90SGFuZGxlO1xuXG4gIC8qKlxuICAgKiBTZWNvbmRhcnkgbG9hZGluZyBibG9jayBhc3NvY2lhdGVkIHdpdGggdGhpcyBkZWZlciBvcC5cbiAgICovXG4gIGxvYWRpbmdWaWV3OiBYcmVmSWQgfCBudWxsO1xuXG4gIGxvYWRpbmdTbG90OiBTbG90SGFuZGxlIHwgbnVsbDtcblxuICAvKipcbiAgICogU2Vjb25kYXJ5IHBsYWNlaG9sZGVyIGJsb2NrIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGRlZmVyIG9wLlxuICAgKi9cbiAgcGxhY2Vob2xkZXJWaWV3OiBYcmVmSWQgfCBudWxsO1xuXG4gIHBsYWNlaG9sZGVyU2xvdDogU2xvdEhhbmRsZSB8IG51bGw7XG5cbiAgLyoqXG4gICAqIFNlY29uZGFyeSBlcnJvciBibG9jayBhc3NvY2lhdGVkIHdpdGggdGhpcyBkZWZlciBvcC5cbiAgICovXG4gIGVycm9yVmlldzogWHJlZklkIHwgbnVsbDtcblxuICBlcnJvclNsb3Q6IFNsb3RIYW5kbGUgfCBudWxsO1xuXG4gIHBsYWNlaG9sZGVyTWluaW11bVRpbWU6IG51bWJlciB8IG51bGw7XG4gIGxvYWRpbmdNaW5pbXVtVGltZTogbnVtYmVyIHwgbnVsbDtcbiAgbG9hZGluZ0FmdGVyVGltZTogbnVtYmVyIHwgbnVsbDtcblxuICBwbGFjZWhvbGRlckNvbmZpZzogby5FeHByZXNzaW9uIHwgbnVsbDtcbiAgbG9hZGluZ0NvbmZpZzogby5FeHByZXNzaW9uIHwgbnVsbDtcblxuICAvKipcbiAgICogRGVwZW5kaW5nIG9uIHRoZSBjb21waWxhdGlvbiBtb2RlLCB0aGVyZSBjYW4gYmUgZWl0aGVyIG9uZSBkZXBlbmRlbmN5IHJlc29sdXRpb24gZnVuY3Rpb25cbiAgICogcGVyIGRlZmVycmVkIGJsb2NrIG9yIG9uZSBmb3IgdGhlIGVudGlyZSB0ZW1wbGF0ZS4gVGhpcyBmaWVsZCBjb250YWlucyB0aGUgZnVuY3Rpb24gdGhhdFxuICAgKiBiZWxvbmdzIHNwZWNpZmljYWxseSB0byB0aGUgY3VycmVudCBkZWZlcnJlZCBibG9jay5cbiAgICovXG4gIG93blJlc29sdmVyRm46IG8uRXhwcmVzc2lvbiB8IG51bGw7XG5cbiAgLyoqXG4gICAqIEFmdGVyIHByb2Nlc3NpbmcsIHRoZSByZXNvbHZlciBmdW5jdGlvbiBmb3IgdGhlIGRlZmVyIGRlcHMgd2lsbCBiZSBleHRyYWN0ZWQgdG8gdGhlIGNvbnN0YW50XG4gICAqIHBvb2wsIGFuZCBhIHJlZmVyZW5jZSB0byB0aGF0IGZ1bmN0aW9uIHdpbGwgYmUgcG9wdWxhdGVkIGhlcmUuXG4gICAqL1xuICByZXNvbHZlckZuOiBvLkV4cHJlc3Npb24gfCBudWxsO1xuXG4gIHNvdXJjZVNwYW46IFBhcnNlU291cmNlU3Bhbjtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZURlZmVyT3AoXG4gIHhyZWY6IFhyZWZJZCxcbiAgbWFpbjogWHJlZklkLFxuICBtYWluU2xvdDogU2xvdEhhbmRsZSxcbiAgb3duUmVzb2x2ZXJGbjogby5FeHByZXNzaW9uIHwgbnVsbCxcbiAgcmVzb2x2ZXJGbjogby5FeHByZXNzaW9uIHwgbnVsbCxcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogRGVmZXJPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLkRlZmVyLFxuICAgIHhyZWYsXG4gICAgaGFuZGxlOiBuZXcgU2xvdEhhbmRsZSgpLFxuICAgIG1haW5WaWV3OiBtYWluLFxuICAgIG1haW5TbG90LFxuICAgIGxvYWRpbmdWaWV3OiBudWxsLFxuICAgIGxvYWRpbmdTbG90OiBudWxsLFxuICAgIGxvYWRpbmdDb25maWc6IG51bGwsXG4gICAgbG9hZGluZ01pbmltdW1UaW1lOiBudWxsLFxuICAgIGxvYWRpbmdBZnRlclRpbWU6IG51bGwsXG4gICAgcGxhY2Vob2xkZXJWaWV3OiBudWxsLFxuICAgIHBsYWNlaG9sZGVyU2xvdDogbnVsbCxcbiAgICBwbGFjZWhvbGRlckNvbmZpZzogbnVsbCxcbiAgICBwbGFjZWhvbGRlck1pbmltdW1UaW1lOiBudWxsLFxuICAgIGVycm9yVmlldzogbnVsbCxcbiAgICBlcnJvclNsb3Q6IG51bGwsXG4gICAgb3duUmVzb2x2ZXJGbixcbiAgICByZXNvbHZlckZuLFxuICAgIHNvdXJjZVNwYW4sXG4gICAgLi4uTkVXX09QLFxuICAgIC4uLlRSQUlUX0NPTlNVTUVTX1NMT1QsXG4gICAgbnVtU2xvdHNVc2VkOiAyLFxuICB9O1xufVxuaW50ZXJmYWNlIERlZmVyVHJpZ2dlckJhc2Uge1xuICBraW5kOiBEZWZlclRyaWdnZXJLaW5kO1xufVxuXG5pbnRlcmZhY2UgRGVmZXJUcmlnZ2VyV2l0aFRhcmdldEJhc2UgZXh0ZW5kcyBEZWZlclRyaWdnZXJCYXNlIHtcbiAgdGFyZ2V0TmFtZTogc3RyaW5nIHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIFhyZWYgb2YgdGhlIHRhcmdldGVkIG5hbWUuIE1heSBiZSBpbiBhIGRpZmZlcmVudCB2aWV3LlxuICAgKi9cbiAgdGFyZ2V0WHJlZjogWHJlZklkIHwgbnVsbDtcblxuICAvKipcbiAgICogVGhlIHNsb3QgaW5kZXggb2YgdGhlIG5hbWVkIHJlZmVyZW5jZSwgaW5zaWRlIHRoZSB2aWV3IHByb3ZpZGVkIGJlbG93LiBUaGlzIHNsb3QgbWF5IG5vdCBiZVxuICAgKiBpbnNpZGUgdGhlIGN1cnJlbnQgdmlldywgYW5kIGlzIGhhbmRsZWQgc3BlY2lhbGx5IGFzIGEgcmVzdWx0LlxuICAgKi9cbiAgdGFyZ2V0U2xvdDogU2xvdEhhbmRsZSB8IG51bGw7XG5cbiAgdGFyZ2V0VmlldzogWHJlZklkIHwgbnVsbDtcblxuICAvKipcbiAgICogTnVtYmVyIG9mIHN0ZXBzIHRvIHdhbGsgdXAgb3IgZG93biB0aGUgdmlldyB0cmVlIHRvIGZpbmQgdGhlIHRhcmdldCBsb2NhbFJlZi5cbiAgICovXG4gIHRhcmdldFNsb3RWaWV3U3RlcHM6IG51bWJlciB8IG51bGw7XG59XG5cbmludGVyZmFjZSBEZWZlcklkbGVUcmlnZ2VyIGV4dGVuZHMgRGVmZXJUcmlnZ2VyQmFzZSB7XG4gIGtpbmQ6IERlZmVyVHJpZ2dlcktpbmQuSWRsZTtcbn1cblxuaW50ZXJmYWNlIERlZmVySW1tZWRpYXRlVHJpZ2dlciBleHRlbmRzIERlZmVyVHJpZ2dlckJhc2Uge1xuICBraW5kOiBEZWZlclRyaWdnZXJLaW5kLkltbWVkaWF0ZTtcbn1cblxuaW50ZXJmYWNlIERlZmVySG92ZXJUcmlnZ2VyIGV4dGVuZHMgRGVmZXJUcmlnZ2VyV2l0aFRhcmdldEJhc2Uge1xuICBraW5kOiBEZWZlclRyaWdnZXJLaW5kLkhvdmVyO1xufVxuXG5pbnRlcmZhY2UgRGVmZXJUaW1lclRyaWdnZXIgZXh0ZW5kcyBEZWZlclRyaWdnZXJCYXNlIHtcbiAga2luZDogRGVmZXJUcmlnZ2VyS2luZC5UaW1lcjtcblxuICBkZWxheTogbnVtYmVyO1xufVxuXG5pbnRlcmZhY2UgRGVmZXJJbnRlcmFjdGlvblRyaWdnZXIgZXh0ZW5kcyBEZWZlclRyaWdnZXJXaXRoVGFyZ2V0QmFzZSB7XG4gIGtpbmQ6IERlZmVyVHJpZ2dlcktpbmQuSW50ZXJhY3Rpb247XG59XG5cbmludGVyZmFjZSBEZWZlclZpZXdwb3J0VHJpZ2dlciBleHRlbmRzIERlZmVyVHJpZ2dlcldpdGhUYXJnZXRCYXNlIHtcbiAga2luZDogRGVmZXJUcmlnZ2VyS2luZC5WaWV3cG9ydDtcbn1cblxuLyoqXG4gKiBUaGUgdW5pb24gdHlwZSBvZiBhbGwgZGVmZXIgdHJpZ2dlciBpbnRlcmZhY2VzLlxuICovXG5leHBvcnQgdHlwZSBEZWZlclRyaWdnZXIgPVxuICB8IERlZmVySWRsZVRyaWdnZXJcbiAgfCBEZWZlckltbWVkaWF0ZVRyaWdnZXJcbiAgfCBEZWZlclRpbWVyVHJpZ2dlclxuICB8IERlZmVySG92ZXJUcmlnZ2VyXG4gIHwgRGVmZXJJbnRlcmFjdGlvblRyaWdnZXJcbiAgfCBEZWZlclZpZXdwb3J0VHJpZ2dlcjtcblxuZXhwb3J0IGludGVyZmFjZSBEZWZlck9uT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4ge1xuICBraW5kOiBPcEtpbmQuRGVmZXJPbjtcblxuICBkZWZlcjogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBUaGUgdHJpZ2dlciBmb3IgdGhpcyBkZWZlciBvcCAoZS5nLiBpZGxlLCBob3ZlciwgZXRjKS5cbiAgICovXG4gIHRyaWdnZXI6IERlZmVyVHJpZ2dlcjtcblxuICAvKipcbiAgICogV2hldGhlciB0byBlbWl0IHRoZSBwcmVmZXRjaCB2ZXJzaW9uIG9mIHRoZSBpbnN0cnVjdGlvbi5cbiAgICovXG4gIHByZWZldGNoOiBib29sZWFuO1xuXG4gIHNvdXJjZVNwYW46IFBhcnNlU291cmNlU3Bhbjtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZURlZmVyT25PcChcbiAgZGVmZXI6IFhyZWZJZCxcbiAgdHJpZ2dlcjogRGVmZXJUcmlnZ2VyLFxuICBwcmVmZXRjaDogYm9vbGVhbixcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogRGVmZXJPbk9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuRGVmZXJPbixcbiAgICBkZWZlcixcbiAgICB0cmlnZ2VyLFxuICAgIHByZWZldGNoLFxuICAgIHNvdXJjZVNwYW4sXG4gICAgLi4uTkVXX09QLFxuICB9O1xufVxuXG4vKipcbiAqIE9wIHRoYXQgcmVzZXJ2ZXMgYSBzbG90IGR1cmluZyBjcmVhdGlvbiB0aW1lIGZvciBhIGBAbGV0YCBkZWNsYXJhdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEZWNsYXJlTGV0T3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4sIENvbnN1bWVzU2xvdE9wVHJhaXQge1xuICBraW5kOiBPcEtpbmQuRGVjbGFyZUxldDtcbiAgeHJlZjogWHJlZklkO1xuICBzb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW47XG4gIGRlY2xhcmVkTmFtZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIENyZWF0ZXMgYSBgRGVjbGFyZUxldE9wYC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZURlY2xhcmVMZXRPcChcbiAgeHJlZjogWHJlZklkLFxuICBkZWNsYXJlZE5hbWU6IHN0cmluZyxcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogRGVjbGFyZUxldE9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuRGVjbGFyZUxldCxcbiAgICB4cmVmLFxuICAgIGRlY2xhcmVkTmFtZSxcbiAgICBzb3VyY2VTcGFuLFxuICAgIGhhbmRsZTogbmV3IFNsb3RIYW5kbGUoKSxcbiAgICAuLi5UUkFJVF9DT05TVU1FU19TTE9ULFxuICAgIC4uLk5FV19PUCxcbiAgfTtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgc2luZ2xlIHZhbHVlIGluIGFuIGkxOG4gcGFyYW0gbWFwLiBFYWNoIHBsYWNlaG9sZGVyIGluIHRoZSBtYXAgbWF5IGhhdmUgbXVsdGlwbGUgb2ZcbiAqIHRoZXNlIHZhbHVlcyBhc3NvY2lhdGVkIHdpdGggaXQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSTE4blBhcmFtVmFsdWUge1xuICAvKipcbiAgICogVGhlIHZhbHVlLiBUaGlzIGNhbiBiZSBlaXRoZXIgYSBzbG90IG51bWJlciwgc3BlY2lhbCBzdHJpbmcsIG9yIGNvbXBvdW5kLXZhbHVlIGNvbnNpc3Rpbmcgb2YgYW5cbiAgICogZWxlbWVudCBzbG90IG51bWJlciBhbmQgdGVtcGxhdGUgc2xvdCBudW1iZXIuXG4gICAqL1xuICB2YWx1ZTogc3RyaW5nIHwgbnVtYmVyIHwge2VsZW1lbnQ6IG51bWJlcjsgdGVtcGxhdGU6IG51bWJlcn07XG5cbiAgLyoqXG4gICAqIFRoZSBzdWItdGVtcGxhdGUgaW5kZXggYXNzb2NpYXRlZCB3aXRoIHRoZSB2YWx1ZS5cbiAgICovXG4gIHN1YlRlbXBsYXRlSW5kZXg6IG51bWJlciB8IG51bGw7XG5cbiAgLyoqXG4gICAqIEZsYWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgdmFsdWUuXG4gICAqL1xuICBmbGFnczogSTE4blBhcmFtVmFsdWVGbGFncztcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGFuIGkxOG4gbWVzc2FnZSB0aGF0IGhhcyBiZWVuIGV4dHJhY3RlZCBmb3IgaW5jbHVzaW9uIGluIHRoZSBjb25zdHMgYXJyYXkuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSTE4bk1lc3NhZ2VPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiB7XG4gIGtpbmQ6IE9wS2luZC5JMThuTWVzc2FnZTtcblxuICAvKipcbiAgICogQW4gaWQgdXNlZCB0byByZWZlcmVuY2UgdGhpcyBtZXNzYWdlLlxuICAgKi9cbiAgeHJlZjogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBUaGUgY29udGV4dCBmcm9tIHdoaWNoIHRoaXMgbWVzc2FnZSB3YXMgZXh0cmFjdGVkXG4gICAqIFRPRE86IHJlbW92ZSB0aGlzLCBhbmQgYWRkIGFub3RoZXIgcHJvcGVydHkgaGVyZSBpbnN0ZWFkIHRvIG1hdGNoIEV4dHJhY3RlZEF0dHJpYnV0ZXNcbiAgICovXG4gIGkxOG5Db250ZXh0OiBYcmVmSWQ7XG5cbiAgLyoqXG4gICAqIEEgcmVmZXJlbmNlIHRvIHRoZSBpMThuIG9wIHRoaXMgbWVzc2FnZSB3YXMgZXh0cmFjdGVkIGZyb20uXG4gICAqXG4gICAqIFRoaXMgbWlnaHQgYmUgbnVsbCwgd2hpY2ggbWVhbnMgdGhpcyBtZXNzYWdlIGlzIG5vdCBhc3NvY2lhdGVkIHdpdGggYSBibG9jay4gVGhpcyBwcm9iYWJseVxuICAgKiBtZWFucyBpdCBpcyBhbiBpMThuIGF0dHJpYnV0ZSdzIG1lc3NhZ2UuXG4gICAqL1xuICBpMThuQmxvY2s6IFhyZWZJZCB8IG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBpMThuIG1lc3NhZ2UgcmVwcmVzZW50ZWQgYnkgdGhpcyBvcC5cbiAgICovXG4gIG1lc3NhZ2U6IGkxOG4uTWVzc2FnZTtcblxuICAvKipcbiAgICogVGhlIHBsYWNlaG9sZGVyIHVzZWQgZm9yIHRoaXMgbWVzc2FnZSB3aGVuIGl0IGlzIHJlZmVyZW5jZWQgaW4gYW5vdGhlciBtZXNzYWdlLlxuICAgKiBGb3IgYSB0b3AtbGV2ZWwgbWVzc2FnZSB0aGF0IGlzbid0IHJlZmVyZW5jZWQgZnJvbSBhbm90aGVyIG1lc3NhZ2UsIHRoaXMgd2lsbCBiZSBudWxsLlxuICAgKi9cbiAgbWVzc2FnZVBsYWNlaG9sZGVyOiBzdHJpbmcgfCBudWxsO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgbWVzc2FnZSBuZWVkcyBwb3N0LXByb2Nlc3NpbmcuXG4gICAqL1xuICBuZWVkc1Bvc3Rwcm9jZXNzaW5nOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgcGFyYW0gbWFwLCB3aXRoIHBsYWNlaG9sZGVycyByZXByZXNlbnRlZCBhcyBhbiBgRXhwcmVzc2lvbmAuXG4gICAqL1xuICBwYXJhbXM6IE1hcDxzdHJpbmcsIG8uRXhwcmVzc2lvbj47XG5cbiAgLyoqXG4gICAqIFRoZSBwb3N0LXByb2Nlc3NpbmcgcGFyYW0gbWFwLCB3aXRoIHBsYWNlaG9sZGVycyByZXByZXNlbnRlZCBhcyBhbiBgRXhwcmVzc2lvbmAuXG4gICAqL1xuICBwb3N0cHJvY2Vzc2luZ1BhcmFtczogTWFwPHN0cmluZywgby5FeHByZXNzaW9uPjtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIHN1Yi1tZXNzYWdlcyB0aGF0IGFyZSByZWZlcmVuY2VkIGJ5IHRoaXMgbWVzc2FnZS5cbiAgICovXG4gIHN1Yk1lc3NhZ2VzOiBYcmVmSWRbXTtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYW4gYEV4dHJhY3RlZE1lc3NhZ2VPcGAuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVJMThuTWVzc2FnZU9wKFxuICB4cmVmOiBYcmVmSWQsXG4gIGkxOG5Db250ZXh0OiBYcmVmSWQsXG4gIGkxOG5CbG9jazogWHJlZklkIHwgbnVsbCxcbiAgbWVzc2FnZTogaTE4bi5NZXNzYWdlLFxuICBtZXNzYWdlUGxhY2Vob2xkZXI6IHN0cmluZyB8IG51bGwsXG4gIHBhcmFtczogTWFwPHN0cmluZywgby5FeHByZXNzaW9uPixcbiAgcG9zdHByb2Nlc3NpbmdQYXJhbXM6IE1hcDxzdHJpbmcsIG8uRXhwcmVzc2lvbj4sXG4gIG5lZWRzUG9zdHByb2Nlc3Npbmc6IGJvb2xlYW4sXG4pOiBJMThuTWVzc2FnZU9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuSTE4bk1lc3NhZ2UsXG4gICAgeHJlZixcbiAgICBpMThuQ29udGV4dCxcbiAgICBpMThuQmxvY2ssXG4gICAgbWVzc2FnZSxcbiAgICBtZXNzYWdlUGxhY2Vob2xkZXIsXG4gICAgcGFyYW1zLFxuICAgIHBvc3Rwcm9jZXNzaW5nUGFyYW1zLFxuICAgIG5lZWRzUG9zdHByb2Nlc3NpbmcsXG4gICAgc3ViTWVzc2FnZXM6IFtdLFxuICAgIC4uLk5FV19PUCxcbiAgfTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJMThuT3BCYXNlIGV4dGVuZHMgT3A8Q3JlYXRlT3A+LCBDb25zdW1lc1Nsb3RPcFRyYWl0IHtcbiAga2luZDogT3BLaW5kLkkxOG5TdGFydCB8IE9wS2luZC5JMThuO1xuXG4gIC8qKlxuICAgKiBgWHJlZklkYCBhbGxvY2F0ZWQgZm9yIHRoaXMgaTE4biBibG9jay5cbiAgICovXG4gIHhyZWY6IFhyZWZJZDtcblxuICAvKipcbiAgICogQSByZWZlcmVuY2UgdG8gdGhlIHJvb3QgaTE4biBibG9jayB0aGF0IHRoaXMgb25lIGJlbG9uZ3MgdG8uIEZvciBhIHJvb3QgaTE4biBibG9jaywgdGhpcyBpc1xuICAgKiB0aGUgc2FtZSBhcyB4cmVmLlxuICAgKi9cbiAgcm9vdDogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBUaGUgaTE4biBtZXRhZGF0YSBhc3NvY2lhdGVkIHdpdGggdGhpcyBvcC5cbiAgICovXG4gIG1lc3NhZ2U6IGkxOG4uTWVzc2FnZTtcblxuICAvKipcbiAgICogVGhlIGluZGV4IGluIHRoZSBjb25zdHMgYXJyYXkgd2hlcmUgdGhlIG1lc3NhZ2UgaTE4biBtZXNzYWdlIGlzIHN0b3JlZC5cbiAgICovXG4gIG1lc3NhZ2VJbmRleDogQ29uc3RJbmRleCB8IG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBpbmRleCBvZiB0aGlzIHN1Yi1ibG9jayBpbiB0aGUgaTE4biBtZXNzYWdlLiBGb3IgYSByb290IGkxOG4gYmxvY2ssIHRoaXMgaXMgbnVsbC5cbiAgICovXG4gIHN1YlRlbXBsYXRlSW5kZXg6IG51bWJlciB8IG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBpMThuIGNvbnRleHQgZ2VuZXJhdGVkIGZyb20gdGhpcyBibG9jay4gSW5pdGlhbGx5IG51bGwsIHVudGlsIHRoZSBjb250ZXh0IGlzIGNyZWF0ZWQuXG4gICAqL1xuICBjb250ZXh0OiBYcmVmSWQgfCBudWxsO1xuXG4gIHNvdXJjZVNwYW46IFBhcnNlU291cmNlU3BhbiB8IG51bGw7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBlbXB0eSBpMThuIGJsb2NrLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEkxOG5PcCBleHRlbmRzIEkxOG5PcEJhc2Uge1xuICBraW5kOiBPcEtpbmQuSTE4bjtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBzdGFydCBvZiBhbiBpMThuIGJsb2NrLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEkxOG5TdGFydE9wIGV4dGVuZHMgSTE4bk9wQmFzZSB7XG4gIGtpbmQ6IE9wS2luZC5JMThuU3RhcnQ7XG59XG5cbi8qKlxuICogQ3JlYXRlIGFuIGBJMThuU3RhcnRPcGAuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVJMThuU3RhcnRPcChcbiAgeHJlZjogWHJlZklkLFxuICBtZXNzYWdlOiBpMThuLk1lc3NhZ2UsXG4gIHJvb3Q6IFhyZWZJZCB8IHVuZGVmaW5lZCxcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuIHwgbnVsbCxcbik6IEkxOG5TdGFydE9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuSTE4blN0YXJ0LFxuICAgIHhyZWYsXG4gICAgaGFuZGxlOiBuZXcgU2xvdEhhbmRsZSgpLFxuICAgIHJvb3Q6IHJvb3QgPz8geHJlZixcbiAgICBtZXNzYWdlLFxuICAgIG1lc3NhZ2VJbmRleDogbnVsbCxcbiAgICBzdWJUZW1wbGF0ZUluZGV4OiBudWxsLFxuICAgIGNvbnRleHQ6IG51bGwsXG4gICAgc291cmNlU3BhbixcbiAgICAuLi5ORVdfT1AsXG4gICAgLi4uVFJBSVRfQ09OU1VNRVNfU0xPVCxcbiAgfTtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBlbmQgb2YgYW4gaTE4biBibG9jay5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJMThuRW5kT3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4ge1xuICBraW5kOiBPcEtpbmQuSTE4bkVuZDtcblxuICAvKipcbiAgICogVGhlIGBYcmVmSWRgIG9mIHRoZSBgSTE4blN0YXJ0T3BgIHRoYXQgY3JlYXRlZCB0aGlzIGJsb2NrLlxuICAgKi9cbiAgeHJlZjogWHJlZklkO1xuXG4gIHNvdXJjZVNwYW46IFBhcnNlU291cmNlU3BhbiB8IG51bGw7XG59XG5cbi8qKlxuICogQ3JlYXRlIGFuIGBJMThuRW5kT3BgLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlSTE4bkVuZE9wKHhyZWY6IFhyZWZJZCwgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuIHwgbnVsbCk6IEkxOG5FbmRPcCB7XG4gIHJldHVybiB7XG4gICAga2luZDogT3BLaW5kLkkxOG5FbmQsXG4gICAgeHJlZixcbiAgICBzb3VyY2VTcGFuLFxuICAgIC4uLk5FV19PUCxcbiAgfTtcbn1cblxuLyoqXG4gKiBBbiBvcCB0aGF0IHJlcHJlc2VudHMgdGhlIHN0YXJ0IG9mIGFuIElDVSBleHByZXNzaW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEljdVN0YXJ0T3AgZXh0ZW5kcyBPcDxDcmVhdGVPcD4ge1xuICBraW5kOiBPcEtpbmQuSWN1U3RhcnQ7XG5cbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgSUNVLlxuICAgKi9cbiAgeHJlZjogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBUaGUgaTE4biBtZXNzYWdlIGZvciB0aGlzIElDVS5cbiAgICovXG4gIG1lc3NhZ2U6IGkxOG4uTWVzc2FnZTtcblxuICAvKipcbiAgICogUGxhY2Vob2xkZXIgdXNlZCB0byByZWZlcmVuY2UgdGhpcyBJQ1UgaW4gb3RoZXIgaTE4biBtZXNzYWdlcy5cbiAgICovXG4gIG1lc3NhZ2VQbGFjZWhvbGRlcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIHJlZmVyZW5jZSB0byB0aGUgaTE4biBjb250ZXh0IGZvciB0aGlzIG9wLiBJbml0aWFsbHkgbnVsbCwgdW50aWwgdGhlIGNvbnRleHQgaXMgY3JlYXRlZC5cbiAgICovXG4gIGNvbnRleHQ6IFhyZWZJZCB8IG51bGw7XG5cbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuO1xufVxuXG4vKipcbiAqIENyZWF0ZXMgYW4gSUNVIHN0YXJ0IG9wLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlSWN1U3RhcnRPcChcbiAgeHJlZjogWHJlZklkLFxuICBtZXNzYWdlOiBpMThuLk1lc3NhZ2UsXG4gIG1lc3NhZ2VQbGFjZWhvbGRlcjogc3RyaW5nLFxuICBzb3VyY2VTcGFuOiBQYXJzZVNvdXJjZVNwYW4sXG4pOiBJY3VTdGFydE9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuSWN1U3RhcnQsXG4gICAgeHJlZixcbiAgICBtZXNzYWdlLFxuICAgIG1lc3NhZ2VQbGFjZWhvbGRlcixcbiAgICBjb250ZXh0OiBudWxsLFxuICAgIHNvdXJjZVNwYW4sXG4gICAgLi4uTkVXX09QLFxuICB9O1xufVxuXG4vKipcbiAqIEFuIG9wIHRoYXQgcmVwcmVzZW50cyB0aGUgZW5kIG9mIGFuIElDVSBleHByZXNzaW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEljdUVuZE9wIGV4dGVuZHMgT3A8Q3JlYXRlT3A+IHtcbiAga2luZDogT3BLaW5kLkljdUVuZDtcblxuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSBjb3JyZXNwb25kaW5nIEljdVN0YXJ0T3AuXG4gICAqL1xuICB4cmVmOiBYcmVmSWQ7XG59XG5cbi8qKlxuICogQ3JlYXRlcyBhbiBJQ1UgZW5kIG9wLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlSWN1RW5kT3AoeHJlZjogWHJlZklkKTogSWN1RW5kT3Age1xuICByZXR1cm4ge1xuICAgIGtpbmQ6IE9wS2luZC5JY3VFbmQsXG4gICAgeHJlZixcbiAgICAuLi5ORVdfT1AsXG4gIH07XG59XG5cbi8qKlxuICogQW4gb3AgdGhhdCByZXByZXNlbnRzIGEgcGxhY2Vob2xkZXIgaW4gYW4gSUNVIGV4cHJlc3Npb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSWN1UGxhY2Vob2xkZXJPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiB7XG4gIGtpbmQ6IE9wS2luZC5JY3VQbGFjZWhvbGRlcjtcblxuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSBJQ1UgcGxhY2Vob2xkZXIuXG4gICAqL1xuICB4cmVmOiBYcmVmSWQ7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBwbGFjZWhvbGRlciBpbiB0aGUgSUNVIGV4cHJlc3Npb24uXG4gICAqL1xuICBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBzdGF0aWMgc3RyaW5ncyB0byBiZSBjb21iaW5lZCB3aXRoIGR5bmFtaWMgZXhwcmVzc2lvbiB2YWx1ZXMgdG8gZm9ybSB0aGUgdGV4dC4gVGhpcyB3b3Jrc1xuICAgKiBsaWtlIGludGVycG9sYXRpb24sIGJ1dCB0aGUgc3RyaW5ncyBhcmUgY29tYmluZWQgYXQgY29tcGlsZSB0aW1lLCB1c2luZyBzcGVjaWFsIHBsYWNlaG9sZGVyc1xuICAgKiBmb3IgdGhlIGR5bmFtaWMgZXhwcmVzc2lvbnMsIGFuZCBwdXQgaW50byB0aGUgdHJhbnNsYXRlZCBtZXNzYWdlLlxuICAgKi9cbiAgc3RyaW5nczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFBsYWNlaG9sZGVyIHZhbHVlcyBmb3IgdGhlIGkxOG4gZXhwcmVzc2lvbnMgdG8gYmUgY29tYmluZWQgd2l0aCB0aGUgc3RhdGljIHN0cmluZ3MgdG8gZm9ybSB0aGVcbiAgICogZnVsbCBwbGFjZWhvbGRlciB2YWx1ZS5cbiAgICovXG4gIGV4cHJlc3Npb25QbGFjZWhvbGRlcnM6IEkxOG5QYXJhbVZhbHVlW107XG59XG5cbi8qKlxuICogQ3JlYXRlcyBhbiBJQ1UgcGxhY2Vob2xkZXIgb3AuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjcmVhdGVJY3VQbGFjZWhvbGRlck9wKFxuICB4cmVmOiBYcmVmSWQsXG4gIG5hbWU6IHN0cmluZyxcbiAgc3RyaW5nczogc3RyaW5nW10sXG4pOiBJY3VQbGFjZWhvbGRlck9wIHtcbiAgcmV0dXJuIHtcbiAgICBraW5kOiBPcEtpbmQuSWN1UGxhY2Vob2xkZXIsXG4gICAgeHJlZixcbiAgICBuYW1lLFxuICAgIHN0cmluZ3MsXG4gICAgZXhwcmVzc2lvblBsYWNlaG9sZGVyczogW10sXG4gICAgLi4uTkVXX09QLFxuICB9O1xufVxuXG4vKipcbiAqIEFuIGkxOG4gY29udGV4dCB0aGF0IGlzIHVzZWQgdG8gZ2VuZXJhdGUgYSB0cmFuc2xhdGVkIGkxOG4gbWVzc2FnZS4gQSBzZXBhcmF0ZSBjb250ZXh0IGlzIGNyZWF0ZWRcbiAqIGZvciB0aHJlZSBkaWZmZXJlbnQgc2NlbmFyaW9zOlxuICpcbiAqIDEuIEZvciBlYWNoIHRvcC1sZXZlbCBpMThuIGJsb2NrLlxuICogMi4gRm9yIGVhY2ggSUNVIHJlZmVyZW5jZWQgYXMgYSBzdWItbWVzc2FnZS4gSUNVcyB0aGF0IGFyZSByZWZlcmVuY2VkIGFzIGEgc3ViLW1lc3NhZ2Ugd2lsbCBiZVxuICogICAgdXNlZCB0byBnZW5lcmF0ZSBhIHNlcGFyYXRlIGkxOG4gbWVzc2FnZSwgYnV0IHdpbGwgbm90IGJlIGV4dHJhY3RlZCBkaXJlY3RseSBpbnRvIHRoZSBjb25zdHNcbiAqICAgIGFycmF5LiBJbnN0ZWFkIHRoZXkgd2lsbCBiZSBwdWxsZWQgaW4gYXMgcGFydCBvZiB0aGUgaW5pdGlhbGl6YXRpb24gc3RhdGVtZW50cyBmb3IgdGhlIG1lc3NhZ2VcbiAqICAgIHRoYXQgcmVmZXJlbmNlcyB0aGVtLlxuICogMy4gRm9yIGVhY2ggaTE4biBhdHRyaWJ1dGUuXG4gKlxuICogQ2hpbGQgaTE4biBibG9ja3MsIHJlc3VsdGluZyBmcm9tIHRoZSB1c2Ugb2YgYW4gbmctdGVtcGxhdGUgaW5zaWRlIG9mIGEgcGFyZW50IGkxOG4gYmxvY2ssIGRvIG5vdFxuICogZ2VuZXJhdGUgYSBzZXBhcmF0ZSBjb250ZXh0LiBJbnN0ZWFkIHRoZWlyIGNvbnRlbnQgaXMgaW5jbHVkZWQgaW4gdGhlIHRyYW5zbGF0ZWQgbWVzc2FnZSBmb3JcbiAqIHRoZWlyIHJvb3QgYmxvY2suXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSTE4bkNvbnRleHRPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiB7XG4gIGtpbmQ6IE9wS2luZC5JMThuQ29udGV4dDtcblxuICBjb250ZXh0S2luZDogSTE4bkNvbnRleHRLaW5kO1xuXG4gIC8qKlxuICAgKiBUaGUgaWQgb2YgdGhpcyBjb250ZXh0LlxuICAgKi9cbiAgeHJlZjogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBBIHJlZmVyZW5jZSB0byB0aGUgSTE4blN0YXJ0T3Agb3IgSTE4bk9wIHRoaXMgY29udGV4dCBiZWxvbmdzIHRvLlxuICAgKlxuICAgKiBJdCBpcyBwb3NzaWJsZSBmb3IgbXVsdGlwbGUgY29udGV4dHMgdG8gYmVsb25nIHRvIHRoZSBzYW1lIGJsb2NrLCBzaW5jZSBib3RoIHRoZSBibG9jayBhbmQgYW55XG4gICAqIElDVXMgaW5zaWRlIHRoZSBibG9jayB3aWxsIGVhY2ggZ2V0IHRoZWlyIG93biBjb250ZXh0LlxuICAgKlxuICAgKiBUaGlzIG1pZ2h0IGJlIGBudWxsYCwgaW4gd2hpY2ggY2FzZSB0aGUgY29udGV4dCBpcyBub3QgYXNzb2NpYXRlZCB3aXRoIGFuIGkxOG4gYmxvY2suIFRoaXNcbiAgICogcHJvYmFibHkgbWVhbnMgdGhhdCBpdCBiZWxvbmdzIHRvIGFuIGkxOG4gYXR0cmlidXRlLlxuICAgKi9cbiAgaTE4bkJsb2NrOiBYcmVmSWQgfCBudWxsO1xuXG4gIC8qKlxuICAgKiBUaGUgaTE4biBtZXNzYWdlIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGNvbnRleHQuXG4gICAqL1xuICBtZXNzYWdlOiBpMThuLk1lc3NhZ2U7XG5cbiAgLyoqXG4gICAqIFRoZSBwYXJhbSBtYXAgZm9yIHRoaXMgY29udGV4dC5cbiAgICovXG4gIHBhcmFtczogTWFwPHN0cmluZywgSTE4blBhcmFtVmFsdWVbXT47XG5cbiAgLyoqXG4gICAqIFRoZSBwb3N0LXByb2Nlc3NpbmcgcGFyYW0gbWFwIGZvciB0aGlzIGNvbnRleHQuXG4gICAqL1xuICBwb3N0cHJvY2Vzc2luZ1BhcmFtczogTWFwPHN0cmluZywgSTE4blBhcmFtVmFsdWVbXT47XG5cbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlSTE4bkNvbnRleHRPcChcbiAgY29udGV4dEtpbmQ6IEkxOG5Db250ZXh0S2luZCxcbiAgeHJlZjogWHJlZklkLFxuICBpMThuQmxvY2s6IFhyZWZJZCB8IG51bGwsXG4gIG1lc3NhZ2U6IGkxOG4uTWVzc2FnZSxcbiAgc291cmNlU3BhbjogUGFyc2VTb3VyY2VTcGFuLFxuKTogSTE4bkNvbnRleHRPcCB7XG4gIGlmIChpMThuQmxvY2sgPT09IG51bGwgJiYgY29udGV4dEtpbmQgIT09IEkxOG5Db250ZXh0S2luZC5BdHRyKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdBc3NlcnRpb25FcnJvcjogaTE4bkJsb2NrIG11c3QgYmUgcHJvdmlkZWQgZm9yIG5vbi1hdHRyaWJ1dGUgY29udGV4dHMuJyk7XG4gIH1cblxuICByZXR1cm4ge1xuICAgIGtpbmQ6IE9wS2luZC5JMThuQ29udGV4dCxcbiAgICBjb250ZXh0S2luZCxcbiAgICB4cmVmLFxuICAgIGkxOG5CbG9jayxcbiAgICBtZXNzYWdlLFxuICAgIHNvdXJjZVNwYW4sXG4gICAgcGFyYW1zOiBuZXcgTWFwKCksXG4gICAgcG9zdHByb2Nlc3NpbmdQYXJhbXM6IG5ldyBNYXAoKSxcbiAgICAuLi5ORVdfT1AsXG4gIH07XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSTE4bkF0dHJpYnV0ZXNPcCBleHRlbmRzIE9wPENyZWF0ZU9wPiwgQ29uc3VtZXNTbG90T3BUcmFpdCB7XG4gIGtpbmQ6IE9wS2luZC5JMThuQXR0cmlidXRlcztcblxuICAvKipcbiAgICogVGhlIGVsZW1lbnQgdGFyZ2V0ZWQgYnkgdGhlc2UgYXR0cmlidXRlcy5cbiAgICovXG4gIHRhcmdldDogWHJlZklkO1xuXG4gIC8qKlxuICAgKiBJMThuQXR0cmlidXRlcyBpbnN0cnVjdGlvbnMgY29ycmVzcG9uZCB0byBhIGNvbnN0IGFycmF5IHdpdGggY29uZmlndXJhdGlvbiBpbmZvcm1hdGlvbi5cbiAgICovXG4gIGkxOG5BdHRyaWJ1dGVzQ29uZmlnOiBDb25zdEluZGV4IHwgbnVsbDtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUkxOG5BdHRyaWJ1dGVzT3AoXG4gIHhyZWY6IFhyZWZJZCxcbiAgaGFuZGxlOiBTbG90SGFuZGxlLFxuICB0YXJnZXQ6IFhyZWZJZCxcbik6IEkxOG5BdHRyaWJ1dGVzT3Age1xuICByZXR1cm4ge1xuICAgIGtpbmQ6IE9wS2luZC5JMThuQXR0cmlidXRlcyxcbiAgICB4cmVmLFxuICAgIGhhbmRsZSxcbiAgICB0YXJnZXQsXG4gICAgaTE4bkF0dHJpYnV0ZXNDb25maWc6IG51bGwsXG4gICAgLi4uTkVXX09QLFxuICAgIC4uLlRSQUlUX0NPTlNVTUVTX1NMT1QsXG4gIH07XG59XG5cbi8qKlxuICogQW4gaW5kZXggaW50byB0aGUgYGNvbnN0c2AgYXJyYXkgd2hpY2ggaXMgc2hhcmVkIGFjcm9zcyB0aGUgY29tcGlsYXRpb24gb2YgYWxsIHZpZXdzIGluIGFcbiAqIGNvbXBvbmVudC5cbiAqL1xuZXhwb3J0IHR5cGUgQ29uc3RJbmRleCA9IG51bWJlciAmIHtfX2JyYW5kOiAnQ29uc3RJbmRleCd9O1xuIl19