/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { OpKind, } from '../enums';
import { TRAIT_CONSUMES_VARS, TRAIT_DEPENDS_ON_SLOT_CONTEXT, } from '../traits';
import { NEW_OP } from './shared';
/**
 * Create an `InterpolationTextOp`.
 */
export function createInterpolateTextOp(xref, interpolation, sourceSpan) {
    return {
        kind: OpKind.InterpolateText,
        target: xref,
        interpolation,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
export class Interpolation {
    constructor(strings, expressions, i18nPlaceholders) {
        this.strings = strings;
        this.expressions = expressions;
        this.i18nPlaceholders = i18nPlaceholders;
        if (i18nPlaceholders.length !== 0 && i18nPlaceholders.length !== expressions.length) {
            throw new Error(`Expected ${expressions.length} placeholders to match interpolation expression count, but got ${i18nPlaceholders.length}`);
        }
    }
}
/**
 * Create a `BindingOp`, not yet transformed into a particular type of binding.
 */
export function createBindingOp(target, kind, name, expression, unit, securityContext, isTextAttribute, isStructuralTemplateAttribute, templateKind, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.Binding,
        bindingKind: kind,
        target,
        name,
        expression,
        unit,
        securityContext,
        isTextAttribute,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext: null,
        i18nMessage,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a `PropertyOp`.
 */
export function createPropertyOp(target, name, expression, isAnimationTrigger, securityContext, isStructuralTemplateAttribute, templateKind, i18nContext, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.Property,
        target,
        name,
        expression,
        isAnimationTrigger,
        securityContext,
        sanitizer: null,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext,
        i18nMessage,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `TwoWayPropertyOp`.
 */
export function createTwoWayPropertyOp(target, name, expression, securityContext, isStructuralTemplateAttribute, templateKind, i18nContext, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.TwoWayProperty,
        target,
        name,
        expression,
        securityContext,
        sanitizer: null,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext,
        i18nMessage,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StylePropOp`. */
export function createStylePropOp(xref, name, expression, unit, sourceSpan) {
    return {
        kind: OpKind.StyleProp,
        target: xref,
        name,
        expression,
        unit,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassPropOp`.
 */
export function createClassPropOp(xref, name, expression, sourceSpan) {
    return {
        kind: OpKind.ClassProp,
        target: xref,
        name,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StyleMapOp`. */
export function createStyleMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.StyleMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassMapOp`.
 */
export function createClassMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.ClassMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AttributeOp`.
 */
export function createAttributeOp(target, namespace, name, expression, securityContext, isTextAttribute, isStructuralTemplateAttribute, templateKind, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.Attribute,
        target,
        namespace,
        name,
        expression,
        securityContext,
        sanitizer: null,
        isTextAttribute,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext: null,
        i18nMessage,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AdvanceOp`.
 */
export function createAdvanceOp(delta, sourceSpan) {
    return {
        kind: OpKind.Advance,
        delta,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a conditional op, which will display an embedded view according to a condtion.
 */
export function createConditionalOp(target, test, conditions, sourceSpan) {
    return {
        kind: OpKind.Conditional,
        target,
        test,
        conditions,
        processed: null,
        sourceSpan,
        contextValue: null,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
    };
}
export function createRepeaterOp(repeaterCreate, targetSlot, collection, sourceSpan) {
    return {
        kind: OpKind.Repeater,
        target: repeaterCreate,
        targetSlot,
        collection,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
    };
}
export function createDeferWhenOp(target, expr, prefetch, sourceSpan) {
    return {
        kind: OpKind.DeferWhen,
        target,
        expr,
        prefetch,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
    };
}
/**
 * Create an i18n expression op.
 */
export function createI18nExpressionOp(context, target, i18nOwner, handle, expression, icuPlaceholder, i18nPlaceholder, resolutionTime, usage, name, sourceSpan) {
    return {
        kind: OpKind.I18nExpression,
        context,
        target,
        i18nOwner,
        handle,
        expression,
        icuPlaceholder,
        i18nPlaceholder,
        resolutionTime,
        usage,
        name,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_VARS,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
    };
}
/**
 * Creates an op to apply i18n expression ops.
 */
export function createI18nApplyOp(owner, handle, sourceSpan) {
    return {
        kind: OpKind.I18nApply,
        owner,
        handle,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates a `StoreLetOp`.
 */
export function createStoreLetOp(target, declaredName, value, sourceSpan) {
    return {
        kind: OpKind.StoreLet,
        target,
        declaredName,
        value,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
//# sourceMappingURL=data:application/json;base64,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