/**
 *
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import * as o from '../../../../src/output/output_ast';
import * as ir from '../ir';
import { CompilationJobKind as Kind, } from './compilation';
import { deleteAnyCasts } from './phases/any_cast';
import { applyI18nExpressions } from './phases/apply_i18n_expressions';
import { assignI18nSlotDependencies } from './phases/assign_i18n_slot_dependencies';
import { extractAttributes } from './phases/attribute_extraction';
import { specializeBindings } from './phases/binding_specialization';
import { chain } from './phases/chaining';
import { collapseSingletonInterpolations } from './phases/collapse_singleton_interpolations';
import { generateConditionalExpressions } from './phases/conditionals';
import { collectElementConsts } from './phases/const_collection';
import { convertI18nBindings } from './phases/convert_i18n_bindings';
import { resolveDeferDepsFns } from './phases/resolve_defer_deps_fns';
import { createI18nContexts } from './phases/create_i18n_contexts';
import { deduplicateTextBindings } from './phases/deduplicate_text_bindings';
import { configureDeferInstructions } from './phases/defer_configs';
import { resolveDeferTargetNames } from './phases/defer_resolve_targets';
import { collapseEmptyInstructions } from './phases/empty_elements';
import { expandSafeReads } from './phases/expand_safe_reads';
import { extractI18nMessages } from './phases/extract_i18n_messages';
import { generateAdvance } from './phases/generate_advance';
import { generateProjectionDefs } from './phases/generate_projection_def';
import { generateVariables } from './phases/generate_variables';
import { collectConstExpressions } from './phases/has_const_expression_collection';
import { parseHostStyleProperties } from './phases/host_style_property_parsing';
import { collectI18nConsts } from './phases/i18n_const_collection';
import { convertI18nText } from './phases/i18n_text_extraction';
import { liftLocalRefs } from './phases/local_refs';
import { emitNamespaceChanges } from './phases/namespace';
import { nameFunctionsAndVariables } from './phases/naming';
import { mergeNextContextExpressions } from './phases/next_context_merging';
import { generateNgContainerOps } from './phases/ng_container';
import { disableBindings } from './phases/nonbindable';
import { generateNullishCoalesceExpressions } from './phases/nullish_coalescing';
import { orderOps } from './phases/ordering';
import { parseExtractedStyles } from './phases/parse_extracted_styles';
import { removeContentSelectors } from './phases/phase_remove_content_selectors';
import { createPipes } from './phases/pipe_creation';
import { createVariadicPipes } from './phases/pipe_variadic';
import { propagateI18nBlocks } from './phases/propagate_i18n_blocks';
import { extractPureFunctions } from './phases/pure_function_extraction';
import { generatePureLiteralStructures } from './phases/pure_literal_structures';
import { reify } from './phases/reify';
import { removeEmptyBindings } from './phases/remove_empty_bindings';
import { removeI18nContexts } from './phases/remove_i18n_contexts';
import { removeUnusedI18nAttributesOps } from './phases/remove_unused_i18n_attrs';
import { resolveContexts } from './phases/resolve_contexts';
import { resolveDollarEvent } from './phases/resolve_dollar_event';
import { resolveI18nElementPlaceholders } from './phases/resolve_i18n_element_placeholders';
import { resolveI18nExpressionPlaceholders } from './phases/resolve_i18n_expression_placeholders';
import { resolveNames } from './phases/resolve_names';
import { resolveSanitizers } from './phases/resolve_sanitizers';
import { transformTwoWayBindingSet } from './phases/transform_two_way_binding_set';
import { saveAndRestoreView } from './phases/save_restore_view';
import { allocateSlots } from './phases/slot_allocation';
import { specializeStyleBindings } from './phases/style_binding_specialization';
import { generateTemporaryVariables } from './phases/temporary_variables';
import { generateTrackFns } from './phases/track_fn_generation';
import { optimizeTrackFns } from './phases/track_fn_optimization';
import { generateTrackVariables } from './phases/track_variables';
import { countVariables } from './phases/var_counting';
import { optimizeVariables } from './phases/variable_optimization';
import { wrapI18nIcus } from './phases/wrap_icus';
import { optimizeStoreLet } from './phases/store_let_optimization';
import { removeIllegalLetReferences } from './phases/remove_illegal_let_references';
import { generateLocalLetReferences } from './phases/generate_local_let_references';
const phases = [
    { kind: Kind.Tmpl, fn: removeContentSelectors },
    { kind: Kind.Host, fn: parseHostStyleProperties },
    { kind: Kind.Tmpl, fn: emitNamespaceChanges },
    { kind: Kind.Tmpl, fn: propagateI18nBlocks },
    { kind: Kind.Tmpl, fn: wrapI18nIcus },
    { kind: Kind.Both, fn: deduplicateTextBindings },
    { kind: Kind.Both, fn: specializeStyleBindings },
    { kind: Kind.Both, fn: specializeBindings },
    { kind: Kind.Both, fn: extractAttributes },
    { kind: Kind.Tmpl, fn: createI18nContexts },
    { kind: Kind.Both, fn: parseExtractedStyles },
    { kind: Kind.Tmpl, fn: removeEmptyBindings },
    { kind: Kind.Both, fn: collapseSingletonInterpolations },
    { kind: Kind.Both, fn: orderOps },
    { kind: Kind.Tmpl, fn: generateConditionalExpressions },
    { kind: Kind.Tmpl, fn: createPipes },
    { kind: Kind.Tmpl, fn: configureDeferInstructions },
    { kind: Kind.Tmpl, fn: convertI18nText },
    { kind: Kind.Tmpl, fn: convertI18nBindings },
    { kind: Kind.Tmpl, fn: removeUnusedI18nAttributesOps },
    { kind: Kind.Tmpl, fn: assignI18nSlotDependencies },
    { kind: Kind.Tmpl, fn: applyI18nExpressions },
    { kind: Kind.Tmpl, fn: createVariadicPipes },
    { kind: Kind.Both, fn: generatePureLiteralStructures },
    { kind: Kind.Tmpl, fn: generateProjectionDefs },
    { kind: Kind.Tmpl, fn: generateLocalLetReferences },
    { kind: Kind.Tmpl, fn: generateVariables },
    { kind: Kind.Tmpl, fn: saveAndRestoreView },
    { kind: Kind.Both, fn: deleteAnyCasts },
    { kind: Kind.Both, fn: resolveDollarEvent },
    { kind: Kind.Tmpl, fn: generateTrackVariables },
    { kind: Kind.Tmpl, fn: removeIllegalLetReferences },
    { kind: Kind.Both, fn: resolveNames },
    { kind: Kind.Tmpl, fn: resolveDeferTargetNames },
    { kind: Kind.Tmpl, fn: transformTwoWayBindingSet },
    { kind: Kind.Tmpl, fn: optimizeTrackFns },
    { kind: Kind.Both, fn: resolveContexts },
    { kind: Kind.Both, fn: resolveSanitizers },
    { kind: Kind.Tmpl, fn: liftLocalRefs },
    { kind: Kind.Both, fn: generateNullishCoalesceExpressions },
    { kind: Kind.Both, fn: expandSafeReads },
    { kind: Kind.Both, fn: generateTemporaryVariables },
    { kind: Kind.Both, fn: optimizeVariables },
    { kind: Kind.Both, fn: optimizeStoreLet },
    { kind: Kind.Tmpl, fn: allocateSlots },
    { kind: Kind.Tmpl, fn: resolveI18nElementPlaceholders },
    { kind: Kind.Tmpl, fn: resolveI18nExpressionPlaceholders },
    { kind: Kind.Tmpl, fn: extractI18nMessages },
    { kind: Kind.Tmpl, fn: generateTrackFns },
    { kind: Kind.Tmpl, fn: collectI18nConsts },
    { kind: Kind.Tmpl, fn: collectConstExpressions },
    { kind: Kind.Both, fn: collectElementConsts },
    { kind: Kind.Tmpl, fn: removeI18nContexts },
    { kind: Kind.Both, fn: countVariables },
    { kind: Kind.Tmpl, fn: generateAdvance },
    { kind: Kind.Both, fn: nameFunctionsAndVariables },
    { kind: Kind.Tmpl, fn: resolveDeferDepsFns },
    { kind: Kind.Tmpl, fn: mergeNextContextExpressions },
    { kind: Kind.Tmpl, fn: generateNgContainerOps },
    { kind: Kind.Tmpl, fn: collapseEmptyInstructions },
    { kind: Kind.Tmpl, fn: disableBindings },
    { kind: Kind.Both, fn: extractPureFunctions },
    { kind: Kind.Both, fn: reify },
    { kind: Kind.Both, fn: chain },
];
/**
 * Run all transformation phases in the correct order against a compilation job. After this
 * processing, the compilation should be in a state where it can be emitted.
 */
export function transform(job, kind) {
    for (const phase of phases) {
        if (phase.kind === kind || phase.kind === Kind.Both) {
            // The type of `Phase` above ensures it is impossible to call a phase that doesn't support the
            // job kind.
            phase.fn(job);
        }
    }
}
/**
 * Compile all views in the given `ComponentCompilation` into the final template function, which may
 * reference constants defined in a `ConstantPool`.
 */
export function emitTemplateFn(tpl, pool) {
    const rootFn = emitView(tpl.root);
    emitChildViews(tpl.root, pool);
    return rootFn;
}
function emitChildViews(parent, pool) {
    for (const unit of parent.job.units) {
        if (unit.parent !== parent.xref) {
            continue;
        }
        // Child views are emitted depth-first.
        emitChildViews(unit, pool);
        const viewFn = emitView(unit);
        pool.statements.push(viewFn.toDeclStmt(viewFn.name));
    }
}
/**
 * Emit a template function for an individual `ViewCompilation` (which may be either the root view
 * or an embedded view).
 */
function emitView(view) {
    if (view.fnName === null) {
        throw new Error(`AssertionError: view ${view.xref} is unnamed`);
    }
    const createStatements = [];
    for (const op of view.create) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all create ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        createStatements.push(op.statement);
    }
    const updateStatements = [];
    for (const op of view.update) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all update ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        updateStatements.push(op.statement);
    }
    const createCond = maybeGenerateRfBlock(1, createStatements);
    const updateCond = maybeGenerateRfBlock(2, updateStatements);
    return o.fn([new o.FnParam('rf'), new o.FnParam('ctx')], [...createCond, ...updateCond], 
    /* type */ undefined, 
    /* sourceSpan */ undefined, view.fnName);
}
function maybeGenerateRfBlock(flag, statements) {
    if (statements.length === 0) {
        return [];
    }
    return [
        o.ifStmt(new o.BinaryOperatorExpr(o.BinaryOperator.BitwiseAnd, o.variable('rf'), o.literal(flag)), statements),
    ];
}
export function emitHostBindingFunction(job) {
    if (job.root.fnName === null) {
        throw new Error(`AssertionError: host binding function is unnamed`);
    }
    const createStatements = [];
    for (const op of job.root.create) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all create ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        createStatements.push(op.statement);
    }
    const updateStatements = [];
    for (const op of job.root.update) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all update ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        updateStatements.push(op.statement);
    }
    if (createStatements.length === 0 && updateStatements.length === 0) {
        return null;
    }
    const createCond = maybeGenerateRfBlock(1, createStatements);
    const updateCond = maybeGenerateRfBlock(2, updateStatements);
    return o.fn([new o.FnParam('rf'), new o.FnParam('ctx')], [...createCond, ...updateCond], 
    /* type */ undefined, 
    /* sourceSpan */ undefined, job.root.fnName);
}
//# sourceMappingURL=data:application/json;base64,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