/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { splitNsName } from '../../../../ml_parser/tags';
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
import { CompilationJobKind } from '../compilation';
/**
 * Looks up an element in the given map by xref ID.
 */
function lookupElement(elements, xref) {
    const el = elements.get(xref);
    if (el === undefined) {
        throw new Error('All attributes should have an element-like target.');
    }
    return el;
}
export function specializeBindings(job) {
    const elements = new Map();
    for (const unit of job.units) {
        for (const op of unit.create) {
            if (!ir.isElementOrContainerOp(op)) {
                continue;
            }
            elements.set(op.xref, op);
        }
    }
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            if (op.kind !== ir.OpKind.Binding) {
                continue;
            }
            switch (op.bindingKind) {
                case ir.BindingKind.Attribute:
                    if (op.name === 'ngNonBindable') {
                        ir.OpList.remove(op);
                        const target = lookupElement(elements, op.target);
                        target.nonBindable = true;
                    }
                    else {
                        const [namespace, name] = splitNsName(op.name);
                        ir.OpList.replace(op, ir.createAttributeOp(op.target, namespace, name, op.expression, op.securityContext, op.isTextAttribute, op.isStructuralTemplateAttribute, op.templateKind, op.i18nMessage, op.sourceSpan));
                    }
                    break;
                case ir.BindingKind.Property:
                case ir.BindingKind.Animation:
                    if (job.kind === CompilationJobKind.Host) {
                        ir.OpList.replace(op, ir.createHostPropertyOp(op.name, op.expression, op.bindingKind === ir.BindingKind.Animation, op.i18nContext, op.securityContext, op.sourceSpan));
                    }
                    else {
                        ir.OpList.replace(op, ir.createPropertyOp(op.target, op.name, op.expression, op.bindingKind === ir.BindingKind.Animation, op.securityContext, op.isStructuralTemplateAttribute, op.templateKind, op.i18nContext, op.i18nMessage, op.sourceSpan));
                    }
                    break;
                case ir.BindingKind.TwoWayProperty:
                    if (!(op.expression instanceof o.Expression)) {
                        // We shouldn't be able to hit this code path since interpolations in two-way bindings
                        // result in a parser error. We assert here so that downstream we can assume that
                        // the value is always an expression.
                        throw new Error(`Expected value of two-way property binding "${op.name}" to be an expression`);
                    }
                    ir.OpList.replace(op, ir.createTwoWayPropertyOp(op.target, op.name, op.expression, op.securityContext, op.isStructuralTemplateAttribute, op.templateKind, op.i18nContext, op.i18nMessage, op.sourceSpan));
                    break;
                case ir.BindingKind.I18n:
                case ir.BindingKind.ClassName:
                case ir.BindingKind.StyleProperty:
                    throw new Error(`Unhandled binding of kind ${ir.BindingKind[op.bindingKind]}`);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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