/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Safe read expressions such as `a?.b` have different semantics in Angular templates as
 * compared to JavaScript. In particular, they default to `null` instead of `undefined`. This phase
 * finds all unresolved safe read expressions, and converts them into the appropriate output AST
 * reads, guarded by null checks. We generate temporaries as needed, to avoid re-evaluating the same
 * sub-expression multiple times.
 */
export function expandSafeReads(job) {
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            ir.transformExpressionsInOp(op, (e) => safeTransform(e, { job }), ir.VisitorContextFlag.None);
            ir.transformExpressionsInOp(op, ternaryTransform, ir.VisitorContextFlag.None);
        }
    }
}
// A lookup set of all the expression kinds that require a temporary variable to be generated.
const requiresTemporary = [
    o.InvokeFunctionExpr,
    o.LiteralArrayExpr,
    o.LiteralMapExpr,
    ir.SafeInvokeFunctionExpr,
    ir.PipeBindingExpr,
].map((e) => e.constructor.name);
function needsTemporaryInSafeAccess(e) {
    // TODO: We probably want to use an expression visitor to recursively visit all descendents.
    // However, that would potentially do a lot of extra work (because it cannot short circuit), so we
    // implement the logic ourselves for now.
    if (e instanceof o.UnaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.BinaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.lhs) || needsTemporaryInSafeAccess(e.rhs);
    }
    else if (e instanceof o.ConditionalExpr) {
        if (e.falseCase && needsTemporaryInSafeAccess(e.falseCase))
            return true;
        return needsTemporaryInSafeAccess(e.condition) || needsTemporaryInSafeAccess(e.trueCase);
    }
    else if (e instanceof o.NotExpr) {
        return needsTemporaryInSafeAccess(e.condition);
    }
    else if (e instanceof ir.AssignTemporaryExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.ReadPropExpr) {
        return needsTemporaryInSafeAccess(e.receiver);
    }
    else if (e instanceof o.ReadKeyExpr) {
        return needsTemporaryInSafeAccess(e.receiver) || needsTemporaryInSafeAccess(e.index);
    }
    // TODO: Switch to a method which is exhaustive of newly added expression subtypes.
    return (e instanceof o.InvokeFunctionExpr ||
        e instanceof o.LiteralArrayExpr ||
        e instanceof o.LiteralMapExpr ||
        e instanceof ir.SafeInvokeFunctionExpr ||
        e instanceof ir.PipeBindingExpr);
}
function temporariesIn(e) {
    const temporaries = new Set();
    // TODO: Although it's not currently supported by the transform helper, we should be able to
    // short-circuit exploring the tree to do less work. In particular, we don't have to penetrate
    // into the subexpressions of temporary assignments.
    ir.transformExpressionsInExpression(e, (e) => {
        if (e instanceof ir.AssignTemporaryExpr) {
            temporaries.add(e.xref);
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return temporaries;
}
function eliminateTemporaryAssignments(e, tmps, ctx) {
    // TODO: We can be more efficient than the transform helper here. We don't need to visit any
    // descendents of temporary assignments.
    ir.transformExpressionsInExpression(e, (e) => {
        if (e instanceof ir.AssignTemporaryExpr && tmps.has(e.xref)) {
            const read = new ir.ReadTemporaryExpr(e.xref);
            // `TemplateDefinitionBuilder` has the (accidental?) behavior of generating assignments of
            // temporary variables to themselves. This happens because some subexpression that the
            // temporary refers to, possibly through nested temporaries, has a function call. We copy that
            // behavior here.
            return ctx.job.compatibility === ir.CompatibilityMode.TemplateDefinitionBuilder
                ? new ir.AssignTemporaryExpr(read, read.xref)
                : read;
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return e;
}
/**
 * Creates a safe ternary guarded by the input expression, and with a body generated by the provided
 * callback on the input expression. Generates a temporary variable assignment if needed, and
 * deduplicates nested temporary assignments if needed.
 */
function safeTernaryWithTemporary(guard, body, ctx) {
    let result;
    if (needsTemporaryInSafeAccess(guard)) {
        const xref = ctx.job.allocateXrefId();
        result = [new ir.AssignTemporaryExpr(guard, xref), new ir.ReadTemporaryExpr(xref)];
    }
    else {
        result = [guard, guard.clone()];
        // Consider an expression like `a?.[b?.c()]?.d`. The `b?.c()` will be transformed first,
        // introducing a temporary assignment into the key. Then, as part of expanding the `?.d`. That
        // assignment will be duplicated into both the guard and expression sides. We de-duplicate it,
        // by transforming it from an assignment into a read on the expression side.
        eliminateTemporaryAssignments(result[1], temporariesIn(result[0]), ctx);
    }
    return new ir.SafeTernaryExpr(result[0], body(result[1]));
}
function isSafeAccessExpression(e) {
    return (e instanceof ir.SafePropertyReadExpr ||
        e instanceof ir.SafeKeyedReadExpr ||
        e instanceof ir.SafeInvokeFunctionExpr);
}
function isUnsafeAccessExpression(e) {
    return (e instanceof o.ReadPropExpr || e instanceof o.ReadKeyExpr || e instanceof o.InvokeFunctionExpr);
}
function isAccessExpression(e) {
    return isSafeAccessExpression(e) || isUnsafeAccessExpression(e);
}
function deepestSafeTernary(e) {
    if (isAccessExpression(e) && e.receiver instanceof ir.SafeTernaryExpr) {
        let st = e.receiver;
        while (st.expr instanceof ir.SafeTernaryExpr) {
            st = st.expr;
        }
        return st;
    }
    return null;
}
// TODO: When strict compatibility with TemplateDefinitionBuilder is not required, we can use `&&`
// instead to save some code size.
function safeTransform(e, ctx) {
    if (!isAccessExpression(e)) {
        return e;
    }
    const dst = deepestSafeTernary(e);
    if (dst) {
        if (e instanceof o.InvokeFunctionExpr) {
            dst.expr = dst.expr.callFn(e.args);
            return e.receiver;
        }
        if (e instanceof o.ReadPropExpr) {
            dst.expr = dst.expr.prop(e.name);
            return e.receiver;
        }
        if (e instanceof o.ReadKeyExpr) {
            dst.expr = dst.expr.key(e.index);
            return e.receiver;
        }
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.callFn(e.args), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.prop(e.name), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.key(e.index), ctx);
            return e.receiver;
        }
    }
    else {
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.callFn(e.args), ctx);
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.prop(e.name), ctx);
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.key(e.index), ctx);
        }
    }
    return e;
}
function ternaryTransform(e) {
    if (!(e instanceof ir.SafeTernaryExpr)) {
        return e;
    }
    return new o.ConditionalExpr(new o.BinaryOperatorExpr(o.BinaryOperator.Equals, e.guard, o.NULL_EXPR), o.NULL_EXPR, e.expr);
}
//# sourceMappingURL=data:application/json;base64,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