/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import * as ir from '../../ir';
const STYLE_DOT = 'style.';
const CLASS_DOT = 'class.';
const STYLE_BANG = 'style!';
const CLASS_BANG = 'class!';
const BANG_IMPORTANT = '!important';
/**
 * Host bindings are compiled using a different parser entrypoint, and are parsed quite differently
 * as a result. Therefore, we need to do some extra parsing for host style properties, as compared
 * to non-host style properties.
 * TODO: Unify host bindings and non-host bindings in the parser.
 */
export function parseHostStyleProperties(job) {
    for (const op of job.root.update) {
        if (!(op.kind === ir.OpKind.Binding && op.bindingKind === ir.BindingKind.Property)) {
            continue;
        }
        if (op.name.endsWith(BANG_IMPORTANT)) {
            // Delete any `!important` suffixes from the binding name.
            op.name = op.name.substring(0, op.name.length - BANG_IMPORTANT.length);
        }
        if (op.name.startsWith(STYLE_DOT)) {
            op.bindingKind = ir.BindingKind.StyleProperty;
            op.name = op.name.substring(STYLE_DOT.length);
            if (!isCssCustomProperty(op.name)) {
                op.name = hyphenate(op.name);
            }
            const { property, suffix } = parseProperty(op.name);
            op.name = property;
            op.unit = suffix;
        }
        else if (op.name.startsWith(STYLE_BANG)) {
            op.bindingKind = ir.BindingKind.StyleProperty;
            op.name = 'style';
        }
        else if (op.name.startsWith(CLASS_DOT)) {
            op.bindingKind = ir.BindingKind.ClassName;
            op.name = parseProperty(op.name.substring(CLASS_DOT.length)).property;
        }
        else if (op.name.startsWith(CLASS_BANG)) {
            op.bindingKind = ir.BindingKind.ClassName;
            op.name = parseProperty(op.name.substring(CLASS_BANG.length)).property;
        }
    }
}
/**
 * Checks whether property name is a custom CSS property.
 * See: https://www.w3.org/TR/css-variables-1
 */
function isCssCustomProperty(name) {
    return name.startsWith('--');
}
function hyphenate(value) {
    return value
        .replace(/[a-z][A-Z]/g, (v) => {
        return v.charAt(0) + '-' + v.charAt(1);
    })
        .toLowerCase();
}
function parseProperty(name) {
    const overrideIndex = name.indexOf('!important');
    if (overrideIndex !== -1) {
        name = overrideIndex > 0 ? name.substring(0, overrideIndex) : '';
    }
    let suffix = null;
    let property = name;
    const unitIndex = name.lastIndexOf('.');
    if (unitIndex > 0) {
        suffix = name.slice(unitIndex + 1);
        property = name.substring(0, unitIndex);
    }
    return { property, suffix };
}
//# sourceMappingURL=data:application/json;base64,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