/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Optimize variables declared and used in the IR.
 *
 * Variables are eagerly generated by pipeline stages for all possible values that could be
 * referenced. This stage processes the list of declared variables and all variable usages,
 * and optimizes where possible. It performs 3 main optimizations:
 *
 *   * It transforms variable declarations to side effectful expressions when the
 *     variable is not used, but its initializer has global effects which other
 *     operations rely upon.
 *   * It removes variable declarations if those variables are not referenced and
 *     either they do not have global effects, or nothing relies on them.
 *   * It inlines variable declarations when those variables are only used once
 *     and the inlining is semantically safe.
 *
 * To guarantee correctness, analysis of "fences" in the instruction lists is used to determine
 * which optimizations are safe to perform.
 */
export function optimizeVariables(job) {
    for (const unit of job.units) {
        inlineAlwaysInlineVariables(unit.create);
        inlineAlwaysInlineVariables(unit.update);
        for (const op of unit.create) {
            if (op.kind === ir.OpKind.Listener || op.kind === ir.OpKind.TwoWayListener) {
                inlineAlwaysInlineVariables(op.handlerOps);
            }
        }
        optimizeVariablesInOpList(unit.create, job.compatibility);
        optimizeVariablesInOpList(unit.update, job.compatibility);
        for (const op of unit.create) {
            if (op.kind === ir.OpKind.Listener || op.kind === ir.OpKind.TwoWayListener) {
                optimizeVariablesInOpList(op.handlerOps, job.compatibility);
            }
        }
    }
}
/**
 * A [fence](https://en.wikipedia.org/wiki/Memory_barrier) flag for an expression which indicates
 * how that expression can be optimized in relation to other expressions or instructions.
 *
 * `Fence`s are a bitfield, so multiple flags may be set on a single expression.
 */
var Fence;
(function (Fence) {
    /**
     * Empty flag (no fence exists).
     */
    Fence[Fence["None"] = 0] = "None";
    /**
     * A context read fence, meaning that the expression in question reads from the "current view"
     * context of the runtime.
     */
    Fence[Fence["ViewContextRead"] = 1] = "ViewContextRead";
    /**
     * A context write fence, meaning that the expression in question writes to the "current view"
     * context of the runtime.
     *
     * Note that all `ContextWrite` fences are implicitly `ContextRead` fences as operations which
     * change the view context do so based on the current one.
     */
    Fence[Fence["ViewContextWrite"] = 2] = "ViewContextWrite";
    /**
     * Indicates that a call is required for its side-effects, even if nothing reads its result.
     *
     * This is also true of `ViewContextWrite` operations **if** they are followed by a
     * `ViewContextRead`.
     */
    Fence[Fence["SideEffectful"] = 4] = "SideEffectful";
})(Fence || (Fence = {}));
function inlineAlwaysInlineVariables(ops) {
    const vars = new Map();
    for (const op of ops) {
        if (op.kind === ir.OpKind.Variable && op.flags & ir.VariableFlags.AlwaysInline) {
            ir.visitExpressionsInOp(op, (expr) => {
                if (ir.isIrExpression(expr) && fencesForIrExpression(expr) !== Fence.None) {
                    throw new Error(`AssertionError: A context-sensitive variable was marked AlwaysInline`);
                }
            });
            vars.set(op.xref, op);
        }
        ir.transformExpressionsInOp(op, (expr) => {
            if (expr instanceof ir.ReadVariableExpr && vars.has(expr.xref)) {
                const varOp = vars.get(expr.xref);
                // Inline by cloning, because we might inline into multiple places.
                return varOp.initializer.clone();
            }
            return expr;
        }, ir.VisitorContextFlag.None);
    }
    for (const op of vars.values()) {
        ir.OpList.remove(op);
    }
}
/**
 * Process a list of operations and optimize variables within that list.
 */
function optimizeVariablesInOpList(ops, compatibility) {
    const varDecls = new Map();
    const varUsages = new Map();
    // Track variables that are used outside of the immediate operation list. For example, within
    // `ListenerOp` handler operations of listeners in the current operation list.
    const varRemoteUsages = new Set();
    const opMap = new Map();
    // First, extract information about variables declared or used within the whole list.
    for (const op of ops) {
        if (op.kind === ir.OpKind.Variable) {
            if (varDecls.has(op.xref) || varUsages.has(op.xref)) {
                throw new Error(`Should not see two declarations of the same variable: ${op.xref}`);
            }
            varDecls.set(op.xref, op);
            varUsages.set(op.xref, 0);
        }
        opMap.set(op, collectOpInfo(op));
        countVariableUsages(op, varUsages, varRemoteUsages);
    }
    // The next step is to remove any variable declarations for variables that aren't used. The
    // variable initializer expressions may be side-effectful, so they may need to be retained as
    // expression statements.
    // Track whether we've seen an operation which reads from the view context yet. This is used to
    // determine whether a write to the view context in a variable initializer can be observed.
    let contextIsUsed = false;
    // Note that iteration through the list happens in reverse, which guarantees that we'll process
    // all reads of a variable prior to processing its declaration.
    for (const op of ops.reversed()) {
        const opInfo = opMap.get(op);
        if (op.kind === ir.OpKind.Variable && varUsages.get(op.xref) === 0) {
            // This variable is unused and can be removed. We might need to keep the initializer around,
            // though, if something depends on it running.
            if ((contextIsUsed && opInfo.fences & Fence.ViewContextWrite) ||
                opInfo.fences & Fence.SideEffectful) {
                // This variable initializer has a side effect which must be retained. Either:
                //  * it writes to the view context, and we know there is a future operation which depends
                //    on that write, or
                //  * it's an operation which is inherently side-effectful.
                // We can't remove the initializer, but we can remove the variable declaration itself and
                // replace it with a side-effectful statement.
                const stmtOp = ir.createStatementOp(op.initializer.toStmt());
                opMap.set(stmtOp, opInfo);
                ir.OpList.replace(op, stmtOp);
            }
            else {
                // It's safe to delete this entire variable declaration as nothing depends on it, even
                // side-effectfully. Note that doing this might make other variables unused. Since we're
                // iterating in reverse order, we should always be processing usages before declarations
                // and therefore by the time we get to a declaration, all removable usages will have been
                // removed.
                uncountVariableUsages(op, varUsages);
                ir.OpList.remove(op);
            }
            opMap.delete(op);
            varDecls.delete(op.xref);
            varUsages.delete(op.xref);
            continue;
        }
        // Does this operation depend on the view context?
        if (opInfo.fences & Fence.ViewContextRead) {
            contextIsUsed = true;
        }
    }
    // Next, inline any remaining variables with exactly one usage.
    const toInline = [];
    for (const [id, count] of varUsages) {
        const decl = varDecls.get(id);
        // We can inline variables that:
        //  - are used exactly once, and
        //  - are not used remotely
        // OR
        //  - are marked for always inlining
        const isAlwaysInline = !!(decl.flags & ir.VariableFlags.AlwaysInline);
        if (count !== 1 || isAlwaysInline) {
            // We can't inline this variable as it's used more than once.
            continue;
        }
        if (varRemoteUsages.has(id)) {
            // This variable is used once, but across an operation boundary, so it can't be inlined.
            continue;
        }
        toInline.push(id);
    }
    let candidate;
    while ((candidate = toInline.pop())) {
        // We will attempt to inline this variable. If inlining fails (due to fences for example),
        // no future operation will make inlining legal.
        const decl = varDecls.get(candidate);
        const varInfo = opMap.get(decl);
        const isAlwaysInline = !!(decl.flags & ir.VariableFlags.AlwaysInline);
        if (isAlwaysInline) {
            throw new Error(`AssertionError: Found an 'AlwaysInline' variable after the always inlining pass.`);
        }
        // Scan operations following the variable declaration and look for the point where that variable
        // is used. There should only be one usage given the precondition above.
        for (let targetOp = decl.next; targetOp.kind !== ir.OpKind.ListEnd; targetOp = targetOp.next) {
            const opInfo = opMap.get(targetOp);
            // Is the variable used in this operation?
            if (opInfo.variablesUsed.has(candidate)) {
                if (compatibility === ir.CompatibilityMode.TemplateDefinitionBuilder &&
                    !allowConservativeInlining(decl, targetOp)) {
                    // We're in conservative mode, and this variable is not eligible for inlining into the
                    // target operation in this mode.
                    break;
                }
                // Yes, try to inline it. Inlining may not be successful if fences in this operation before
                // the variable's usage cannot be safely crossed.
                if (tryInlineVariableInitializer(candidate, decl.initializer, targetOp, varInfo.fences)) {
                    // Inlining was successful! Update the tracking structures to reflect the inlined
                    // variable.
                    opInfo.variablesUsed.delete(candidate);
                    // Add all variables used in the variable's initializer to its new usage site.
                    for (const id of varInfo.variablesUsed) {
                        opInfo.variablesUsed.add(id);
                    }
                    // Merge fences in the variable's initializer into its new usage site.
                    opInfo.fences |= varInfo.fences;
                    // Delete tracking info related to the declaration.
                    varDecls.delete(candidate);
                    varUsages.delete(candidate);
                    opMap.delete(decl);
                    // And finally, delete the original declaration from the operation list.
                    ir.OpList.remove(decl);
                }
                // Whether inlining succeeded or failed, we're done processing this variable.
                break;
            }
            // If the variable is not used in this operation, then we'd need to inline across it. Check if
            // that's safe to do.
            if (!safeToInlinePastFences(opInfo.fences, varInfo.fences)) {
                // We can't safely inline this variable beyond this operation, so don't proceed with
                // inlining this variable.
                break;
            }
        }
    }
}
/**
 * Given an `ir.Expression`, returns the `Fence` flags for that expression type.
 */
function fencesForIrExpression(expr) {
    switch (expr.kind) {
        case ir.ExpressionKind.NextContext:
            return Fence.ViewContextRead | Fence.ViewContextWrite;
        case ir.ExpressionKind.RestoreView:
            return Fence.ViewContextRead | Fence.ViewContextWrite | Fence.SideEffectful;
        case ir.ExpressionKind.StoreLet:
            return Fence.SideEffectful;
        case ir.ExpressionKind.Reference:
        case ir.ExpressionKind.ContextLetReference:
            return Fence.ViewContextRead;
        default:
            return Fence.None;
    }
}
/**
 * Build the `OpInfo` structure for the given `op`. This performs two operations:
 *
 *  * It tracks which variables are used in the operation's expressions.
 *  * It rolls up fence flags for expressions within the operation.
 */
function collectOpInfo(op) {
    let fences = Fence.None;
    const variablesUsed = new Set();
    ir.visitExpressionsInOp(op, (expr) => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                variablesUsed.add(expr.xref);
                break;
            default:
                fences |= fencesForIrExpression(expr);
        }
    });
    return { fences, variablesUsed };
}
/**
 * Count the number of usages of each variable, being careful to track whether those usages are
 * local or remote.
 */
function countVariableUsages(op, varUsages, varRemoteUsage) {
    ir.visitExpressionsInOp(op, (expr, flags) => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        varUsages.set(expr.xref, count + 1);
        if (flags & ir.VisitorContextFlag.InChildOperation) {
            varRemoteUsage.add(expr.xref);
        }
    });
}
/**
 * Remove usages of a variable in `op` from the `varUsages` tracking.
 */
function uncountVariableUsages(op, varUsages) {
    ir.visitExpressionsInOp(op, (expr) => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        else if (count === 0) {
            throw new Error(`Inaccurate variable count: ${expr.xref} - found another read but count is already 0`);
        }
        varUsages.set(expr.xref, count - 1);
    });
}
/**
 * Checks whether it's safe to inline a variable across a particular operation.
 *
 * @param fences the fences of the operation which the inlining will cross
 * @param declFences the fences of the variable being inlined.
 */
function safeToInlinePastFences(fences, declFences) {
    if (fences & Fence.ViewContextWrite) {
        // It's not safe to inline context reads across context writes.
        if (declFences & Fence.ViewContextRead) {
            return false;
        }
    }
    else if (fences & Fence.ViewContextRead) {
        // It's not safe to inline context writes across context reads.
        if (declFences & Fence.ViewContextWrite) {
            return false;
        }
    }
    return true;
}
/**
 * Attempt to inline the initializer of a variable into a target operation's expressions.
 *
 * This may or may not be safe to do. For example, the variable could be read following the
 * execution of an expression with fences that don't permit the variable to be inlined across them.
 */
function tryInlineVariableInitializer(id, initializer, target, declFences) {
    // We use `ir.transformExpressionsInOp` to walk the expressions and inline the variable if
    // possible. Since this operation is callback-based, once inlining succeeds or fails we can't
    // "stop" the expression processing, and have to keep track of whether inlining has succeeded or
    // is no longer allowed.
    let inlined = false;
    let inliningAllowed = true;
    ir.transformExpressionsInOp(target, (expr, flags) => {
        if (!ir.isIrExpression(expr)) {
            return expr;
        }
        if (inlined || !inliningAllowed) {
            // Either the inlining has already succeeded, or we've passed a fence that disallows inlining
            // at this point, so don't try.
            return expr;
        }
        else if (flags & ir.VisitorContextFlag.InChildOperation &&
            declFences & Fence.ViewContextRead) {
            // We cannot inline variables that are sensitive to the current context across operation
            // boundaries.
            return expr;
        }
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                if (expr.xref === id) {
                    // This is the usage site of the variable. Since nothing has disallowed inlining, it's
                    // safe to inline the initializer here.
                    inlined = true;
                    return initializer;
                }
                break;
            default:
                // For other types of `ir.Expression`s, whether inlining is allowed depends on their fences.
                const exprFences = fencesForIrExpression(expr);
                inliningAllowed = inliningAllowed && safeToInlinePastFences(exprFences, declFences);
                break;
        }
        return expr;
    }, ir.VisitorContextFlag.None);
    return inlined;
}
/**
 * Determines whether inlining of `decl` should be allowed in "conservative" mode.
 *
 * In conservative mode, inlining behavior is limited to those operations which the
 * `TemplateDefinitionBuilder` supported, with the goal of producing equivalent output.
 */
function allowConservativeInlining(decl, target) {
    // TODO(alxhub): understand exactly how TemplateDefinitionBuilder approaches inlining, and record
    // that behavior here.
    switch (decl.variable.kind) {
        case ir.SemanticVariableKind.Identifier:
            if (decl.initializer instanceof o.ReadVarExpr && decl.initializer.name === 'ctx') {
                // Although TemplateDefinitionBuilder is cautious about inlining, we still want to do so
                // when the variable is the context, to imitate its behavior with aliases in control flow
                // blocks. This quirky behavior will become dead code once compatibility mode is no longer
                // supported.
                return true;
            }
            return false;
        case ir.SemanticVariableKind.Context:
            // Context can only be inlined into other variables.
            return target.kind === ir.OpKind.Variable;
        default:
            return true;
    }
}
//# sourceMappingURL=data:application/json;base64,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