/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import * as eventLib from './event';
import * as eventInfoLib from './event_info';
import { EventType } from './event_type';
/**
 * Update `EventInfo` to be `eventType = 'click'` and sets `a11yClickKey` if it
 * is a a11y click.
 */
export function updateEventInfoForA11yClick(eventInfo) {
    if (!eventLib.isActionKeyEvent(eventInfoLib.getEvent(eventInfo))) {
        return;
    }
    eventInfoLib.setA11yClickKey(eventInfo, true);
    // A 'click' triggered by a DOM keypress should be mapped to the 'click'
    // jsaction.
    eventInfoLib.setEventType(eventInfo, EventType.CLICK);
}
/**
 * Call `preventDefault` on an a11y click if it is space key or to prevent the
 * browser's default action for native HTML controls.
 */
export function preventDefaultForA11yClick(eventInfo) {
    if (!eventInfoLib.getA11yClickKey(eventInfo) ||
        (!eventLib.isSpaceKeyEvent(eventInfoLib.getEvent(eventInfo)) &&
            !eventLib.shouldCallPreventDefaultOnNativeHtmlControl(eventInfoLib.getEvent(eventInfo)))) {
        return;
    }
    eventLib.preventDefault(eventInfoLib.getEvent(eventInfo));
}
/**
 * Sets the `action` to `clickonly` for a click event that is not an a11y click
 * and if there is not already a click action.
 */
export function populateClickOnlyAction(actionElement, eventInfo, actionMap) {
    if (
    // If there's already an action, don't attempt to set a CLICKONLY
    eventInfoLib.getAction(eventInfo) ||
        // Only attempt CLICKONLY if the type is CLICK
        eventInfoLib.getEventType(eventInfo) !== EventType.CLICK ||
        // a11y clicks are never CLICKONLY
        eventInfoLib.getA11yClickKey(eventInfo) ||
        actionMap[EventType.CLICKONLY] === undefined) {
        return;
    }
    eventInfoLib.setEventType(eventInfo, EventType.CLICKONLY);
    eventInfoLib.setAction(eventInfo, actionMap[EventType.CLICKONLY], actionElement);
}
//# sourceMappingURL=data:application/json;base64,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