/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { Property } from './property';
/**
 * Map from jsaction annotation to a parsed map from event name to action name.
 */
const parseCache = {};
/**
 * Reads the jsaction parser cache from the given DOM Element.
 */
export function get(element) {
    return element[Property.JSACTION];
}
/**
 * Reads the jsaction parser cache for the given DOM element. If no cache is yet present,
 * creates an empty one.
 */
export function getDefaulted(element) {
    const cache = get(element) ?? {};
    set(element, cache);
    return cache;
}
/**
 * Writes the jsaction parser cache to the given DOM Element.
 */
export function set(element, actionMap) {
    element[Property.JSACTION] = actionMap;
}
/**
 * Looks up the parsed action map from the source jsaction attribute value.
 *
 * @param text Unparsed jsaction attribute value.
 * @return Parsed jsaction attribute value, if already present in the cache.
 */
export function getParsed(text) {
    return parseCache[text];
}
/**
 * Inserts the parse result for the given source jsaction value into the cache.
 *
 * @param text Unparsed jsaction attribute value.
 * @param parsed Attribute value parsed into the action map.
 */
export function setParsed(text, parsed) {
    parseCache[text] = parsed;
}
/**
 * Clears the jsaction parser cache from the given DOM Element.
 *
 * @param element .
 */
export function clear(element) {
    if (Property.JSACTION in element) {
        delete element[Property.JSACTION];
    }
}
//# sourceMappingURL=data:application/json;base64,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