/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { createEventInfoFromParameters } from './event_info';
/**
 * EarlyEventContract intercepts events in the bubbling phase at the
 * boundary of the document body. This mapping will be passed to the
 * late-loaded EventContract.
 */
export class EarlyEventContract {
    constructor(dataContainer = window, container = window.document.documentElement) {
        this.dataContainer = dataContainer;
        dataContainer._ejsa = createEarlyJsactionData(container);
    }
    /**
     * Installs a list of event types for container .
     */
    addEvents(types, capture) {
        addEvents(this.dataContainer._ejsa, types, capture);
    }
}
/** Creates an `EarlyJsactionData` object. */
export function createEarlyJsactionData(container) {
    const q = [];
    const d = (eventInfo) => {
        q.push(eventInfo);
    };
    const h = (event) => {
        d(createEventInfoFromParameters(event.type, event, event.target, container, Date.now()));
    };
    return {
        c: container,
        q,
        et: [],
        etc: [],
        d,
        h,
    };
}
/** Add all the events to the container stored in the `EarlyJsactionData`. */
export function addEvents(earlyJsactionData, types, capture) {
    for (let i = 0; i < types.length; i++) {
        const eventType = types[i];
        const eventTypes = capture ? earlyJsactionData.etc : earlyJsactionData.et;
        eventTypes.push(eventType);
        earlyJsactionData.c.addEventListener(eventType, earlyJsactionData.h, capture);
    }
}
/** Get the queued `EventInfo` objects that were dispatched before a dispatcher was registered. */
export function getQueuedEventInfos(earlyJsactionData) {
    return earlyJsactionData?.q ?? [];
}
/** Register a different dispatcher function on the `EarlyJsactionData`. */
export function registerDispatcher(earlyJsactionData, dispatcher) {
    if (!earlyJsactionData) {
        return;
    }
    earlyJsactionData.d = dispatcher;
}
/** Removes all event listener handlers. */
export function removeAllEventListeners(earlyJsactionData) {
    if (!earlyJsactionData) {
        return;
    }
    removeEventListeners(earlyJsactionData.c, earlyJsactionData.et, earlyJsactionData.h);
    removeEventListeners(earlyJsactionData.c, earlyJsactionData.etc, earlyJsactionData.h, true);
}
function removeEventListeners(container, eventTypes, earlyEventHandler, capture) {
    for (let i = 0; i < eventTypes.length; i++) {
        container.removeEventListener(eventTypes[i], earlyEventHandler, /* useCapture */ capture);
    }
}
//# sourceMappingURL=data:application/json;base64,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