/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { ActionResolver } from './action_resolver';
import { Dispatcher } from './dispatcher';
import { isCaptureEventType } from './event_type';
import { Restriction } from './restriction';
/** An internal symbol used to indicate whether propagation should be stopped or not. */
export const PROPAGATION_STOPPED_SYMBOL = Symbol.for('propagationStopped');
/** Extra event phases beyond what the browser provides. */
export const EventPhase = {
    REPLAY: 101,
};
const PREVENT_DEFAULT_ERROR_MESSAGE_DETAILS = ' Because event replay occurs after browser dispatch, `preventDefault` would have no ' +
    'effect. You can check whether an event is being replayed by accessing the event phase: ' +
    '`event.eventPhase === EventPhase.REPLAY`.';
const PREVENT_DEFAULT_ERROR_MESSAGE = `\`preventDefault\` called during event replay.`;
const COMPOSED_PATH_ERROR_MESSAGE_DETAILS = ' Because event replay occurs after browser ' +
    'dispatch, `composedPath()` will be empty. Iterate parent nodes from `event.target` or ' +
    '`event.currentTarget` if you need to check elements in the event path.';
const COMPOSED_PATH_ERROR_MESSAGE = `\`composedPath\` called during event replay.`;
/**
 * A dispatcher that uses browser-based `Event` semantics, for example bubbling, `stopPropagation`,
 * `currentTarget`, etc.
 */
export class EventDispatcher {
    constructor(dispatchDelegate, clickModSupport = true) {
        this.dispatchDelegate = dispatchDelegate;
        this.clickModSupport = clickModSupport;
        this.actionResolver = new ActionResolver({ clickModSupport });
        this.dispatcher = new Dispatcher((eventInfoWrapper) => {
            this.dispatchToDelegate(eventInfoWrapper);
        }, {
            actionResolver: this.actionResolver,
        });
    }
    /**
     * The entrypoint for the `EventContract` dispatch.
     */
    dispatch(eventInfo) {
        this.dispatcher.dispatch(eventInfo);
    }
    /** Internal method that does basic disaptching. */
    dispatchToDelegate(eventInfoWrapper) {
        if (eventInfoWrapper.getIsReplay()) {
            prepareEventForReplay(eventInfoWrapper);
        }
        prepareEventForBubbling(eventInfoWrapper);
        while (eventInfoWrapper.getAction()) {
            prepareEventForDispatch(eventInfoWrapper);
            // If this is a capture event, ONLY dispatch if the action element is the target.
            if (isCaptureEventType(eventInfoWrapper.getEventType()) &&
                eventInfoWrapper.getAction().element !== eventInfoWrapper.getTargetElement()) {
                return;
            }
            this.dispatchDelegate(eventInfoWrapper.getEvent(), eventInfoWrapper.getAction().name);
            if (propagationStopped(eventInfoWrapper)) {
                return;
            }
            this.actionResolver.resolveParentAction(eventInfoWrapper.eventInfo);
        }
    }
}
function prepareEventForBubbling(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    const originalStopPropagation = eventInfoWrapper.getEvent().stopPropagation.bind(event);
    const stopPropagation = () => {
        event[PROPAGATION_STOPPED_SYMBOL] = true;
        originalStopPropagation();
    };
    patchEventInstance(event, 'stopPropagation', stopPropagation);
    patchEventInstance(event, 'stopImmediatePropagation', stopPropagation);
}
function propagationStopped(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    return !!event[PROPAGATION_STOPPED_SYMBOL];
}
function prepareEventForReplay(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    const target = eventInfoWrapper.getTargetElement();
    const originalPreventDefault = event.preventDefault.bind(event);
    patchEventInstance(event, 'target', target);
    patchEventInstance(event, 'eventPhase', EventPhase.REPLAY);
    patchEventInstance(event, 'preventDefault', () => {
        originalPreventDefault();
        throw new Error(PREVENT_DEFAULT_ERROR_MESSAGE + (ngDevMode ? PREVENT_DEFAULT_ERROR_MESSAGE_DETAILS : ''));
    });
    patchEventInstance(event, 'composedPath', () => {
        throw new Error(COMPOSED_PATH_ERROR_MESSAGE + (ngDevMode ? COMPOSED_PATH_ERROR_MESSAGE_DETAILS : ''));
    });
}
function prepareEventForDispatch(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    const currentTarget = eventInfoWrapper.getAction()?.element;
    if (currentTarget) {
        patchEventInstance(event, 'currentTarget', currentTarget, {
            // `currentTarget` is going to get reassigned every dispatch.
            configurable: true,
        });
    }
}
/**
 * Patch `Event` instance during non-standard `Event` dispatch. This patches just the `Event`
 * instance that the browser created, it does not patch global properties or methods.
 *
 * This is necessary because dispatching an `Event` outside of browser dispatch results in
 * incorrect properties and methods that need to be polyfilled or do not work.
 *
 * JSAction dispatch adds two extra "phases" to event dispatch:
 * 1. Event delegation - the event is being dispatched by a delegating event handler on a container
 *    (typically `window.document.documentElement`), to a delegated event handler on some child
 *    element. Certain `Event` properties will be unintuitive, such as `currentTarget`, which would
 *    be the container rather than the child element. Bubbling would also not work. In order to
 *    emulate the browser, these properties and methods on the `Event` are patched.
 * 2. Event replay - the event is being dispatched by the framework once the handlers have been
 *    loaded (during hydration, or late-loaded). Certain `Event` properties can be unset by the
 *    browser because the `Event` is no longer actively being dispatched, such as `target`. Other
 *    methods have no effect because the `Event` has already been dispatched, such as
 *    `preventDefault`. Bubbling would also not work. These properties and methods are patched,
 *    either to fill in information that the browser may have removed, or to throw errors in methods
 *    that no longer behave as expected.
 */
function patchEventInstance(event, property, value, { configurable = false } = {}) {
    Object.defineProperty(event, property, { value, configurable });
}
/**
 * Registers deferred functionality for an EventContract and a Jsaction
 * Dispatcher.
 */
export function registerDispatcher(eventContract, dispatcher) {
    eventContract.ecrd((eventInfo) => {
        dispatcher.dispatch(eventInfo);
    }, Restriction.I_AM_THE_JSACTION_FRAMEWORK);
}
//# sourceMappingURL=data:application/json;base64,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