/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { defaultEquals } from './equality';
import { throwInvalidWriteToSignalError } from './errors';
import { producerAccessed, producerIncrementEpoch, producerNotifyConsumers, producerUpdatesAllowed, REACTIVE_NODE, SIGNAL, } from './graph';
/**
 * If set, called after `WritableSignal`s are updated.
 *
 * This hook can be used to achieve various effects, such as running effects synchronously as part
 * of setting a signal.
 */
let postSignalSetFn = null;
/**
 * Create a `Signal` that can be set or updated directly.
 */
export function createSignal(initialValue) {
    const node = Object.create(SIGNAL_NODE);
    node.value = initialValue;
    const getter = (() => {
        producerAccessed(node);
        return node.value;
    });
    getter[SIGNAL] = node;
    return getter;
}
export function setPostSignalSetFn(fn) {
    const prev = postSignalSetFn;
    postSignalSetFn = fn;
    return prev;
}
export function signalGetFn() {
    producerAccessed(this);
    return this.value;
}
export function signalSetFn(node, newValue) {
    if (!producerUpdatesAllowed()) {
        throwInvalidWriteToSignalError();
    }
    if (!node.equal(node.value, newValue)) {
        node.value = newValue;
        signalValueChanged(node);
    }
}
export function signalUpdateFn(node, updater) {
    if (!producerUpdatesAllowed()) {
        throwInvalidWriteToSignalError();
    }
    signalSetFn(node, updater(node.value));
}
export function runPostSignalSetFn() {
    postSignalSetFn?.();
}
// Note: Using an IIFE here to ensure that the spread assignment is not considered
// a side-effect, ending up preserving `COMPUTED_NODE` and `REACTIVE_NODE`.
// TODO: remove when https://github.com/evanw/esbuild/issues/3392 is resolved.
export const SIGNAL_NODE = /* @__PURE__ */ (() => {
    return {
        ...REACTIVE_NODE,
        equal: defaultEquals,
        value: undefined,
    };
})();
function signalValueChanged(node) {
    node.version++;
    producerIncrementEpoch();
    producerNotifyConsumers(node);
    postSignalSetFn?.();
}
//# sourceMappingURL=data:application/json;base64,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