/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { consumerAfterComputation, consumerBeforeComputation, consumerDestroy, consumerMarkDirty, consumerPollProducersForChange, isInNotificationPhase, REACTIVE_NODE, SIGNAL, } from './graph';
export function createWatch(fn, schedule, allowSignalWrites) {
    const node = Object.create(WATCH_NODE);
    if (allowSignalWrites) {
        node.consumerAllowSignalWrites = true;
    }
    node.fn = fn;
    node.schedule = schedule;
    const registerOnCleanup = (cleanupFn) => {
        node.cleanupFn = cleanupFn;
    };
    function isWatchNodeDestroyed(node) {
        return node.fn === null && node.schedule === null;
    }
    function destroyWatchNode(node) {
        if (!isWatchNodeDestroyed(node)) {
            consumerDestroy(node); // disconnect watcher from the reactive graph
            node.cleanupFn();
            // nullify references to the integration functions to mark node as destroyed
            node.fn = null;
            node.schedule = null;
            node.cleanupFn = NOOP_CLEANUP_FN;
        }
    }
    const run = () => {
        if (node.fn === null) {
            // trying to run a destroyed watch is noop
            return;
        }
        if (isInNotificationPhase()) {
            throw new Error(`Schedulers cannot synchronously execute watches while scheduling.`);
        }
        node.dirty = false;
        if (node.hasRun && !consumerPollProducersForChange(node)) {
            return;
        }
        node.hasRun = true;
        const prevConsumer = consumerBeforeComputation(node);
        try {
            node.cleanupFn();
            node.cleanupFn = NOOP_CLEANUP_FN;
            node.fn(registerOnCleanup);
        }
        finally {
            consumerAfterComputation(node, prevConsumer);
        }
    };
    node.ref = {
        notify: () => consumerMarkDirty(node),
        run,
        cleanup: () => node.cleanupFn(),
        destroy: () => destroyWatchNode(node),
        [SIGNAL]: node,
    };
    return node.ref;
}
const NOOP_CLEANUP_FN = () => { };
// Note: Using an IIFE here to ensure that the spread assignment is not considered
// a side-effect, ending up preserving `COMPUTED_NODE` and `REACTIVE_NODE`.
// TODO: remove when https://github.com/evanw/esbuild/issues/3392 is resolved.
const WATCH_NODE = /* @__PURE__ */ (() => {
    return {
        ...REACTIVE_NODE,
        consumerIsAlwaysLive: true,
        consumerAllowSignalWrites: false,
        consumerMarkedDirty: (node) => {
            if (node.schedule !== null) {
                node.schedule(node.ref);
            }
        },
        hasRun: false,
        cleanupFn: NOOP_CLEANUP_FN,
    };
})();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2F0Y2guanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9wYWNrYWdlcy9jb3JlL3ByaW1pdGl2ZXMvc2lnbmFscy9zcmMvd2F0Y2gudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBRUgsT0FBTyxFQUNMLHdCQUF3QixFQUN4Qix5QkFBeUIsRUFDekIsZUFBZSxFQUNmLGlCQUFpQixFQUNqQiw4QkFBOEIsRUFDOUIscUJBQXFCLEVBQ3JCLGFBQWEsRUFFYixNQUFNLEdBQ1AsTUFBTSxTQUFTLENBQUM7QUEyQ2pCLE1BQU0sVUFBVSxXQUFXLENBQ3pCLEVBQStDLEVBQy9DLFFBQWdDLEVBQ2hDLGlCQUEwQjtJQUUxQixNQUFNLElBQUksR0FBYyxNQUFNLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ2xELElBQUksaUJBQWlCLEVBQUUsQ0FBQztRQUN0QixJQUFJLENBQUMseUJBQXlCLEdBQUcsSUFBSSxDQUFDO0lBQ3hDLENBQUM7SUFFRCxJQUFJLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQztJQUNiLElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO0lBRXpCLE1BQU0saUJBQWlCLEdBQUcsQ0FBQyxTQUF5QixFQUFFLEVBQUU7UUFDdEQsSUFBSSxDQUFDLFNBQVMsR0FBRyxTQUFTLENBQUM7SUFDN0IsQ0FBQyxDQUFDO0lBRUYsU0FBUyxvQkFBb0IsQ0FBQyxJQUFlO1FBQzNDLE9BQU8sSUFBSSxDQUFDLEVBQUUsS0FBSyxJQUFJLElBQUksSUFBSSxDQUFDLFFBQVEsS0FBSyxJQUFJLENBQUM7SUFDcEQsQ0FBQztJQUVELFNBQVMsZ0JBQWdCLENBQUMsSUFBZTtRQUN2QyxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUNoQyxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyw2Q0FBNkM7WUFDcEUsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDO1lBRWpCLDRFQUE0RTtZQUM1RSxJQUFJLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQztZQUNmLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDO1lBQ3JCLElBQUksQ0FBQyxTQUFTLEdBQUcsZUFBZSxDQUFDO1FBQ25DLENBQUM7SUFDSCxDQUFDO0lBRUQsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxJQUFJLENBQUMsRUFBRSxLQUFLLElBQUksRUFBRSxDQUFDO1lBQ3JCLDBDQUEwQztZQUMxQyxPQUFPO1FBQ1QsQ0FBQztRQUVELElBQUkscUJBQXFCLEVBQUUsRUFBRSxDQUFDO1lBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUVBQW1FLENBQUMsQ0FBQztRQUN2RixDQUFDO1FBRUQsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7UUFDbkIsSUFBSSxJQUFJLENBQUMsTUFBTSxJQUFJLENBQUMsOEJBQThCLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUN6RCxPQUFPO1FBQ1QsQ0FBQztRQUNELElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDO1FBRW5CLE1BQU0sWUFBWSxHQUFHLHlCQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3JELElBQUksQ0FBQztZQUNILElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQztZQUNqQixJQUFJLENBQUMsU0FBUyxHQUFHLGVBQWUsQ0FBQztZQUNqQyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDN0IsQ0FBQztnQkFBUyxDQUFDO1lBQ1Qsd0JBQXdCLENBQUMsSUFBSSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQy9DLENBQUM7SUFDSCxDQUFDLENBQUM7SUFFRixJQUFJLENBQUMsR0FBRyxHQUFHO1FBQ1QsTUFBTSxFQUFFLEdBQUcsRUFBRSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQztRQUNyQyxHQUFHO1FBQ0gsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7UUFDL0IsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQztRQUNyQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLElBQUk7S0FDZixDQUFDO0lBRUYsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDO0FBQ2xCLENBQUM7QUFFRCxNQUFNLGVBQWUsR0FBbUIsR0FBRyxFQUFFLEdBQUUsQ0FBQyxDQUFDO0FBRWpELGtGQUFrRjtBQUNsRiwyRUFBMkU7QUFDM0UsOEVBQThFO0FBQzlFLE1BQU0sVUFBVSxHQUF1QixlQUFlLENBQUMsQ0FBQyxHQUFHLEVBQUU7SUFDM0QsT0FBTztRQUNMLEdBQUcsYUFBYTtRQUNoQixvQkFBb0IsRUFBRSxJQUFJO1FBQzFCLHlCQUF5QixFQUFFLEtBQUs7UUFDaEMsbUJBQW1CLEVBQUUsQ0FBQyxJQUFlLEVBQUUsRUFBRTtZQUN2QyxJQUFJLElBQUksQ0FBQyxRQUFRLEtBQUssSUFBSSxFQUFFLENBQUM7Z0JBQzNCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQzFCLENBQUM7UUFDSCxDQUFDO1FBQ0QsTUFBTSxFQUFFLEtBQUs7UUFDYixTQUFTLEVBQUUsZUFBZTtLQUMzQixDQUFDO0FBQ0osQ0FBQyxDQUFDLEVBQUUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmRldi9saWNlbnNlXG4gKi9cblxuaW1wb3J0IHtcbiAgY29uc3VtZXJBZnRlckNvbXB1dGF0aW9uLFxuICBjb25zdW1lckJlZm9yZUNvbXB1dGF0aW9uLFxuICBjb25zdW1lckRlc3Ryb3ksXG4gIGNvbnN1bWVyTWFya0RpcnR5LFxuICBjb25zdW1lclBvbGxQcm9kdWNlcnNGb3JDaGFuZ2UsXG4gIGlzSW5Ob3RpZmljYXRpb25QaGFzZSxcbiAgUkVBQ1RJVkVfTk9ERSxcbiAgUmVhY3RpdmVOb2RlLFxuICBTSUdOQUwsXG59IGZyb20gJy4vZ3JhcGgnO1xuXG4vKipcbiAqIEEgY2xlYW51cCBmdW5jdGlvbiB0aGF0IGNhbiBiZSBvcHRpb25hbGx5IHJlZ2lzdGVyZWQgZnJvbSB0aGUgd2F0Y2ggbG9naWMuIElmIHJlZ2lzdGVyZWQsIHRoZVxuICogY2xlYW51cCBsb2dpYyBydW5zIGJlZm9yZSB0aGUgbmV4dCB3YXRjaCBleGVjdXRpb24uXG4gKi9cbmV4cG9ydCB0eXBlIFdhdGNoQ2xlYW51cEZuID0gKCkgPT4gdm9pZDtcblxuLyoqXG4gKiBBIGNhbGxiYWNrIHBhc3NlZCB0byB0aGUgd2F0Y2ggZnVuY3Rpb24gdGhhdCBtYWtlcyBpdCBwb3NzaWJsZSB0byByZWdpc3RlciBjbGVhbnVwIGxvZ2ljLlxuICovXG5leHBvcnQgdHlwZSBXYXRjaENsZWFudXBSZWdpc3RlckZuID0gKGNsZWFudXBGbjogV2F0Y2hDbGVhbnVwRm4pID0+IHZvaWQ7XG5cbmV4cG9ydCBpbnRlcmZhY2UgV2F0Y2gge1xuICBub3RpZnkoKTogdm9pZDtcblxuICAvKipcbiAgICogRXhlY3V0ZSB0aGUgcmVhY3RpdmUgZXhwcmVzc2lvbiBpbiB0aGUgY29udGV4dCBvZiB0aGlzIGBXYXRjaGAgY29uc3VtZXIuXG4gICAqXG4gICAqIFNob3VsZCBiZSBjYWxsZWQgYnkgdGhlIHVzZXIgc2NoZWR1bGluZyBhbGdvcml0aG0gd2hlbiB0aGUgcHJvdmlkZWRcbiAgICogYHNjaGVkdWxlYCBob29rIGlzIGNhbGxlZCBieSBgV2F0Y2hgLlxuICAgKi9cbiAgcnVuKCk6IHZvaWQ7XG5cbiAgY2xlYW51cCgpOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBEZXN0cm95IHRoZSB3YXRjaGVyOlxuICAgKiAtIGRpc2Nvbm5lY3QgaXQgZnJvbSB0aGUgcmVhY3RpdmUgZ3JhcGg7XG4gICAqIC0gbWFyayBpdCBhcyBkZXN0cm95ZWQgc28gc3Vic2VxdWVudCBydW4gYW5kIG5vdGlmeSBvcGVyYXRpb25zIGFyZSBub29wLlxuICAgKi9cbiAgZGVzdHJveSgpOiB2b2lkO1xuXG4gIFtTSUdOQUxdOiBXYXRjaE5vZGU7XG59XG5leHBvcnQgaW50ZXJmYWNlIFdhdGNoTm9kZSBleHRlbmRzIFJlYWN0aXZlTm9kZSB7XG4gIGhhc1J1bjogYm9vbGVhbjtcbiAgZm46ICgob25DbGVhbnVwOiBXYXRjaENsZWFudXBSZWdpc3RlckZuKSA9PiB2b2lkKSB8IG51bGw7XG4gIHNjaGVkdWxlOiAoKHdhdGNoOiBXYXRjaCkgPT4gdm9pZCkgfCBudWxsO1xuICBjbGVhbnVwRm46IFdhdGNoQ2xlYW51cEZuO1xuICByZWY6IFdhdGNoO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlV2F0Y2goXG4gIGZuOiAob25DbGVhbnVwOiBXYXRjaENsZWFudXBSZWdpc3RlckZuKSA9PiB2b2lkLFxuICBzY2hlZHVsZTogKHdhdGNoOiBXYXRjaCkgPT4gdm9pZCxcbiAgYWxsb3dTaWduYWxXcml0ZXM6IGJvb2xlYW4sXG4pOiBXYXRjaCB7XG4gIGNvbnN0IG5vZGU6IFdhdGNoTm9kZSA9IE9iamVjdC5jcmVhdGUoV0FUQ0hfTk9ERSk7XG4gIGlmIChhbGxvd1NpZ25hbFdyaXRlcykge1xuICAgIG5vZGUuY29uc3VtZXJBbGxvd1NpZ25hbFdyaXRlcyA9IHRydWU7XG4gIH1cblxuICBub2RlLmZuID0gZm47XG4gIG5vZGUuc2NoZWR1bGUgPSBzY2hlZHVsZTtcblxuICBjb25zdCByZWdpc3Rlck9uQ2xlYW51cCA9IChjbGVhbnVwRm46IFdhdGNoQ2xlYW51cEZuKSA9PiB7XG4gICAgbm9kZS5jbGVhbnVwRm4gPSBjbGVhbnVwRm47XG4gIH07XG5cbiAgZnVuY3Rpb24gaXNXYXRjaE5vZGVEZXN0cm95ZWQobm9kZTogV2F0Y2hOb2RlKSB7XG4gICAgcmV0dXJuIG5vZGUuZm4gPT09IG51bGwgJiYgbm9kZS5zY2hlZHVsZSA9PT0gbnVsbDtcbiAgfVxuXG4gIGZ1bmN0aW9uIGRlc3Ryb3lXYXRjaE5vZGUobm9kZTogV2F0Y2hOb2RlKSB7XG4gICAgaWYgKCFpc1dhdGNoTm9kZURlc3Ryb3llZChub2RlKSkge1xuICAgICAgY29uc3VtZXJEZXN0cm95KG5vZGUpOyAvLyBkaXNjb25uZWN0IHdhdGNoZXIgZnJvbSB0aGUgcmVhY3RpdmUgZ3JhcGhcbiAgICAgIG5vZGUuY2xlYW51cEZuKCk7XG5cbiAgICAgIC8vIG51bGxpZnkgcmVmZXJlbmNlcyB0byB0aGUgaW50ZWdyYXRpb24gZnVuY3Rpb25zIHRvIG1hcmsgbm9kZSBhcyBkZXN0cm95ZWRcbiAgICAgIG5vZGUuZm4gPSBudWxsO1xuICAgICAgbm9kZS5zY2hlZHVsZSA9IG51bGw7XG4gICAgICBub2RlLmNsZWFudXBGbiA9IE5PT1BfQ0xFQU5VUF9GTjtcbiAgICB9XG4gIH1cblxuICBjb25zdCBydW4gPSAoKSA9PiB7XG4gICAgaWYgKG5vZGUuZm4gPT09IG51bGwpIHtcbiAgICAgIC8vIHRyeWluZyB0byBydW4gYSBkZXN0cm95ZWQgd2F0Y2ggaXMgbm9vcFxuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmIChpc0luTm90aWZpY2F0aW9uUGhhc2UoKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBTY2hlZHVsZXJzIGNhbm5vdCBzeW5jaHJvbm91c2x5IGV4ZWN1dGUgd2F0Y2hlcyB3aGlsZSBzY2hlZHVsaW5nLmApO1xuICAgIH1cblxuICAgIG5vZGUuZGlydHkgPSBmYWxzZTtcbiAgICBpZiAobm9kZS5oYXNSdW4gJiYgIWNvbnN1bWVyUG9sbFByb2R1Y2Vyc0ZvckNoYW5nZShub2RlKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBub2RlLmhhc1J1biA9IHRydWU7XG5cbiAgICBjb25zdCBwcmV2Q29uc3VtZXIgPSBjb25zdW1lckJlZm9yZUNvbXB1dGF0aW9uKG5vZGUpO1xuICAgIHRyeSB7XG4gICAgICBub2RlLmNsZWFudXBGbigpO1xuICAgICAgbm9kZS5jbGVhbnVwRm4gPSBOT09QX0NMRUFOVVBfRk47XG4gICAgICBub2RlLmZuKHJlZ2lzdGVyT25DbGVhbnVwKTtcbiAgICB9IGZpbmFsbHkge1xuICAgICAgY29uc3VtZXJBZnRlckNvbXB1dGF0aW9uKG5vZGUsIHByZXZDb25zdW1lcik7XG4gICAgfVxuICB9O1xuXG4gIG5vZGUucmVmID0ge1xuICAgIG5vdGlmeTogKCkgPT4gY29uc3VtZXJNYXJrRGlydHkobm9kZSksXG4gICAgcnVuLFxuICAgIGNsZWFudXA6ICgpID0+IG5vZGUuY2xlYW51cEZuKCksXG4gICAgZGVzdHJveTogKCkgPT4gZGVzdHJveVdhdGNoTm9kZShub2RlKSxcbiAgICBbU0lHTkFMXTogbm9kZSxcbiAgfTtcblxuICByZXR1cm4gbm9kZS5yZWY7XG59XG5cbmNvbnN0IE5PT1BfQ0xFQU5VUF9GTjogV2F0Y2hDbGVhbnVwRm4gPSAoKSA9PiB7fTtcblxuLy8gTm90ZTogVXNpbmcgYW4gSUlGRSBoZXJlIHRvIGVuc3VyZSB0aGF0IHRoZSBzcHJlYWQgYXNzaWdubWVudCBpcyBub3QgY29uc2lkZXJlZFxuLy8gYSBzaWRlLWVmZmVjdCwgZW5kaW5nIHVwIHByZXNlcnZpbmcgYENPTVBVVEVEX05PREVgIGFuZCBgUkVBQ1RJVkVfTk9ERWAuXG4vLyBUT0RPOiByZW1vdmUgd2hlbiBodHRwczovL2dpdGh1Yi5jb20vZXZhbncvZXNidWlsZC9pc3N1ZXMvMzM5MiBpcyByZXNvbHZlZC5cbmNvbnN0IFdBVENIX05PREU6IFBhcnRpYWw8V2F0Y2hOb2RlPiA9IC8qIEBfX1BVUkVfXyAqLyAoKCkgPT4ge1xuICByZXR1cm4ge1xuICAgIC4uLlJFQUNUSVZFX05PREUsXG4gICAgY29uc3VtZXJJc0Fsd2F5c0xpdmU6IHRydWUsXG4gICAgY29uc3VtZXJBbGxvd1NpZ25hbFdyaXRlczogZmFsc2UsXG4gICAgY29uc3VtZXJNYXJrZWREaXJ0eTogKG5vZGU6IFdhdGNoTm9kZSkgPT4ge1xuICAgICAgaWYgKG5vZGUuc2NoZWR1bGUgIT09IG51bGwpIHtcbiAgICAgICAgbm9kZS5zY2hlZHVsZShub2RlLnJlZik7XG4gICAgICB9XG4gICAgfSxcbiAgICBoYXNSdW46IGZhbHNlLFxuICAgIGNsZWFudXBGbjogTk9PUF9DTEVBTlVQX0ZOLFxuICB9O1xufSkoKTtcbiJdfQ==