/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { assertInInjectionContext } from '../../di';
import { createInputSignal, } from './input_signal';
import { REQUIRED_UNSET_VALUE } from './input_signal_node';
export function inputFunction(initialValue, opts) {
    ngDevMode && assertInInjectionContext(input);
    return createInputSignal(initialValue, opts);
}
export function inputRequiredFunction(opts) {
    ngDevMode && assertInInjectionContext(input);
    return createInputSignal(REQUIRED_UNSET_VALUE, opts);
}
/**
 * The `input` function allows declaration of Angular inputs in directives
 * and components.
 *
 * There are two variants of inputs that can be declared:
 *
 *   1. **Optional inputs** with an initial value.
 *   2. **Required inputs** that consumers need to set.
 *
 * By default, the `input` function will declare optional inputs that
 * always have an initial value. Required inputs can be declared
 * using the `input.required()` function.
 *
 * Inputs are signals. The values of an input are exposed as a `Signal`.
 * The signal always holds the latest value of the input that is bound
 * from the parent.
 *
 * @usageNotes
 * To use signal-based inputs, import `input` from `@angular/core`.
 *
 * ```
 * import {input} from '@angular/core`;
 * ```
 *
 * Inside your component, introduce a new class member and initialize
 * it with a call to `input` or `input.required`.
 *
 * ```ts
 * @Component({
 *   ...
 * })
 * export class UserProfileComponent {
 *   firstName = input<string>();             // Signal<string|undefined>
 *   lastName  = input.required<string>();    // Signal<string>
 *   age       = input(0)                     // Signal<number>
 * }
 * ```
 *
 * Inside your component template, you can display values of the inputs
 * by calling the signal.
 *
 * ```html
 * <span>{{firstName()}}</span>
 * ```
 *
 * @developerPreview
 * @initializerApiFunction
 */
export const input = (() => {
    // Note: This may be considered a side-effect, but nothing will depend on
    // this assignment, unless this `input` constant export is accessed. It's a
    // self-contained side effect that is local to the user facing`input` export.
    inputFunction.required = inputRequiredFunction;
    return inputFunction;
})();
//# sourceMappingURL=data:application/json;base64,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