/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { producerAccessed, SIGNAL } from '@angular/core/primitives/signals';
import { RuntimeError } from '../../errors';
import { INPUT_SIGNAL_NODE, REQUIRED_UNSET_VALUE } from './input_signal_node';
export const ɵINPUT_SIGNAL_BRAND_READ_TYPE = /* @__PURE__ */ Symbol();
export const ɵINPUT_SIGNAL_BRAND_WRITE_TYPE = /* @__PURE__ */ Symbol();
/**
 * Creates an input signal.
 *
 * @param initialValue The initial value.
 *   Can be set to {@link REQUIRED_UNSET_VALUE} for required inputs.
 * @param options Additional options for the input. e.g. a transform, or an alias.
 */
export function createInputSignal(initialValue, options) {
    const node = Object.create(INPUT_SIGNAL_NODE);
    node.value = initialValue;
    // Perf note: Always set `transformFn` here to ensure that `node` always
    // has the same v8 class shape, allowing monomorphic reads on input signals.
    node.transformFn = options?.transform;
    function inputValueFn() {
        // Record that someone looked at this signal.
        producerAccessed(node);
        if (node.value === REQUIRED_UNSET_VALUE) {
            throw new RuntimeError(-950 /* RuntimeErrorCode.REQUIRED_INPUT_NO_VALUE */, ngDevMode && 'Input is required but no value is available yet.');
        }
        return node.value;
    }
    inputValueFn[SIGNAL] = node;
    if (ngDevMode) {
        inputValueFn.toString = () => `[Input Signal: ${inputValueFn()}]`;
    }
    return inputValueFn;
}
//# sourceMappingURL=data:application/json;base64,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