/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { producerAccessed, SIGNAL, signalSetFn } from '@angular/core/primitives/signals';
import { RuntimeError } from '../../errors';
import { signalAsReadonlyFn, } from '../../render3/reactivity/signal';
import { INPUT_SIGNAL_NODE, REQUIRED_UNSET_VALUE } from '../input/input_signal_node';
import { OutputEmitterRef } from '../output/output_emitter_ref';
/**
 * Creates a model signal.
 *
 * @param initialValue The initial value.
 *   Can be set to {@link REQUIRED_UNSET_VALUE} for required model signals.
 * @param options Additional options for the model.
 */
export function createModelSignal(initialValue) {
    const node = Object.create(INPUT_SIGNAL_NODE);
    const emitterRef = new OutputEmitterRef();
    node.value = initialValue;
    function getter() {
        producerAccessed(node);
        assertModelSet(node.value);
        return node.value;
    }
    getter[SIGNAL] = node;
    getter.asReadonly = signalAsReadonlyFn.bind(getter);
    // TODO: Should we throw an error when updating a destroyed model?
    getter.set = (newValue) => {
        if (!node.equal(node.value, newValue)) {
            signalSetFn(node, newValue);
            emitterRef.emit(newValue);
        }
    };
    getter.update = (updateFn) => {
        assertModelSet(node.value);
        getter.set(updateFn(node.value));
    };
    getter.subscribe = emitterRef.subscribe.bind(emitterRef);
    getter.destroyRef = emitterRef.destroyRef;
    if (ngDevMode) {
        getter.toString = () => `[Model Signal: ${getter()}]`;
    }
    return getter;
}
/** Asserts that a model's value is set. */
function assertModelSet(value) {
    if (value === REQUIRED_UNSET_VALUE) {
        throw new RuntimeError(952 /* RuntimeErrorCode.REQUIRED_MODEL_NO_VALUE */, ngDevMode && 'Model is required but no value is available yet.');
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibW9kZWxfc2lnbmFsLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvY29yZS9zcmMvYXV0aG9yaW5nL21vZGVsL21vZGVsX3NpZ25hbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7QUFFSCxPQUFPLEVBQUMsZ0JBQWdCLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBQyxNQUFNLGtDQUFrQyxDQUFDO0FBRXZGLE9BQU8sRUFBQyxZQUFZLEVBQW1CLE1BQU0sY0FBYyxDQUFDO0FBRTVELE9BQU8sRUFDTCxrQkFBa0IsR0FHbkIsTUFBTSxpQ0FBaUMsQ0FBQztBQU16QyxPQUFPLEVBQUMsaUJBQWlCLEVBQW1CLG9CQUFvQixFQUFDLE1BQU0sNEJBQTRCLENBQUM7QUFDcEcsT0FBTyxFQUFDLGdCQUFnQixFQUFDLE1BQU0sOEJBQThCLENBQUM7QUE0QjlEOzs7Ozs7R0FNRztBQUNILE1BQU0sVUFBVSxpQkFBaUIsQ0FBSSxZQUFlO0lBQ2xELE1BQU0sSUFBSSxHQUEwQixNQUFNLENBQUMsTUFBTSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDckUsTUFBTSxVQUFVLEdBQUcsSUFBSSxnQkFBZ0IsRUFBSyxDQUFDO0lBRTdDLElBQUksQ0FBQyxLQUFLLEdBQUcsWUFBWSxDQUFDO0lBRTFCLFNBQVMsTUFBTTtRQUNiLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZCLGNBQWMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3BCLENBQUM7SUFFRCxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQUcsSUFBSSxDQUFDO0lBQ3RCLE1BQU0sQ0FBQyxVQUFVLEdBQUcsa0JBQWtCLENBQUMsSUFBSSxDQUFDLE1BQWEsQ0FBb0IsQ0FBQztJQUU5RSxrRUFBa0U7SUFDbEUsTUFBTSxDQUFDLEdBQUcsR0FBRyxDQUFDLFFBQVcsRUFBRSxFQUFFO1FBQzNCLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLEVBQUUsQ0FBQztZQUN0QyxXQUFXLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQzVCLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDNUIsQ0FBQztJQUNILENBQUMsQ0FBQztJQUVGLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxRQUF5QixFQUFFLEVBQUU7UUFDNUMsY0FBYyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUMzQixNQUFNLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUNuQyxDQUFDLENBQUM7SUFFRixNQUFNLENBQUMsU0FBUyxHQUFHLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3pELE1BQU0sQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDLFVBQVUsQ0FBQztJQUUxQyxJQUFJLFNBQVMsRUFBRSxDQUFDO1FBQ2QsTUFBTSxDQUFDLFFBQVEsR0FBRyxHQUFHLEVBQUUsQ0FBQyxrQkFBa0IsTUFBTSxFQUFFLEdBQUcsQ0FBQztJQUN4RCxDQUFDO0lBRUQsT0FBTyxNQU1KLENBQUM7QUFDTixDQUFDO0FBRUQsMkNBQTJDO0FBQzNDLFNBQVMsY0FBYyxDQUFDLEtBQWM7SUFDcEMsSUFBSSxLQUFLLEtBQUssb0JBQW9CLEVBQUUsQ0FBQztRQUNuQyxNQUFNLElBQUksWUFBWSxxREFFcEIsU0FBUyxJQUFJLGtEQUFrRCxDQUNoRSxDQUFDO0lBQ0osQ0FBQztBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIExMQyBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5kZXYvbGljZW5zZVxuICovXG5cbmltcG9ydCB7cHJvZHVjZXJBY2Nlc3NlZCwgU0lHTkFMLCBzaWduYWxTZXRGbn0gZnJvbSAnQGFuZ3VsYXIvY29yZS9wcmltaXRpdmVzL3NpZ25hbHMnO1xuXG5pbXBvcnQge1J1bnRpbWVFcnJvciwgUnVudGltZUVycm9yQ29kZX0gZnJvbSAnLi4vLi4vZXJyb3JzJztcbmltcG9ydCB7U2lnbmFsfSBmcm9tICcuLi8uLi9yZW5kZXIzL3JlYWN0aXZpdHkvYXBpJztcbmltcG9ydCB7XG4gIHNpZ25hbEFzUmVhZG9ubHlGbixcbiAgV3JpdGFibGVTaWduYWwsXG4gIMm1V1JJVEFCTEVfU0lHTkFMLFxufSBmcm9tICcuLi8uLi9yZW5kZXIzL3JlYWN0aXZpdHkvc2lnbmFsJztcbmltcG9ydCB7XG4gIElucHV0U2lnbmFsLFxuICDJtUlOUFVUX1NJR05BTF9CUkFORF9SRUFEX1RZUEUsXG4gIMm1SU5QVVRfU0lHTkFMX0JSQU5EX1dSSVRFX1RZUEUsXG59IGZyb20gJy4uL2lucHV0L2lucHV0X3NpZ25hbCc7XG5pbXBvcnQge0lOUFVUX1NJR05BTF9OT0RFLCBJbnB1dFNpZ25hbE5vZGUsIFJFUVVJUkVEX1VOU0VUX1ZBTFVFfSBmcm9tICcuLi9pbnB1dC9pbnB1dF9zaWduYWxfbm9kZSc7XG5pbXBvcnQge091dHB1dEVtaXR0ZXJSZWZ9IGZyb20gJy4uL291dHB1dC9vdXRwdXRfZW1pdHRlcl9yZWYnO1xuaW1wb3J0IHtPdXRwdXRSZWZ9IGZyb20gJy4uL291dHB1dC9vdXRwdXRfcmVmJztcblxuLyoqXG4gKiBAZGV2ZWxvcGVyUHJldmlld1xuICpcbiAqIE9wdGlvbnMgZm9yIG1vZGVsIHNpZ25hbHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTW9kZWxPcHRpb25zIHtcbiAgLyoqXG4gICAqIE9wdGlvbmFsIHB1YmxpYyBuYW1lIG9mIHRoZSBpbnB1dCBzaWRlIG9mIHRoZSBtb2RlbC4gVGhlIG91dHB1dCBzaWRlIHdpbGwgaGF2ZSB0aGUgc2FtZVxuICAgKiBuYW1lIGFzIHRoZSBpbnB1dCwgYnV0IHN1ZmZpeGVkIHdpdGggYENoYW5nZWAuIEJ5IGRlZmF1bHQsIHRoZSBjbGFzcyBmaWVsZCBuYW1lIGlzIHVzZWQuXG4gICAqL1xuICBhbGlhcz86IHN0cmluZztcbn1cblxuLyoqXG4gKiBgTW9kZWxTaWduYWxgIHJlcHJlc2VudHMgYSBzcGVjaWFsIGBTaWduYWxgIGZvciBhIGRpcmVjdGl2ZS9jb21wb25lbnQgbW9kZWwgZmllbGQuXG4gKlxuICogQSBtb2RlbCBzaWduYWwgaXMgYSB3cml0ZWFibGUgc2lnbmFsIHRoYXQgY2FuIGJlIGV4cG9zZWQgYXMgYW4gb3V0cHV0LlxuICogV2hlbmV2ZXIgaXRzIHZhbHVlIGlzIHVwZGF0ZWQsIGl0IGVtaXRzIHRvIHRoZSBvdXRwdXQuXG4gKlxuICogQGRldmVsb3BlclByZXZpZXdcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNb2RlbFNpZ25hbDxUPiBleHRlbmRzIFdyaXRhYmxlU2lnbmFsPFQ+LCBJbnB1dFNpZ25hbDxUPiwgT3V0cHV0UmVmPFQ+IHtcbiAgW1NJR05BTF06IElucHV0U2lnbmFsTm9kZTxULCBUPjtcbn1cblxuLyoqXG4gKiBDcmVhdGVzIGEgbW9kZWwgc2lnbmFsLlxuICpcbiAqIEBwYXJhbSBpbml0aWFsVmFsdWUgVGhlIGluaXRpYWwgdmFsdWUuXG4gKiAgIENhbiBiZSBzZXQgdG8ge0BsaW5rIFJFUVVJUkVEX1VOU0VUX1ZBTFVFfSBmb3IgcmVxdWlyZWQgbW9kZWwgc2lnbmFscy5cbiAqIEBwYXJhbSBvcHRpb25zIEFkZGl0aW9uYWwgb3B0aW9ucyBmb3IgdGhlIG1vZGVsLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlTW9kZWxTaWduYWw8VD4oaW5pdGlhbFZhbHVlOiBUKTogTW9kZWxTaWduYWw8VD4ge1xuICBjb25zdCBub2RlOiBJbnB1dFNpZ25hbE5vZGU8VCwgVD4gPSBPYmplY3QuY3JlYXRlKElOUFVUX1NJR05BTF9OT0RFKTtcbiAgY29uc3QgZW1pdHRlclJlZiA9IG5ldyBPdXRwdXRFbWl0dGVyUmVmPFQ+KCk7XG5cbiAgbm9kZS52YWx1ZSA9IGluaXRpYWxWYWx1ZTtcblxuICBmdW5jdGlvbiBnZXR0ZXIoKTogVCB7XG4gICAgcHJvZHVjZXJBY2Nlc3NlZChub2RlKTtcbiAgICBhc3NlcnRNb2RlbFNldChub2RlLnZhbHVlKTtcbiAgICByZXR1cm4gbm9kZS52YWx1ZTtcbiAgfVxuXG4gIGdldHRlcltTSUdOQUxdID0gbm9kZTtcbiAgZ2V0dGVyLmFzUmVhZG9ubHkgPSBzaWduYWxBc1JlYWRvbmx5Rm4uYmluZChnZXR0ZXIgYXMgYW55KSBhcyAoKSA9PiBTaWduYWw8VD47XG5cbiAgLy8gVE9ETzogU2hvdWxkIHdlIHRocm93IGFuIGVycm9yIHdoZW4gdXBkYXRpbmcgYSBkZXN0cm95ZWQgbW9kZWw/XG4gIGdldHRlci5zZXQgPSAobmV3VmFsdWU6IFQpID0+IHtcbiAgICBpZiAoIW5vZGUuZXF1YWwobm9kZS52YWx1ZSwgbmV3VmFsdWUpKSB7XG4gICAgICBzaWduYWxTZXRGbihub2RlLCBuZXdWYWx1ZSk7XG4gICAgICBlbWl0dGVyUmVmLmVtaXQobmV3VmFsdWUpO1xuICAgIH1cbiAgfTtcblxuICBnZXR0ZXIudXBkYXRlID0gKHVwZGF0ZUZuOiAodmFsdWU6IFQpID0+IFQpID0+IHtcbiAgICBhc3NlcnRNb2RlbFNldChub2RlLnZhbHVlKTtcbiAgICBnZXR0ZXIuc2V0KHVwZGF0ZUZuKG5vZGUudmFsdWUpKTtcbiAgfTtcblxuICBnZXR0ZXIuc3Vic2NyaWJlID0gZW1pdHRlclJlZi5zdWJzY3JpYmUuYmluZChlbWl0dGVyUmVmKTtcbiAgZ2V0dGVyLmRlc3Ryb3lSZWYgPSBlbWl0dGVyUmVmLmRlc3Ryb3lSZWY7XG5cbiAgaWYgKG5nRGV2TW9kZSkge1xuICAgIGdldHRlci50b1N0cmluZyA9ICgpID0+IGBbTW9kZWwgU2lnbmFsOiAke2dldHRlcigpfV1gO1xuICB9XG5cbiAgcmV0dXJuIGdldHRlciBhcyB0eXBlb2YgZ2V0dGVyICZcbiAgICBQaWNrPFxuICAgICAgTW9kZWxTaWduYWw8VD4sXG4gICAgICB8IHR5cGVvZiDJtUlOUFVUX1NJR05BTF9CUkFORF9SRUFEX1RZUEVcbiAgICAgIHwgdHlwZW9mIMm1SU5QVVRfU0lHTkFMX0JSQU5EX1dSSVRFX1RZUEVcbiAgICAgIHwgdHlwZW9mIMm1V1JJVEFCTEVfU0lHTkFMXG4gICAgPjtcbn1cblxuLyoqIEFzc2VydHMgdGhhdCBhIG1vZGVsJ3MgdmFsdWUgaXMgc2V0LiAqL1xuZnVuY3Rpb24gYXNzZXJ0TW9kZWxTZXQodmFsdWU6IHVua25vd24pOiB2b2lkIHtcbiAgaWYgKHZhbHVlID09PSBSRVFVSVJFRF9VTlNFVF9WQUxVRSkge1xuICAgIHRocm93IG5ldyBSdW50aW1lRXJyb3IoXG4gICAgICBSdW50aW1lRXJyb3JDb2RlLlJFUVVJUkVEX01PREVMX05PX1ZBTFVFLFxuICAgICAgbmdEZXZNb2RlICYmICdNb2RlbCBpcyByZXF1aXJlZCBidXQgbm8gdmFsdWUgaXMgYXZhaWxhYmxlIHlldC4nLFxuICAgICk7XG4gIH1cbn1cbiJdfQ==