/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { assertInInjectionContext } from '../di';
import { createMultiResultQuerySignalFn, createSingleResultOptionalQuerySignalFn, createSingleResultRequiredQuerySignalFn, } from '../render3/query_reactive';
function viewChildFn(locator, opts) {
    ngDevMode && assertInInjectionContext(viewChild);
    return createSingleResultOptionalQuerySignalFn();
}
function viewChildRequiredFn(locator, opts) {
    ngDevMode && assertInInjectionContext(viewChild);
    return createSingleResultRequiredQuerySignalFn();
}
/**
 * Initializes a view child query.
 *
 * Consider using `viewChild.required` for queries that should always match.
 *
 * @usageNotes
 * Create a child query in your component by declaring a
 * class field and initializing it with the `viewChild()` function.
 *
 * ```ts
 * @Component({template: '<div #el></div><my-component #cmp />'})
 * export class TestComponent {
 *   divEl = viewChild<ElementRef>('el');                   // Signal<ElementRef|undefined>
 *   divElRequired = viewChild.required<ElementRef>('el');  // Signal<ElementRef>
 *   cmp = viewChild(MyComponent);                          // Signal<MyComponent|undefined>
 *   cmpRequired = viewChild.required(MyComponent);         // Signal<MyComponent>
 * }
 * ```
 *
 * @developerPreview
 * @initializerApiFunction
 */
export const viewChild = (() => {
    // Note: This may be considered a side-effect, but nothing will depend on
    // this assignment, unless this `viewChild` constant export is accessed. It's a
    // self-contained side effect that is local to the user facing `viewChild` export.
    viewChildFn.required = viewChildRequiredFn;
    return viewChildFn;
})();
/**
 * Initializes a view children query.
 *
 * Query results are represented as a signal of a read-only collection containing all matched
 * elements.
 *
 * @usageNotes
 * Create a children query in your component by declaring a
 * class field and initializing it with the `viewChildren()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   divEls = viewChildren<ElementRef>('el');   // Signal<ReadonlyArray<ElementRef>>
 * }
 * ```
 *
 * @initializerApiFunction
 * @developerPreview
 */
export function viewChildren(locator, opts) {
    ngDevMode && assertInInjectionContext(viewChildren);
    return createMultiResultQuerySignalFn();
}
export function contentChildFn(locator, opts) {
    ngDevMode && assertInInjectionContext(contentChild);
    return createSingleResultOptionalQuerySignalFn();
}
function contentChildRequiredFn(locator, opts) {
    ngDevMode && assertInInjectionContext(contentChildren);
    return createSingleResultRequiredQuerySignalFn();
}
/**
 * Initializes a content child query. Consider using `contentChild.required` for queries that should
 * always match.
 *
 * @usageNotes
 * Create a child query in your component by declaring a
 * class field and initializing it with the `contentChild()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   headerEl = contentChild<ElementRef>('h');                    // Signal<ElementRef|undefined>
 *   headerElElRequired = contentChild.required<ElementRef>('h'); // Signal<ElementRef>
 *   header = contentChild(MyHeader);                             // Signal<MyHeader|undefined>
 *   headerRequired = contentChild.required(MyHeader);            // Signal<MyHeader>
 * }
 * ```
 *
 * @initializerApiFunction
 * @developerPreview
 */
export const contentChild = (() => {
    // Note: This may be considered a side-effect, but nothing will depend on
    // this assignment, unless this `viewChild` constant export is accessed. It's a
    // self-contained side effect that is local to the user facing `viewChild` export.
    contentChildFn.required = contentChildRequiredFn;
    return contentChildFn;
})();
/**
 * Initializes a content children query.
 *
 * Query results are represented as a signal of a read-only collection containing all matched
 * elements.
 *
 * @usageNotes
 * Create a children query in your component by declaring a
 * class field and initializing it with the `contentChildren()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   headerEl = contentChildren<ElementRef>('h');   // Signal<ReadonlyArray<ElementRef>>
 * }
 * ```
 *
 * @initializerApiFunction
 * @developerPreview
 */
export function contentChildren(locator, opts) {
    return createMultiResultQuerySignalFn();
}
//# sourceMappingURL=data:application/json;base64,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