/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { ɵɵdefineInjectable } from '../../di/interface/defs';
import { Optional, SkipSelf } from '../../di/metadata';
import { RuntimeError } from '../../errors';
import { DefaultIterableDifferFactory } from '../differs/default_iterable_differ';
export function defaultIterableDiffersFactory() {
    return new IterableDiffers([new DefaultIterableDifferFactory()]);
}
/**
 * A repository of different iterable diffing strategies used by NgFor, NgClass, and others.
 *
 * @publicApi
 */
export class IterableDiffers {
    /** @nocollapse */
    static { this.ɵprov = ɵɵdefineInjectable({
        token: IterableDiffers,
        providedIn: 'root',
        factory: defaultIterableDiffersFactory,
    }); }
    constructor(factories) {
        this.factories = factories;
    }
    static create(factories, parent) {
        if (parent != null) {
            const copied = parent.factories.slice();
            factories = factories.concat(copied);
        }
        return new IterableDiffers(factories);
    }
    /**
     * Takes an array of {@link IterableDifferFactory} and returns a provider used to extend the
     * inherited {@link IterableDiffers} instance with the provided factories and return a new
     * {@link IterableDiffers} instance.
     *
     * @usageNotes
     * ### Example
     *
     * The following example shows how to extend an existing list of factories,
     * which will only be applied to the injector for this component and its children.
     * This step is all that's required to make a new {@link IterableDiffer} available.
     *
     * ```
     * @Component({
     *   viewProviders: [
     *     IterableDiffers.extend([new ImmutableListDiffer()])
     *   ]
     * })
     * ```
     */
    static extend(factories) {
        return {
            provide: IterableDiffers,
            useFactory: (parent) => {
                // if parent is null, it means that we are in the root injector and we have just overridden
                // the default injection mechanism for IterableDiffers, in such a case just assume
                // `defaultIterableDiffersFactory`.
                return IterableDiffers.create(factories, parent || defaultIterableDiffersFactory());
            },
            // Dependency technically isn't optional, but we can provide a better error message this way.
            deps: [[IterableDiffers, new SkipSelf(), new Optional()]],
        };
    }
    find(iterable) {
        const factory = this.factories.find((f) => f.supports(iterable));
        if (factory != null) {
            return factory;
        }
        else {
            throw new RuntimeError(901 /* RuntimeErrorCode.NO_SUPPORTING_DIFFER_FACTORY */, ngDevMode &&
                `Cannot find a differ supporting object '${iterable}' of type '${getTypeNameForDebugging(iterable)}'`);
        }
    }
}
export function getTypeNameForDebugging(type) {
    return type['name'] || typeof type;
}
//# sourceMappingURL=data:application/json;base64,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