/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { throwProviderNotFoundError } from '../render3/errors_di';
import { assertNotEqual } from '../util/assert';
import { getInjectableDef } from './interface/defs';
import { InjectFlags } from './interface/injector';
/**
 * Current implementation of inject.
 *
 * By default, it is `injectInjectorOnly`, which makes it `Injector`-only aware. It can be changed
 * to `directiveInject`, which brings in the `NodeInjector` system of ivy. It is designed this
 * way for two reasons:
 *  1. `Injector` should not depend on ivy logic.
 *  2. To maintain tree shake-ability we don't want to bring in unnecessary code.
 */
let _injectImplementation;
export function getInjectImplementation() {
    return _injectImplementation;
}
/**
 * Sets the current inject implementation.
 */
export function setInjectImplementation(impl) {
    const previous = _injectImplementation;
    _injectImplementation = impl;
    return previous;
}
/**
 * Injects `root` tokens in limp mode.
 *
 * If no injector exists, we can still inject tree-shakable providers which have `providedIn` set to
 * `"root"`. This is known as the limp mode injection. In such case the value is stored in the
 * injectable definition.
 */
export function injectRootLimpMode(token, notFoundValue, flags) {
    const injectableDef = getInjectableDef(token);
    if (injectableDef && injectableDef.providedIn == 'root') {
        return injectableDef.value === undefined
            ? (injectableDef.value = injectableDef.factory())
            : injectableDef.value;
    }
    if (flags & InjectFlags.Optional)
        return null;
    if (notFoundValue !== undefined)
        return notFoundValue;
    throwProviderNotFoundError(token, 'Injector');
}
/**
 * Assert that `_injectImplementation` is not `fn`.
 *
 * This is useful, to prevent infinite recursion.
 *
 * @param fn Function which it should not equal to
 */
export function assertInjectImplementationNotEqual(fn) {
    ngDevMode &&
        assertNotEqual(_injectImplementation, fn, 'Calling ɵɵinject would cause infinite recursion');
}
//# sourceMappingURL=data:application/json;base64,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