/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { ERROR_DETAILS_PAGE_BASE_URL } from './error_details_base_url';
/**
 * Class that represents a runtime error.
 * Formats and outputs the error message in a consistent way.
 *
 * Example:
 * ```
 *  throw new RuntimeError(
 *    RuntimeErrorCode.INJECTOR_ALREADY_DESTROYED,
 *    ngDevMode && 'Injector has already been destroyed.');
 * ```
 *
 * Note: the `message` argument contains a descriptive error message as a string in development
 * mode (when the `ngDevMode` is defined). In production mode (after tree-shaking pass), the
 * `message` argument becomes `false`, thus we account for it in the typings and the runtime
 * logic.
 */
export class RuntimeError extends Error {
    constructor(code, message) {
        super(formatRuntimeError(code, message));
        this.code = code;
    }
}
/**
 * Called to format a runtime error.
 * See additional info on the `message` argument type in the `RuntimeError` class description.
 */
export function formatRuntimeError(code, message) {
    // Error code might be a negative number, which is a special marker that instructs the logic to
    // generate a link to the error details page on angular.io.
    // We also prepend `0` to non-compile-time errors.
    const fullCode = `NG0${Math.abs(code)}`;
    let errorMessage = `${fullCode}${message ? ': ' + message : ''}`;
    if (ngDevMode && code < 0) {
        const addPeriodSeparator = !errorMessage.match(/[.,;!?\n]$/);
        const separator = addPeriodSeparator ? '.' : '';
        errorMessage = `${errorMessage}${separator} Find more at ${ERROR_DETAILS_PAGE_BASE_URL}/${fullCode}`;
    }
    return errorMessage;
}
//# sourceMappingURL=data:application/json;base64,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