/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { stringify } from '../util/stringify';
/**
 * Map of module-id to the corresponding NgModule.
 */
const modules = new Map();
/**
 * Whether to check for duplicate NgModule registrations.
 *
 * This can be disabled for testing.
 */
let checkForDuplicateNgModules = true;
function assertSameOrNotExisting(id, type, incoming) {
    if (type && type !== incoming && checkForDuplicateNgModules) {
        throw new Error(`Duplicate module registered for ${id} - ${stringify(type)} vs ${stringify(type.name)}`);
    }
}
/**
 * Adds the given NgModule type to Angular's NgModule registry.
 *
 * This is generated as a side-effect of NgModule compilation. Note that the `id` is passed in
 * explicitly and not read from the NgModule definition. This is for two reasons: it avoids a
 * megamorphic read, and in JIT there's a chicken-and-egg problem where the NgModule may not be
 * fully resolved when it's registered.
 *
 * @codeGenApi
 */
export function registerNgModuleType(ngModuleType, id) {
    const existing = modules.get(id) || null;
    assertSameOrNotExisting(id, existing, ngModuleType);
    modules.set(id, ngModuleType);
}
export function clearModulesForTest() {
    modules.clear();
}
export function getRegisteredNgModuleType(id) {
    return modules.get(id);
}
/**
 * Control whether the NgModule registration system enforces that each NgModule type registered has
 * a unique id.
 *
 * This is useful for testing as the NgModule registry cannot be properly reset between tests with
 * Angular's current API.
 */
export function setAllowDuplicateNgModuleIdsForTest(allowDuplicates) {
    checkForDuplicateNgModules = !allowDuplicates;
}
//# sourceMappingURL=data:application/json;base64,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