/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { getCurrentTNode, getLView } from '../render3/state';
import { createAndRenderEmbeddedLView } from '../render3/view_manipulation';
import { ViewRef as R3_ViewRef } from '../render3/view_ref';
import { assertDefined } from '../util/assert';
import { createElementRef } from './element_ref';
/**
 * Represents an embedded template that can be used to instantiate embedded views.
 * To instantiate embedded views based on a template, use the `ViewContainerRef`
 * method `createEmbeddedView()`.
 *
 * Access a `TemplateRef` instance by placing a directive on an `<ng-template>`
 * element (or directive prefixed with `*`). The `TemplateRef` for the embedded view
 * is injected into the constructor of the directive,
 * using the `TemplateRef` token.
 *
 * You can also use a `Query` to find a `TemplateRef` associated with
 * a component or a directive.
 *
 * @see {@link ViewContainerRef}
 *
 * @publicApi
 */
export class TemplateRef {
    /**
     * @internal
     * @nocollapse
     */
    static { this.__NG_ELEMENT_ID__ = injectTemplateRef; }
}
const ViewEngineTemplateRef = TemplateRef;
// TODO(alxhub): combine interface and implementation. Currently this is challenging since something
// in g3 depends on them being separate.
const R3TemplateRef = class TemplateRef extends ViewEngineTemplateRef {
    constructor(_declarationLView, _declarationTContainer, elementRef) {
        super();
        this._declarationLView = _declarationLView;
        this._declarationTContainer = _declarationTContainer;
        this.elementRef = elementRef;
    }
    /**
     * Returns an `ssrId` associated with a TView, which was used to
     * create this instance of the `TemplateRef`.
     *
     * @internal
     */
    get ssrId() {
        return this._declarationTContainer.tView?.ssrId || null;
    }
    createEmbeddedView(context, injector) {
        return this.createEmbeddedViewImpl(context, injector);
    }
    /**
     * @internal
     */
    createEmbeddedViewImpl(context, injector, dehydratedView) {
        const embeddedLView = createAndRenderEmbeddedLView(this._declarationLView, this._declarationTContainer, context, { embeddedViewInjector: injector, dehydratedView });
        return new R3_ViewRef(embeddedLView);
    }
};
/**
 * Creates a TemplateRef given a node.
 *
 * @returns The TemplateRef instance to use
 */
export function injectTemplateRef() {
    return createTemplateRef(getCurrentTNode(), getLView());
}
/**
 * Creates a TemplateRef and stores it on the injector.
 *
 * @param hostTNode The node on which a TemplateRef is requested
 * @param hostLView The `LView` to which the node belongs
 * @returns The TemplateRef instance or null if we can't create a TemplateRef on a given node type
 */
export function createTemplateRef(hostTNode, hostLView) {
    if (hostTNode.type & 4 /* TNodeType.Container */) {
        ngDevMode && assertDefined(hostTNode.tView, 'TView must be allocated');
        return new R3TemplateRef(hostLView, hostTNode, createElementRef(hostTNode, hostLView));
    }
    return null;
}
//# sourceMappingURL=data:application/json;base64,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