/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { compileComponent, compileDirective } from '../render3/jit/directive';
import { compilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export const Directive = makeDecorator('Directive', (dir = {}) => dir, undefined, undefined, (type, meta) => compileDirective(type, meta));
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const Component = makeDecorator('Component', (c = {}) => ({ changeDetection: ChangeDetectionStrategy.Default, ...c }), Directive, undefined, (type, meta) => compileComponent(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Pipe = makeDecorator('Pipe', (p) => ({ pure: true, ...p }), undefined, undefined, (type, meta) => compilePipe(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Input = makePropDecorator('Input', (arg) => {
    if (!arg) {
        return {};
    }
    return typeof arg === 'string' ? { alias: arg } : arg;
});
/**
 * @Annotation
 * @publicApi
 */
export const Output = makePropDecorator('Output', (alias) => ({ alias }));
/**
 * @Annotation
 * @publicApi
 */
export const HostBinding = makePropDecorator('HostBinding', (hostPropertyName) => ({ hostPropertyName }));
/**
 * @Annotation
 * @publicApi
 */
export const HostListener = makePropDecorator('HostListener', (eventName, args) => ({ eventName, args }));
//# sourceMappingURL=data:application/json;base64,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