/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { assertIndexInRange, assertLessThan, assertNotSame } from '../util/assert';
import { devModeEqual } from '../util/comparison';
import { getExpressionChangedErrorDetails, throwErrorIfNoChangesMode } from './errors';
import { isInCheckNoChangesMode } from './state';
import { NO_CHANGE } from './tokens';
// TODO(misko): consider inlining
/** Updates binding and returns the value. */
export function updateBinding(lView, bindingIndex, value) {
    return (lView[bindingIndex] = value);
}
/** Gets the current binding value. */
export function getBinding(lView, bindingIndex) {
    ngDevMode && assertIndexInRange(lView, bindingIndex);
    ngDevMode &&
        assertNotSame(lView[bindingIndex], NO_CHANGE, 'Stored value should never be NO_CHANGE.');
    return lView[bindingIndex];
}
/**
 * Updates binding if changed, then returns whether it was updated.
 *
 * This function also checks the `CheckNoChangesMode` and throws if changes are made.
 * Some changes (Objects/iterables) during `CheckNoChangesMode` are exempt to comply with VE
 * behavior.
 *
 * @param lView current `LView`
 * @param bindingIndex The binding in the `LView` to check
 * @param value New value to check against `lView[bindingIndex]`
 * @returns `true` if the bindings has changed. (Throws if binding has changed during
 *          `CheckNoChangesMode`)
 */
export function bindingUpdated(lView, bindingIndex, value) {
    ngDevMode && assertNotSame(value, NO_CHANGE, 'Incoming value should never be NO_CHANGE.');
    ngDevMode &&
        assertLessThan(bindingIndex, lView.length, `Slot should have been initialized to NO_CHANGE`);
    const oldValue = lView[bindingIndex];
    if (Object.is(oldValue, value)) {
        return false;
    }
    else {
        if (ngDevMode && isInCheckNoChangesMode()) {
            // View engine didn't report undefined values as changed on the first checkNoChanges pass
            // (before the change detection was run).
            const oldValueToCompare = oldValue !== NO_CHANGE ? oldValue : undefined;
            if (!devModeEqual(oldValueToCompare, value)) {
                const details = getExpressionChangedErrorDetails(lView, bindingIndex, oldValueToCompare, value);
                throwErrorIfNoChangesMode(oldValue === NO_CHANGE, details.oldValue, details.newValue, details.propName, lView);
            }
            // There was a change, but the `devModeEqual` decided that the change is exempt from an error.
            // For this reason we exit as if no change. The early exit is needed to prevent the changed
            // value to be written into `LView` (If we would write the new value that we would not see it
            // as change on next CD.)
            return false;
        }
        lView[bindingIndex] = value;
        return true;
    }
}
/** Updates 2 bindings if changed, then returns whether either was updated. */
export function bindingUpdated2(lView, bindingIndex, exp1, exp2) {
    const different = bindingUpdated(lView, bindingIndex, exp1);
    return bindingUpdated(lView, bindingIndex + 1, exp2) || different;
}
/** Updates 3 bindings if changed, then returns whether any was updated. */
export function bindingUpdated3(lView, bindingIndex, exp1, exp2, exp3) {
    const different = bindingUpdated2(lView, bindingIndex, exp1, exp2);
    return bindingUpdated(lView, bindingIndex + 2, exp3) || different;
}
/** Updates 4 bindings if changed, then returns whether any was updated. */
export function bindingUpdated4(lView, bindingIndex, exp1, exp2, exp3, exp4) {
    const different = bindingUpdated2(lView, bindingIndex, exp1, exp2);
    return bindingUpdated2(lView, bindingIndex + 2, exp3, exp4) || different;
}
//# sourceMappingURL=data:application/json;base64,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