/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { EnvironmentInjector } from '../../di/r3_injector';
import { assertDefined, throwError } from '../../util/assert';
import { assertTNodeForLView } from '../assert';
import { getComponentDef } from '../definition';
import { getNodeInjectorLView, getNodeInjectorTNode, NodeInjector } from '../di';
import { setInjectorProfiler, } from './injector_profiler';
/**
 * These are the data structures that our framework injector profiler will fill with data in order
 * to support DI debugging APIs.
 *
 * resolverToTokenToDependencies: Maps an injector to a Map of tokens to an Array of
 * dependencies. Injector -> Token -> Dependencies This is used to support the
 * getDependenciesFromInjectable API, which takes in an injector and a token and returns it's
 * dependencies.
 *
 * resolverToProviders: Maps a DI resolver (an Injector or a TNode) to the providers configured
 * within it This is used to support the getInjectorProviders API, which takes in an injector and
 * returns the providers that it was configured with. Note that for the element injector case we
 * use the TNode instead of the LView as the DI resolver. This is because the registration of
 * providers happens only once per type of TNode. If an injector is created with an identical TNode,
 * the providers for that injector will not be reconfigured.
 *
 * standaloneInjectorToComponent: Maps the injector of a standalone component to the standalone
 * component that it is associated with. Used in the getInjectorProviders API, specificially in the
 * discovery of import paths for each provider. This is necessary because the imports array of a
 * standalone component is processed and configured in its standalone injector, but exists within
 * the component's definition. Because getInjectorProviders takes in an injector, if that injector
 * is the injector of a standalone component, we need to be able to discover the place where the
 * imports array is located (the component) in order to flatten the imports array within it to
 * discover all of it's providers.
 *
 *
 * All of these data structures are instantiated with WeakMaps. This will ensure that the presence
 * of any object in the keys of these maps does not prevent the garbage collector from collecting
 * those objects. Because of this property of WeakMaps, these data structures will never be the
 * source of a memory leak.
 *
 * An example of this advantage: When components are destroyed, we don't need to do
 * any additional work to remove that component from our mappings.
 *
 */
class DIDebugData {
    constructor() {
        this.resolverToTokenToDependencies = new WeakMap();
        this.resolverToProviders = new WeakMap();
        this.standaloneInjectorToComponent = new WeakMap();
    }
    reset() {
        this.resolverToTokenToDependencies = new WeakMap();
        this.resolverToProviders = new WeakMap();
        this.standaloneInjectorToComponent = new WeakMap();
    }
}
let frameworkDIDebugData = new DIDebugData();
export function getFrameworkDIDebugData() {
    return frameworkDIDebugData;
}
/**
 * Initalize default handling of injector events. This handling parses events
 * as they are emitted and constructs the data structures necessary to support
 * some of debug APIs.
 *
 * See handleInjectEvent, handleCreateEvent and handleProviderConfiguredEvent
 * for descriptions of each handler
 *
 * Supported APIs:
 *               - getDependenciesFromInjectable
 *               - getInjectorProviders
 */
export function setupFrameworkInjectorProfiler() {
    frameworkDIDebugData.reset();
    setInjectorProfiler((injectorProfilerEvent) => handleInjectorProfilerEvent(injectorProfilerEvent));
}
function handleInjectorProfilerEvent(injectorProfilerEvent) {
    const { context, type } = injectorProfilerEvent;
    if (type === 0 /* InjectorProfilerEventType.Inject */) {
        handleInjectEvent(context, injectorProfilerEvent.service);
    }
    else if (type === 1 /* InjectorProfilerEventType.InstanceCreatedByInjector */) {
        handleInstanceCreatedByInjectorEvent(context, injectorProfilerEvent.instance);
    }
    else if (type === 2 /* InjectorProfilerEventType.ProviderConfigured */) {
        handleProviderConfiguredEvent(context, injectorProfilerEvent.providerRecord);
    }
}
/**
 *
 * Stores the injected service in frameworkDIDebugData.resolverToTokenToDependencies
 * based on it's injector and token.
 *
 * @param context InjectorProfilerContext the injection context that this event occurred in.
 * @param data InjectedService the service associated with this inject event.
 *
 */
function handleInjectEvent(context, data) {
    const diResolver = getDIResolver(context.injector);
    if (diResolver === null) {
        throwError('An Inject event must be run within an injection context.');
    }
    const diResolverToInstantiatedToken = frameworkDIDebugData.resolverToTokenToDependencies;
    if (!diResolverToInstantiatedToken.has(diResolver)) {
        diResolverToInstantiatedToken.set(diResolver, new WeakMap());
    }
    // if token is a primitive type, ignore this event. We do this because we cannot keep track of
    // non-primitive tokens in WeakMaps since they are not garbage collectable.
    if (!canBeHeldWeakly(context.token)) {
        return;
    }
    const instantiatedTokenToDependencies = diResolverToInstantiatedToken.get(diResolver);
    if (!instantiatedTokenToDependencies.has(context.token)) {
        instantiatedTokenToDependencies.set(context.token, []);
    }
    const { token, value, flags } = data;
    assertDefined(context.token, 'Injector profiler context token is undefined.');
    const dependencies = instantiatedTokenToDependencies.get(context.token);
    assertDefined(dependencies, 'Could not resolve dependencies for token.');
    if (context.injector instanceof NodeInjector) {
        dependencies.push({ token, value, flags, injectedIn: getNodeInjectorContext(context.injector) });
    }
    else {
        dependencies.push({ token, value, flags });
    }
}
/**
 *
 * Returns the LView and TNode associated with a NodeInjector. Returns undefined if the injector
 * is not a NodeInjector.
 *
 * @param injector
 * @returns {lView: LView, tNode: TNode}|undefined
 */
function getNodeInjectorContext(injector) {
    if (!(injector instanceof NodeInjector)) {
        throwError('getNodeInjectorContext must be called with a NodeInjector');
    }
    const lView = getNodeInjectorLView(injector);
    const tNode = getNodeInjectorTNode(injector);
    if (tNode === null) {
        return;
    }
    assertTNodeForLView(tNode, lView);
    return { lView, tNode };
}
/**
 *
 * If the created instance is an instance of a standalone component, maps the injector to that
 * standalone component in frameworkDIDebugData.standaloneInjectorToComponent
 *
 * @param context InjectorProfilerContext the injection context that this event occurred in.
 * @param data InjectorCreatedInstance an object containing the instance that was just created
 *
 */
function handleInstanceCreatedByInjectorEvent(context, data) {
    const { value } = data;
    if (getDIResolver(context.injector) === null) {
        throwError('An InjectorCreatedInstance event must be run within an injection context.');
    }
    // if our value is an instance of a standalone component, map the injector of that standalone
    // component to the component class. Otherwise, this event is a noop.
    let standaloneComponent = undefined;
    if (typeof value === 'object') {
        standaloneComponent = value?.constructor;
    }
    // We want to also cover if `standaloneComponent === null` in addition to `undefined`
    if (standaloneComponent == undefined || !isStandaloneComponent(standaloneComponent)) {
        return;
    }
    const environmentInjector = context.injector.get(EnvironmentInjector, null, { optional: true });
    // Standalone components should have an environment injector. If one cannot be
    // found we may be in a test case for low level functionality that did not explicitly
    // setup this injector. In those cases, we simply ignore this event.
    if (environmentInjector === null) {
        return;
    }
    const { standaloneInjectorToComponent } = frameworkDIDebugData;
    // If our injector has already been mapped, as is the case
    // when a standalone component imports another standalone component,
    // we consider the original component (the component doing the importing)
    // as the component connected to our injector.
    if (standaloneInjectorToComponent.has(environmentInjector)) {
        return;
    }
    // If our injector hasn't been mapped, then we map it to the standalone component
    standaloneInjectorToComponent.set(environmentInjector, standaloneComponent);
}
function isStandaloneComponent(value) {
    const def = getComponentDef(value);
    return !!def?.standalone;
}
/**
 *
 * Stores the emitted ProviderRecords from the InjectorProfilerEventType.ProviderConfigured
 * event in frameworkDIDebugData.resolverToProviders
 *
 * @param context InjectorProfilerContext the injection context that this event occurred in.
 * @param data ProviderRecord an object containing the instance that was just created
 *
 */
function handleProviderConfiguredEvent(context, data) {
    const { resolverToProviders } = frameworkDIDebugData;
    let diResolver;
    if (context?.injector instanceof NodeInjector) {
        diResolver = getNodeInjectorTNode(context.injector);
    }
    else {
        diResolver = context.injector;
    }
    if (diResolver === null) {
        throwError('A ProviderConfigured event must be run within an injection context.');
    }
    if (!resolverToProviders.has(diResolver)) {
        resolverToProviders.set(diResolver, []);
    }
    resolverToProviders.get(diResolver).push(data);
}
function getDIResolver(injector) {
    let diResolver = null;
    if (injector === undefined) {
        return diResolver;
    }
    // We use the LView as the diResolver for NodeInjectors because they
    // do not persist anywhere in the framework. They are simply wrappers around an LView and a TNode
    // that do persist. Because of this, we rely on the LView of the NodeInjector in order to use
    // as a concrete key to represent this injector. If we get the same LView back later, we know
    // we're looking at the same injector.
    if (injector instanceof NodeInjector) {
        diResolver = getNodeInjectorLView(injector);
    }
    // Other injectors can be used a keys for a map because their instances
    // persist
    else {
        diResolver = injector;
    }
    return diResolver;
}
// inspired by
// https://tc39.es/ecma262/multipage/executable-code-and-execution-contexts.html#sec-canbeheldweakly
function canBeHeldWeakly(value) {
    // we check for value !== null here because typeof null === 'object
    return (value !== null &&
        (typeof value === 'object' || typeof value === 'function' || typeof value === 'symbol'));
}
//# sourceMappingURL=data:application/json;base64,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