/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { resolveForwardRef } from '../../di/forward_ref';
import { InjectionToken } from '../../di/injection_token';
import { throwError } from '../../util/assert';
let _injectorProfilerContext;
export function getInjectorProfilerContext() {
    !ngDevMode && throwError('getInjectorProfilerContext should never be called in production mode');
    return _injectorProfilerContext;
}
export function setInjectorProfilerContext(context) {
    !ngDevMode && throwError('setInjectorProfilerContext should never be called in production mode');
    const previous = _injectorProfilerContext;
    _injectorProfilerContext = context;
    return previous;
}
let injectorProfilerCallback = null;
/**
 * Sets the callback function which will be invoked during certain DI events within the
 * runtime (for example: injecting services, creating injectable instances, configuring providers)
 *
 * Warning: this function is *INTERNAL* and should not be relied upon in application's code.
 * The contract of the function might be changed in any release and/or the function can be removed
 * completely.
 *
 * @param profiler function provided by the caller or null value to disable profiling.
 */
export const setInjectorProfiler = (injectorProfiler) => {
    !ngDevMode && throwError('setInjectorProfiler should never be called in production mode');
    injectorProfilerCallback = injectorProfiler;
};
/**
 * Injector profiler function which emits on DI events executed by the runtime.
 *
 * @param event InjectorProfilerEvent corresponding to the DI event being emitted
 */
function injectorProfiler(event) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    if (injectorProfilerCallback != null /* both `null` and `undefined` */) {
        injectorProfilerCallback(event);
    }
}
/**
 * Emits an InjectorProfilerEventType.ProviderConfigured to the injector profiler. The data in the
 * emitted event includes the raw provider, as well as the token that provider is providing.
 *
 * @param eventProvider A provider object
 */
export function emitProviderConfiguredEvent(eventProvider, isViewProvider = false) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    let token;
    // if the provider is a TypeProvider (typeof provider is function) then the token is the
    // provider itself
    if (typeof eventProvider === 'function') {
        token = eventProvider;
    }
    // if the provider is an injection token, then the token is the injection token.
    else if (eventProvider instanceof InjectionToken) {
        token = eventProvider;
    }
    // in all other cases we can access the token via the `provide` property of the provider
    else {
        token = resolveForwardRef(eventProvider.provide);
    }
    let provider = eventProvider;
    // Injection tokens may define their own default provider which gets attached to the token itself
    // as `ɵprov`. In this case, we want to emit the provider that is attached to the token, not the
    // token itself.
    if (eventProvider instanceof InjectionToken) {
        provider = eventProvider.ɵprov || eventProvider;
    }
    injectorProfiler({
        type: 2 /* InjectorProfilerEventType.ProviderConfigured */,
        context: getInjectorProfilerContext(),
        providerRecord: { token, provider, isViewProvider },
    });
}
/**
 * Emits an event to the injector profiler with the instance that was created. Note that
 * the injector associated with this emission can be accessed by using getDebugInjectContext()
 *
 * @param instance an object created by an injector
 */
export function emitInstanceCreatedByInjectorEvent(instance) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    injectorProfiler({
        type: 1 /* InjectorProfilerEventType.InstanceCreatedByInjector */,
        context: getInjectorProfilerContext(),
        instance: { value: instance },
    });
}
/**
 * @param token DI token associated with injected service
 * @param value the instance of the injected service (i.e the result of `inject(token)`)
 * @param flags the flags that the token was injected with
 */
export function emitInjectEvent(token, value, flags) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    injectorProfiler({
        type: 0 /* InjectorProfilerEventType.Inject */,
        context: getInjectorProfilerContext(),
        service: { token, value, flags },
    });
}
export function runInInjectorProfilerContext(injector, token, callback) {
    !ngDevMode &&
        throwError('runInInjectorProfilerContext should never be called in production mode');
    const prevInjectContext = setInjectorProfilerContext({ injector, token });
    try {
        callback();
    }
    finally {
        setInjectorProfilerContext(prevInjectContext);
    }
}
//# sourceMappingURL=data:application/json;base64,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