/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { formatRuntimeError } from '../errors';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY, EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { NG_COMP_DEF, NG_DIR_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
import { InputFlags } from './interfaces/input_flags';
import { stringifyCSSSelectorList } from './node_selector_matcher';
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyComponent {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const baseDef = getNgDirectiveDef(componentDefinition);
        const def = {
            ...baseDef,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            template: componentDefinition.template,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null, // assigned in noSideEffects
            pipeDefs: null, // assigned in noSideEffects
            dependencies: (baseDef.standalone && componentDefinition.dependencies) || null,
            getStandaloneInjector: null,
            signals: componentDefinition.signals ?? false,
            data: componentDefinition.data || {},
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
            id: '',
        };
        initFeatures(def);
        const dependencies = componentDefinition.dependencies;
        def.directiveDefs = extractDefListOrFactory(dependencies, /* pipeDef */ false);
        def.pipeDefs = extractDefListOrFactory(dependencies, /* pipeDef */ true);
        def.id = getComponentId(def);
        return def;
    });
}
export function extractDirectiveDef(type) {
    return getComponentDef(type) || getDirectiveDef(type);
}
function nonNull(value) {
    return value !== null;
}
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    return noSideEffects(() => {
        const res = {
            type: def.type,
            bootstrap: def.bootstrap || EMPTY_ARRAY,
            declarations: def.declarations || EMPTY_ARRAY,
            imports: def.imports || EMPTY_ARRAY,
            exports: def.exports || EMPTY_ARRAY,
            transitiveCompileScopes: null,
            schemas: def.schemas || null,
            id: def.id || null,
        };
        return res;
    });
}
function parseAndConvertBindingsForDefinition(obj, declaredInputs) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            const value = obj[minifiedKey];
            let publicName;
            let declaredName;
            let inputFlags = InputFlags.None;
            if (Array.isArray(value)) {
                inputFlags = value[0];
                publicName = value[1];
                declaredName = value[2] ?? publicName; // declared name might not be set to save bytes.
            }
            else {
                publicName = value;
                declaredName = value;
            }
            // For inputs, capture the declared name, or if some flags are set.
            if (declaredInputs) {
                // Perf note: An array is only allocated for the input if there are flags.
                newLookup[publicName] =
                    inputFlags !== InputFlags.None ? [minifiedKey, inputFlags] : minifiedKey;
                declaredInputs[publicName] = declaredName;
            }
            else {
                newLookup[publicName] = minifiedKey;
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export function ɵɵdefineDirective(directiveDefinition) {
    return noSideEffects(() => {
        const def = getNgDirectiveDef(directiveDefinition);
        initFeatures(def);
        return def;
    });
}
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        standalone: pipeDef.standalone === true,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null,
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
/**
 * Checks whether a given Component, Directive or Pipe is marked as standalone.
 * This will return false if passed anything other than a Component, Directive, or Pipe class
 * See [this guide](guide/components/importing) for additional information:
 *
 * @param type A reference to a Component, Directive or Pipe.
 * @publicApi
 */
export function isStandalone(type) {
    const def = getComponentDef(type) || getDirectiveDef(type) || getPipeDef(type);
    return def !== null ? def.standalone : false;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
function getNgDirectiveDef(directiveDefinition) {
    const declaredInputs = {};
    return {
        type: directiveDefinition.type,
        providersResolver: null,
        factory: null,
        hostBindings: directiveDefinition.hostBindings || null,
        hostVars: directiveDefinition.hostVars || 0,
        hostAttrs: directiveDefinition.hostAttrs || null,
        contentQueries: directiveDefinition.contentQueries || null,
        declaredInputs: declaredInputs,
        inputTransforms: null,
        inputConfig: directiveDefinition.inputs || EMPTY_OBJ,
        exportAs: directiveDefinition.exportAs || null,
        standalone: directiveDefinition.standalone === true,
        signals: directiveDefinition.signals === true,
        selectors: directiveDefinition.selectors || EMPTY_ARRAY,
        viewQuery: directiveDefinition.viewQuery || null,
        features: directiveDefinition.features || null,
        setInput: null,
        findHostDirectiveDefs: null,
        hostDirectives: null,
        inputs: parseAndConvertBindingsForDefinition(directiveDefinition.inputs, declaredInputs),
        outputs: parseAndConvertBindingsForDefinition(directiveDefinition.outputs),
        debugInfo: null,
    };
}
function initFeatures(definition) {
    definition.features?.forEach((fn) => fn(definition));
}
export function extractDefListOrFactory(dependencies, pipeDef) {
    if (!dependencies) {
        return null;
    }
    const defExtractor = pipeDef ? getPipeDef : extractDirectiveDef;
    return () => (typeof dependencies === 'function' ? dependencies() : dependencies)
        .map((dep) => defExtractor(dep))
        .filter(nonNull);
}
/**
 * A map that contains the generated component IDs and type.
 */
export const GENERATED_COMP_IDS = new Map();
/**
 * A method can returns a component ID from the component definition using a variant of DJB2 hash
 * algorithm.
 */
function getComponentId(componentDef) {
    let hash = 0;
    // We cannot rely solely on the component selector as the same selector can be used in different
    // modules.
    //
    // `componentDef.style` is not used, due to it causing inconsistencies. Ex: when server
    // component styles has no sourcemaps and browsers do.
    //
    // Example:
    // https://github.com/angular/components/blob/d9f82c8f95309e77a6d82fd574c65871e91354c2/src/material/core/option/option.ts#L248
    // https://github.com/angular/components/blob/285f46dc2b4c5b127d356cb7c4714b221f03ce50/src/material/legacy-core/option/option.ts#L32
    const hashSelectors = [
        componentDef.selectors,
        componentDef.ngContentSelectors,
        componentDef.hostVars,
        componentDef.hostAttrs,
        componentDef.consts,
        componentDef.vars,
        componentDef.decls,
        componentDef.encapsulation,
        componentDef.standalone,
        componentDef.signals,
        componentDef.exportAs,
        JSON.stringify(componentDef.inputs),
        JSON.stringify(componentDef.outputs),
        // We cannot use 'componentDef.type.name' as the name of the symbol will change and will not
        // match in the server and browser bundles.
        Object.getOwnPropertyNames(componentDef.type.prototype),
        !!componentDef.contentQueries,
        !!componentDef.viewQuery,
    ].join('|');
    for (const char of hashSelectors) {
        hash = (Math.imul(31, hash) + char.charCodeAt(0)) << 0;
    }
    // Force positive number hash.
    // 2147483647 = equivalent of Integer.MAX_VALUE.
    hash += 2147483647 + 1;
    const compId = 'c' + hash;
    if (typeof ngDevMode === 'undefined' || ngDevMode) {
        if (GENERATED_COMP_IDS.has(compId)) {
            const previousCompDefType = GENERATED_COMP_IDS.get(compId);
            if (previousCompDefType !== componentDef.type) {
                console.warn(formatRuntimeError(-912 /* RuntimeErrorCode.COMPONENT_ID_COLLISION */, `Component ID generation collision detected. Components '${previousCompDefType.name}' and '${componentDef.type.name}' with selector '${stringifyCSSSelectorList(componentDef.selectors)}' generated the same component ID. To fix this, you can change the selector of one of those components or add an extra host attribute to force a different ID.`));
            }
        }
        else {
            GENERATED_COMP_IDS.set(compId, componentDef.type);
        }
    }
    return compId;
}
//# sourceMappingURL=data:application/json;base64,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