/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { isEnvironmentProviders } from '../di/interface/provider';
import { RuntimeError } from '../errors';
import { stringify } from '../util/stringify';
import { stringifyForError } from './util/stringify_utils';
/** Called when directives inject each other (creating a circular dependency) */
export function throwCyclicDependencyError(token, path) {
    const depPath = path ? `. Dependency path: ${path.join(' > ')} > ${token}` : '';
    throw new RuntimeError(-200 /* RuntimeErrorCode.CYCLIC_DI_DEPENDENCY */, ngDevMode ? `Circular dependency in DI detected for ${token}${depPath}` : token);
}
export function throwMixedMultiProviderError() {
    throw new Error(`Cannot mix multi providers and regular providers`);
}
export function throwInvalidProviderError(ngModuleType, providers, provider) {
    if (ngModuleType && providers) {
        const providerDetail = providers.map((v) => (v == provider ? '?' + provider + '?' : '...'));
        throw new Error(`Invalid provider for the NgModule '${stringify(ngModuleType)}' - only instances of Provider and Type are allowed, got: [${providerDetail.join(', ')}]`);
    }
    else if (isEnvironmentProviders(provider)) {
        if (provider.ɵfromNgModule) {
            throw new RuntimeError(207 /* RuntimeErrorCode.PROVIDER_IN_WRONG_CONTEXT */, `Invalid providers from 'importProvidersFrom' present in a non-environment injector. 'importProvidersFrom' can't be used for component providers.`);
        }
        else {
            throw new RuntimeError(207 /* RuntimeErrorCode.PROVIDER_IN_WRONG_CONTEXT */, `Invalid providers present in a non-environment injector. 'EnvironmentProviders' can't be used for component providers.`);
        }
    }
    else {
        throw new Error('Invalid provider');
    }
}
/** Throws an error when a token is not found in DI. */
export function throwProviderNotFoundError(token, injectorName) {
    const errorMessage = ngDevMode &&
        `No provider for ${stringifyForError(token)} found${injectorName ? ` in ${injectorName}` : ''}`;
    throw new RuntimeError(-201 /* RuntimeErrorCode.PROVIDER_NOT_FOUND */, errorMessage);
}
//# sourceMappingURL=data:application/json;base64,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