/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { assertDomNode, assertNumber, assertNumberInRange } from '../../util/assert';
import { EMPTY_ARRAY } from '../../util/empty';
import { assertTIcu, assertTNodeForLView } from '../assert';
import { getCurrentICUCaseIndex } from '../i18n/i18n_util';
import { TVIEW } from '../interfaces/view';
function enterIcu(state, tIcu, lView) {
    state.index = 0;
    const currentCase = getCurrentICUCaseIndex(tIcu, lView);
    if (currentCase !== null) {
        ngDevMode && assertNumberInRange(currentCase, 0, tIcu.cases.length - 1);
        state.removes = tIcu.remove[currentCase];
    }
    else {
        state.removes = EMPTY_ARRAY;
    }
}
function icuContainerIteratorNext(state) {
    if (state.index < state.removes.length) {
        const removeOpCode = state.removes[state.index++];
        ngDevMode && assertNumber(removeOpCode, 'Expecting OpCode number');
        if (removeOpCode > 0) {
            const rNode = state.lView[removeOpCode];
            ngDevMode && assertDomNode(rNode);
            return rNode;
        }
        else {
            state.stack.push(state.index, state.removes);
            // ICUs are represented by negative indices
            const tIcuIndex = ~removeOpCode;
            const tIcu = state.lView[TVIEW].data[tIcuIndex];
            ngDevMode && assertTIcu(tIcu);
            enterIcu(state, tIcu, state.lView);
            return icuContainerIteratorNext(state);
        }
    }
    else {
        if (state.stack.length === 0) {
            return null;
        }
        else {
            state.removes = state.stack.pop();
            state.index = state.stack.pop();
            return icuContainerIteratorNext(state);
        }
    }
}
export function loadIcuContainerVisitor() {
    const _state = {
        stack: [],
        index: -1,
    };
    /**
     * Retrieves a set of root nodes from `TIcu.remove`. Used by `TNodeType.ICUContainer`
     * to determine which root belong to the ICU.
     *
     * Example of usage.
     * ```
     * const nextRNode = icuContainerIteratorStart(tIcuContainerNode, lView);
     * let rNode: RNode|null;
     * while(rNode = nextRNode()) {
     *   console.log(rNode);
     * }
     * ```
     *
     * @param tIcuContainerNode Current `TIcuContainerNode`
     * @param lView `LView` where the `RNode`s should be looked up.
     */
    function icuContainerIteratorStart(tIcuContainerNode, lView) {
        _state.lView = lView;
        while (_state.stack.length)
            _state.stack.pop();
        ngDevMode && assertTNodeForLView(tIcuContainerNode, lView);
        enterIcu(_state, tIcuContainerNode.value, lView);
        return icuContainerIteratorNext.bind(null, _state);
    }
    return icuContainerIteratorStart;
}
export function createIcuIterator(tIcu, lView) {
    const state = {
        stack: [],
        index: -1,
        lView,
    };
    ngDevMode && assertTIcu(tIcu);
    enterIcu(state, tIcu, lView);
    return icuContainerIteratorNext.bind(null, state);
}
//# sourceMappingURL=data:application/json;base64,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