/*!
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { RuntimeError } from '../../errors';
import { performanceMarkFeature } from '../../util/performance';
import { HEADER_OFFSET } from '../interfaces/view';
import { getContextLView, getLView, getSelectedIndex, getTView, setCurrentTNode } from '../state';
import { load } from '../util/view_utils';
import { getOrCreateTNode } from './shared';
import { store } from './storage';
/** Object that indicates the value of a `@let` declaration that hasn't been initialized yet. */
const UNINITIALIZED_LET = {};
/**
 * Declares an `@let` at a specific data slot. Returns itself to allow chaining.
 *
 * @param index Index at which to declare the `@let`.
 *
 * @codeGenApi
 */
export function ɵɵdeclareLet(index) {
    const tView = getTView();
    const lView = getLView();
    const adjustedIndex = index + HEADER_OFFSET;
    const tNode = getOrCreateTNode(tView, adjustedIndex, 128 /* TNodeType.LetDeclaration */, null, null);
    setCurrentTNode(tNode, false);
    store(tView, lView, adjustedIndex, UNINITIALIZED_LET);
    return ɵɵdeclareLet;
}
/**
 * Instruction that stores the value of a `@let` declaration on the current view.
 * Returns the value to allow usage inside variable initializers.
 *
 * @codeGenApi
 */
export function ɵɵstoreLet(value) {
    performanceMarkFeature('NgLet');
    const tView = getTView();
    const lView = getLView();
    const index = getSelectedIndex();
    store(tView, lView, index, value);
    return value;
}
/**
 * Retrieves the value of a `@let` declaration defined in a parent view.
 *
 * @param index Index of the declaration within the view.
 *
 * @codeGenApi
 */
export function ɵɵreadContextLet(index) {
    const contextLView = getContextLView();
    const value = load(contextLView, HEADER_OFFSET + index);
    if (value === UNINITIALIZED_LET) {
        throw new RuntimeError(314 /* RuntimeErrorCode.UNINITIALIZED_LET_ACCESS */, ngDevMode && 'Attempting to access a @let declaration whose value is not available yet');
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,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