/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { assertDefined, assertEqual } from '../../util/assert';
export const NO_PARENT_INJECTOR = -1;
/**
 * Each injector is saved in 9 contiguous slots in `LView` and 9 contiguous slots in
 * `TView.data`. This allows us to store information about the current node's tokens (which
 * can be shared in `TView`) as well as the tokens of its ancestor nodes (which cannot be
 * shared, so they live in `LView`).
 *
 * Each of these slots (aside from the last slot) contains a bloom filter. This bloom filter
 * determines whether a directive is available on the associated node or not. This prevents us
 * from searching the directives array at this level unless it's probable the directive is in it.
 *
 * See: https://en.wikipedia.org/wiki/Bloom_filter for more about bloom filters.
 *
 * Because all injectors have been flattened into `LView` and `TViewData`, they cannot typed
 * using interfaces as they were previously. The start index of each `LInjector` and `TInjector`
 * will differ based on where it is flattened into the main array, so it's not possible to know
 * the indices ahead of time and save their types here. The interfaces are still included here
 * for documentation purposes.
 *
 * export interface LInjector extends Array<any> {
 *
 *    // Cumulative bloom for directive IDs 0-31  (IDs are % BLOOM_SIZE)
 *    [0]: number;
 *
 *    // Cumulative bloom for directive IDs 32-63
 *    [1]: number;
 *
 *    // Cumulative bloom for directive IDs 64-95
 *    [2]: number;
 *
 *    // Cumulative bloom for directive IDs 96-127
 *    [3]: number;
 *
 *    // Cumulative bloom for directive IDs 128-159
 *    [4]: number;
 *
 *    // Cumulative bloom for directive IDs 160 - 191
 *    [5]: number;
 *
 *    // Cumulative bloom for directive IDs 192 - 223
 *    [6]: number;
 *
 *    // Cumulative bloom for directive IDs 224 - 255
 *    [7]: number;
 *
 *    // We need to store a reference to the injector's parent so DI can keep looking up
 *    // the injector tree until it finds the dependency it's looking for.
 *    [PARENT_INJECTOR]: number;
 * }
 *
 * export interface TInjector extends Array<any> {
 *
 *    // Shared node bloom for directive IDs 0-31  (IDs are % BLOOM_SIZE)
 *    [0]: number;
 *
 *    // Shared node bloom for directive IDs 32-63
 *    [1]: number;
 *
 *    // Shared node bloom for directive IDs 64-95
 *    [2]: number;
 *
 *    // Shared node bloom for directive IDs 96-127
 *    [3]: number;
 *
 *    // Shared node bloom for directive IDs 128-159
 *    [4]: number;
 *
 *    // Shared node bloom for directive IDs 160 - 191
 *    [5]: number;
 *
 *    // Shared node bloom for directive IDs 192 - 223
 *    [6]: number;
 *
 *    // Shared node bloom for directive IDs 224 - 255
 *    [7]: number;
 *
 *    // Necessary to find directive indices for a particular node.
 *    [TNODE]: TElementNode|TElementContainerNode|TContainerNode;
 *  }
 */
/**
 * Factory for creating instances of injectors in the NodeInjector.
 *
 * This factory is complicated by the fact that it can resolve `multi` factories as well.
 *
 * NOTE: Some of the fields are optional which means that this class has two hidden classes.
 * - One without `multi` support (most common)
 * - One with `multi` values, (rare).
 *
 * Since VMs can cache up to 4 inline hidden classes this is OK.
 *
 * - Single factory: Only `resolving` and `factory` is defined.
 * - `providers` factory: `componentProviders` is a number and `index = -1`.
 * - `viewProviders` factory: `componentProviders` is a number and `index` points to `providers`.
 */
export class NodeInjectorFactory {
    constructor(
    /**
     * Factory to invoke in order to create a new instance.
     */
    factory, 
    /**
     * Set to `true` if the token is declared in `viewProviders` (or if it is component).
     */
    isViewProvider, injectImplementation) {
        this.factory = factory;
        /**
         * Marker set to true during factory invocation to see if we get into recursive loop.
         * Recursive loop causes an error to be displayed.
         */
        this.resolving = false;
        ngDevMode && assertDefined(factory, 'Factory not specified');
        ngDevMode && assertEqual(typeof factory, 'function', 'Expected factory function.');
        this.canSeeViewProviders = isViewProvider;
        this.injectImpl = injectImplementation;
    }
}
export function isFactory(obj) {
    return obj instanceof NodeInjectorFactory;
}
//# sourceMappingURL=data:application/json;base64,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