/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { assertNumber } from '../../util/assert';
import { ID } from './view';
// Keeps track of the currently-active LViews.
const TRACKED_LVIEWS = new Map();
// Used for generating unique IDs for LViews.
let uniqueIdCounter = 0;
/** Gets a unique ID that can be assigned to an LView. */
export function getUniqueLViewId() {
    return uniqueIdCounter++;
}
/** Starts tracking an LView. */
export function registerLView(lView) {
    ngDevMode && assertNumber(lView[ID], 'LView must have an ID in order to be registered');
    TRACKED_LVIEWS.set(lView[ID], lView);
}
/** Gets an LView by its unique ID. */
export function getLViewById(id) {
    ngDevMode && assertNumber(id, 'ID used for LView lookup must be a number');
    return TRACKED_LVIEWS.get(id) || null;
}
/** Stops tracking an LView. */
export function unregisterLView(lView) {
    ngDevMode && assertNumber(lView[ID], 'Cannot stop tracking an LView that does not have an ID');
    TRACKED_LVIEWS.delete(lView[ID]);
}
//# sourceMappingURL=data:application/json;base64,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