/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { FactoryTarget, getCompilerFacade, } from '../../compiler/compiler_facade';
import { setClassMetadata, setClassMetadataAsync } from '../metadata';
import { angularCoreEnv } from './environment';
/**
 * Compiles a partial directive declaration object into a full directive definition object.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareDirective(decl) {
    const compiler = getCompilerFacade({
        usage: 1 /* JitCompilerUsage.PartialDeclaration */,
        kind: 'directive',
        type: decl.type,
    });
    return compiler.compileDirectiveDeclaration(angularCoreEnv, `ng:///${decl.type.name}/ɵfac.js`, decl);
}
/**
 * Evaluates the class metadata declaration.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareClassMetadata(decl) {
    setClassMetadata(decl.type, decl.decorators, decl.ctorParameters ?? null, decl.propDecorators ?? null);
}
/**
 * Evaluates the class metadata of a component that contains deferred blocks.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareClassMetadataAsync(decl) {
    setClassMetadataAsync(decl.type, decl.resolveDeferredDeps, (...types) => {
        const meta = decl.resolveMetadata(...types);
        setClassMetadata(decl.type, meta.decorators, meta.ctorParameters, meta.propDecorators);
    });
}
/**
 * Compiles a partial component declaration object into a full component definition object.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareComponent(decl) {
    const compiler = getCompilerFacade({
        usage: 1 /* JitCompilerUsage.PartialDeclaration */,
        kind: 'component',
        type: decl.type,
    });
    return compiler.compileComponentDeclaration(angularCoreEnv, `ng:///${decl.type.name}/ɵcmp.js`, decl);
}
/**
 * Compiles a partial pipe declaration object into a full pipe definition object.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareFactory(decl) {
    const compiler = getCompilerFacade({
        usage: 1 /* JitCompilerUsage.PartialDeclaration */,
        kind: getFactoryKind(decl.target),
        type: decl.type,
    });
    return compiler.compileFactoryDeclaration(angularCoreEnv, `ng:///${decl.type.name}/ɵfac.js`, decl);
}
function getFactoryKind(target) {
    switch (target) {
        case FactoryTarget.Directive:
            return 'directive';
        case FactoryTarget.Component:
            return 'component';
        case FactoryTarget.Injectable:
            return 'injectable';
        case FactoryTarget.Pipe:
            return 'pipe';
        case FactoryTarget.NgModule:
            return 'NgModule';
    }
}
/**
 * Compiles a partial injectable declaration object into a full injectable definition object.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareInjectable(decl) {
    const compiler = getCompilerFacade({
        usage: 1 /* JitCompilerUsage.PartialDeclaration */,
        kind: 'injectable',
        type: decl.type,
    });
    return compiler.compileInjectableDeclaration(angularCoreEnv, `ng:///${decl.type.name}/ɵprov.js`, decl);
}
/**
 * These enums are used in the partial factory declaration calls.
 */
export { FactoryTarget } from '../../compiler/compiler_facade';
/**
 * Compiles a partial injector declaration object into a full injector definition object.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareInjector(decl) {
    const compiler = getCompilerFacade({
        usage: 1 /* JitCompilerUsage.PartialDeclaration */,
        kind: 'NgModule',
        type: decl.type,
    });
    return compiler.compileInjectorDeclaration(angularCoreEnv, `ng:///${decl.type.name}/ɵinj.js`, decl);
}
/**
 * Compiles a partial NgModule declaration object into a full NgModule definition object.
 *
 * @codeGenApi
 */
export function ɵɵngDeclareNgModule(decl) {
    const compiler = getCompilerFacade({
        usage: 1 /* JitCompilerUsage.PartialDeclaration */,
        kind: 'NgModule',
        type: decl.type,
    });
    return compiler.compileNgModuleDeclaration(angularCoreEnv, `ng:///${decl.type.name}/ɵmod.js`, decl);
}
/**
 * Compiles a partial pipe declaration object into a full pipe definition object.
 *
 * @codeGenApi
 */
export function ɵɵngDeclarePipe(decl) {
    const compiler = getCompilerFacade({
        usage: 1 /* JitCompilerUsage.PartialDeclaration */,
        kind: 'pipe',
        type: decl.type,
    });
    return compiler.compilePipeDeclaration(angularCoreEnv, `ng:///${decl.type.name}/ɵpipe.js`, decl);
}
//# sourceMappingURL=data:application/json;base64,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