/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { isForwardRef, resolveForwardRef } from '../../di/forward_ref';
import { getComponentDef, getDirectiveDef, getNgModuleDef, getPipeDef } from '../definition';
import { stringifyForError } from '../util/stringify_utils';
export function isModuleWithProviders(value) {
    return value.ngModule !== undefined;
}
export function isNgModule(value) {
    return !!getNgModuleDef(value);
}
export function isPipe(value) {
    return !!getPipeDef(value);
}
export function isDirective(value) {
    return !!getDirectiveDef(value);
}
export function isComponent(value) {
    return !!getComponentDef(value);
}
function getDependencyTypeForError(type) {
    if (getComponentDef(type))
        return 'component';
    if (getDirectiveDef(type))
        return 'directive';
    if (getPipeDef(type))
        return 'pipe';
    return 'type';
}
export function verifyStandaloneImport(depType, importingType) {
    if (isForwardRef(depType)) {
        depType = resolveForwardRef(depType);
        if (!depType) {
            throw new Error(`Expected forwardRef function, imported from "${stringifyForError(importingType)}", to return a standalone entity or NgModule but got "${stringifyForError(depType) || depType}".`);
        }
    }
    if (getNgModuleDef(depType) == null) {
        const def = getComponentDef(depType) || getDirectiveDef(depType) || getPipeDef(depType);
        if (def != null) {
            // if a component, directive or pipe is imported make sure that it is standalone
            if (!def.standalone) {
                throw new Error(`The "${stringifyForError(depType)}" ${getDependencyTypeForError(depType)}, imported from "${stringifyForError(importingType)}", is not standalone. Did you forget to add the standalone: true flag?`);
            }
        }
        else {
            // it can be either a module with provider or an unknown (not annotated) type
            if (isModuleWithProviders(depType)) {
                throw new Error(`A module with providers was imported from "${stringifyForError(importingType)}". Modules with providers are not supported in standalone components imports.`);
            }
            else {
                throw new Error(`The "${stringifyForError(depType)}" type, imported from "${stringifyForError(importingType)}", must be a standalone component / directive / pipe or an NgModule. Did you forget to add the required @Component / @Directive / @Pipe or @NgModule annotation?`);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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