/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { createInjectorWithoutInjectorInstances } from '../di/create_injector';
import { getNullInjector, R3Injector } from '../di/r3_injector';
import { ComponentFactoryResolver as viewEngine_ComponentFactoryResolver } from '../linker/component_factory_resolver';
import { NgModuleFactory as viewEngine_NgModuleFactory, NgModuleRef as viewEngine_NgModuleRef, } from '../linker/ng_module_factory';
import { assertDefined } from '../util/assert';
import { stringify } from '../util/stringify';
import { ComponentFactoryResolver } from './component_ref';
import { getNgModuleDef } from './definition';
import { maybeUnwrapFn } from './util/misc_utils';
/**
 * Returns a new NgModuleRef instance based on the NgModule class and parent injector provided.
 *
 * @param ngModule NgModule class.
 * @param parentInjector Optional injector instance to use as a parent for the module injector. If
 *     not provided, `NullInjector` will be used instead.
 * @returns NgModuleRef that represents an NgModule instance.
 *
 * @publicApi
 */
export function createNgModule(ngModule, parentInjector) {
    return new NgModuleRef(ngModule, parentInjector ?? null, []);
}
/**
 * The `createNgModule` function alias for backwards-compatibility.
 * Please avoid using it directly and use `createNgModule` instead.
 *
 * @deprecated Use `createNgModule` instead.
 */
export const createNgModuleRef = createNgModule;
export class NgModuleRef extends viewEngine_NgModuleRef {
    constructor(ngModuleType, _parent, additionalProviders, runInjectorInitializers = true) {
        super();
        this.ngModuleType = ngModuleType;
        this._parent = _parent;
        // tslint:disable-next-line:require-internal-with-underscore
        this._bootstrapComponents = [];
        this.destroyCbs = [];
        // When bootstrapping a module we have a dependency graph that looks like this:
        // ApplicationRef -> ComponentFactoryResolver -> NgModuleRef. The problem is that if the
        // module being resolved tries to inject the ComponentFactoryResolver, it'll create a
        // circular dependency which will result in a runtime error, because the injector doesn't
        // exist yet. We work around the issue by creating the ComponentFactoryResolver ourselves
        // and providing it, rather than letting the injector resolve it.
        this.componentFactoryResolver = new ComponentFactoryResolver(this);
        const ngModuleDef = getNgModuleDef(ngModuleType);
        ngDevMode &&
            assertDefined(ngModuleDef, `NgModule '${stringify(ngModuleType)}' is not a subtype of 'NgModuleType'.`);
        this._bootstrapComponents = maybeUnwrapFn(ngModuleDef.bootstrap);
        this._r3Injector = createInjectorWithoutInjectorInstances(ngModuleType, _parent, [
            { provide: viewEngine_NgModuleRef, useValue: this },
            {
                provide: viewEngine_ComponentFactoryResolver,
                useValue: this.componentFactoryResolver,
            },
            ...additionalProviders,
        ], stringify(ngModuleType), new Set(['environment']));
        // We need to resolve the injector types separately from the injector creation, because
        // the module might be trying to use this ref in its constructor for DI which will cause a
        // circular error that will eventually error out, because the injector isn't created yet.
        if (runInjectorInitializers) {
            this.resolveInjectorInitializers();
        }
    }
    resolveInjectorInitializers() {
        this._r3Injector.resolveInjectorInitializers();
        this.instance = this._r3Injector.get(this.ngModuleType);
    }
    get injector() {
        return this._r3Injector;
    }
    destroy() {
        ngDevMode && assertDefined(this.destroyCbs, 'NgModule already destroyed');
        const injector = this._r3Injector;
        !injector.destroyed && injector.destroy();
        this.destroyCbs.forEach((fn) => fn());
        this.destroyCbs = null;
    }
    onDestroy(callback) {
        ngDevMode && assertDefined(this.destroyCbs, 'NgModule already destroyed');
        this.destroyCbs.push(callback);
    }
}
export class NgModuleFactory extends viewEngine_NgModuleFactory {
    constructor(moduleType) {
        super();
        this.moduleType = moduleType;
    }
    create(parentInjector) {
        return new NgModuleRef(this.moduleType, parentInjector, []);
    }
}
export function createNgModuleRefWithProviders(moduleType, parentInjector, additionalProviders) {
    return new NgModuleRef(moduleType, parentInjector, additionalProviders, false);
}
export class EnvironmentNgModuleRefAdapter extends viewEngine_NgModuleRef {
    constructor(config) {
        super();
        this.componentFactoryResolver = new ComponentFactoryResolver(this);
        this.instance = null;
        const injector = new R3Injector([
            ...config.providers,
            { provide: viewEngine_NgModuleRef, useValue: this },
            { provide: viewEngine_ComponentFactoryResolver, useValue: this.componentFactoryResolver },
        ], config.parent || getNullInjector(), config.debugName, new Set(['environment']));
        this.injector = injector;
        if (config.runEnvironmentInitializers) {
            injector.resolveInjectorInitializers();
        }
    }
    destroy() {
        this.injector.destroy();
    }
    onDestroy(callback) {
        this.injector.onDestroy(callback);
    }
}
/**
 * Create a new environment injector.
 *
 * @param providers An array of providers.
 * @param parent A parent environment injector.
 * @param debugName An optional name for this injector instance, which will be used in error
 *     messages.
 *
 * @publicApi
 */
export function createEnvironmentInjector(providers, parent, debugName = null) {
    const adapter = new EnvironmentNgModuleRefAdapter({
        providers,
        parent,
        debugName,
        runEnvironmentInitializers: true,
    });
    return adapter.injector;
}
//# sourceMappingURL=data:application/json;base64,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