/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { createComputed, SIGNAL } from '@angular/core/primitives/signals';
import { RuntimeError } from '../errors';
import { unwrapElementRef } from '../linker/element_ref';
import { EMPTY_ARRAY } from '../util/empty';
import { FLAGS } from './interfaces/view';
import { getQueryResults, loadQueryInternal } from './query';
import { signal } from './reactivity/signal';
import { getLView } from './state';
/**
 * A signal factory function in charge of creating a new computed signal capturing query
 * results. This centralized creation function is used by all types of queries (child / children,
 * required / optional).
 *
 * @param firstOnly indicates if all or only the first result should be returned
 * @param required indicates if at least one result is required
 * @returns a read-only signal with query results
 */
function createQuerySignalFn(firstOnly, required) {
    let node;
    const signalFn = createComputed(() => {
        // A dedicated signal that increments its value every time a query changes its dirty status. By
        // using this signal we can implement a query as computed and avoid creation of a specialized
        // reactive node type. Please note that a query gets marked dirty under the following
        // circumstances:
        // - a view (where a query is active) finished its first creation pass;
        // - a new view is inserted / deleted and it impacts query results.
        node._dirtyCounter();
        const value = refreshSignalQuery(node, firstOnly);
        if (required && value === undefined) {
            throw new RuntimeError(-951 /* RuntimeErrorCode.REQUIRED_QUERY_NO_VALUE */, ngDevMode && 'Child query result is required but no value is available.');
        }
        return value;
    });
    node = signalFn[SIGNAL];
    node._dirtyCounter = signal(0);
    node._flatValue = undefined;
    if (ngDevMode) {
        signalFn.toString = () => `[Query Signal]`;
    }
    return signalFn;
}
export function createSingleResultOptionalQuerySignalFn() {
    return createQuerySignalFn(/* firstOnly */ true, /* required */ false);
}
export function createSingleResultRequiredQuerySignalFn() {
    return createQuerySignalFn(/* firstOnly */ true, /* required */ true);
}
export function createMultiResultQuerySignalFn() {
    return createQuerySignalFn(/* firstOnly */ false, /* required */ false);
}
export function bindQueryToSignal(target, queryIndex) {
    const node = target[SIGNAL];
    node._lView = getLView();
    node._queryIndex = queryIndex;
    node._queryList = loadQueryInternal(node._lView, queryIndex);
    node._queryList.onDirty(() => node._dirtyCounter.update((v) => v + 1));
}
function refreshSignalQuery(node, firstOnly) {
    const lView = node._lView;
    const queryIndex = node._queryIndex;
    // There are 2 conditions under which we want to return "empty" results instead of the ones
    // collected by a query:
    //
    // 1) a given query wasn't created yet (this is a period of time between the directive creation
    // and execution of the query creation function) - in this case a query doesn't exist yet and we
    // don't have any results to return.
    //
    // 2) we are in the process of constructing a view (the first
    // creation pass didn't finish) and a query might have partial results, but we don't want to
    // return those - instead we do delay results collection until all nodes had a chance of matching
    // and we can present consistent, "atomic" (on a view level) results.
    if (lView === undefined || queryIndex === undefined || lView[FLAGS] & 4 /* LViewFlags.CreationMode */) {
        return (firstOnly ? undefined : EMPTY_ARRAY);
    }
    const queryList = loadQueryInternal(lView, queryIndex);
    const results = getQueryResults(lView, queryIndex);
    queryList.reset(results, unwrapElementRef);
    if (firstOnly) {
        return queryList.first;
    }
    else {
        // TODO: remove access to the private _changesDetected field by abstracting / removing usage of
        // QueryList in the signal-based queries (perf follow-up)
        const resultChanged = queryList._changesDetected;
        if (resultChanged || node._flatValue === undefined) {
            return (node._flatValue = queryList.toArray());
        }
        return node._flatValue;
    }
}
//# sourceMappingURL=data:application/json;base64,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