/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { REACTIVE_NODE } from '@angular/core/primitives/signals';
import { REACTIVE_TEMPLATE_CONSUMER, TVIEW, } from './interfaces/view';
import { getLViewParent, markAncestorsForTraversal, markViewForRefresh } from './util/view_utils';
let freeConsumers = [];
/**
 * Create a new template consumer pointing at the specified LView.
 * Sometimes, a previously created consumer may be reused, in order to save on allocations. In that
 * case, the LView will be updated.
 */
export function getOrBorrowReactiveLViewConsumer(lView) {
    return lView[REACTIVE_TEMPLATE_CONSUMER] ?? borrowReactiveLViewConsumer(lView);
}
function borrowReactiveLViewConsumer(lView) {
    const consumer = freeConsumers.pop() ?? Object.create(REACTIVE_LVIEW_CONSUMER_NODE);
    consumer.lView = lView;
    return consumer;
}
export function maybeReturnReactiveLViewConsumer(consumer) {
    if (consumer.lView[REACTIVE_TEMPLATE_CONSUMER] === consumer) {
        // The consumer got committed.
        return;
    }
    consumer.lView = null;
    freeConsumers.push(consumer);
}
const REACTIVE_LVIEW_CONSUMER_NODE = {
    ...REACTIVE_NODE,
    consumerIsAlwaysLive: true,
    consumerMarkedDirty: (node) => {
        markAncestorsForTraversal(node.lView);
    },
    consumerOnSignalRead() {
        this.lView[REACTIVE_TEMPLATE_CONSUMER] = this;
    },
};
/**
 * Creates a temporary consumer for use with `LView`s that should not have consumers.
 * If the LView already has a consumer, returns the existing one instead.
 *
 * This is necessary because some APIs may cause change detection directly on an LView
 * that we do not want to have a consumer (Embedded views today). As a result, there
 * would be no active consumer from running change detection on its host component
 * and any signals in the LView template would be untracked. Instead, we create
 * this temporary consumer that marks the first parent that _should_ have a consumer
 * for refresh. Once change detection runs as part of that refresh, we throw away
 * this consumer because its signals will then be tracked by the parent's consumer.
 */
export function getOrCreateTemporaryConsumer(lView) {
    const consumer = lView[REACTIVE_TEMPLATE_CONSUMER] ?? Object.create(TEMPORARY_CONSUMER_NODE);
    consumer.lView = lView;
    return consumer;
}
const TEMPORARY_CONSUMER_NODE = {
    ...REACTIVE_NODE,
    consumerIsAlwaysLive: true,
    consumerMarkedDirty: (node) => {
        let parent = getLViewParent(node.lView);
        while (parent && !viewShouldHaveReactiveConsumer(parent[TVIEW])) {
            parent = getLViewParent(parent);
        }
        if (!parent) {
            // If we can't find an appropriate parent that should have a consumer, we
            // don't have a way of appropriately refreshing this LView as part of application synchronization.
            return;
        }
        markViewForRefresh(parent);
    },
    consumerOnSignalRead() {
        this.lView[REACTIVE_TEMPLATE_CONSUMER] = this;
    },
};
/**
 * Indicates if the view should get its own reactive consumer node.
 *
 * In the current design, all embedded views share a consumer with the component view. This allows
 * us to refresh at the component level rather than at a per-view level. In addition, root views get
 * their own reactive node because root component will have a host view that executes the
 * component's host bindings. This needs to be tracked in a consumer as well.
 *
 * To get a more granular change detection than per-component, all we would just need to update the
 * condition here so that a given view gets a reactive consumer which can become dirty independently
 * from its parent component. For example embedded views for signal components could be created with
 * a new type "SignalEmbeddedView" and the condition here wouldn't even need updating in order to
 * get granular per-view change detection for signal components.
 */
export function viewShouldHaveReactiveConsumer(tView) {
    return tView.type !== 2 /* TViewType.Embedded */;
}
//# sourceMappingURL=data:application/json;base64,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