/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { createSignal, SIGNAL, signalSetFn, signalUpdateFn, } from '@angular/core/primitives/signals';
import { performanceMarkFeature } from '../../util/performance';
import { isSignal } from './api';
/** Symbol used distinguish `WritableSignal` from other non-writable signals and functions. */
export const ɵWRITABLE_SIGNAL = /* @__PURE__ */ Symbol('WRITABLE_SIGNAL');
/**
 * Utility function used during template type checking to extract the value from a `WritableSignal`.
 * @codeGenApi
 */
export function ɵunwrapWritableSignal(value) {
    // Note: the function uses `WRITABLE_SIGNAL` as a brand instead of `WritableSignal<T>`,
    // because the latter incorrectly unwraps non-signal getter functions.
    return null;
}
/**
 * Create a `Signal` that can be set or updated directly.
 */
export function signal(initialValue, options) {
    performanceMarkFeature('NgSignals');
    const signalFn = createSignal(initialValue);
    const node = signalFn[SIGNAL];
    if (options?.equal) {
        node.equal = options.equal;
    }
    signalFn.set = (newValue) => signalSetFn(node, newValue);
    signalFn.update = (updateFn) => signalUpdateFn(node, updateFn);
    signalFn.asReadonly = signalAsReadonlyFn.bind(signalFn);
    if (ngDevMode) {
        signalFn.toString = () => `[Signal: ${signalFn()}]`;
    }
    return signalFn;
}
export function signalAsReadonlyFn() {
    const node = this[SIGNAL];
    if (node.readonlyFn === undefined) {
        const readonlyFn = () => this();
        readonlyFn[SIGNAL] = node;
        node.readonlyFn = readonlyFn;
    }
    return node.readonlyFn;
}
/**
 * Checks if the given `value` is a writeable signal.
 */
export function isWritableSignal(value) {
    return isSignal(value) && typeof value.set === 'function';
}
//# sourceMappingURL=data:application/json;base64,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