/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { isForwardRef, resolveForwardRef } from '../di/forward_ref';
import { flatten } from '../util/array_utils';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY } from '../util/empty';
import { extractDefListOrFactory, getNgModuleDef } from './definition';
import { depsTracker } from './deps_tracker/deps_tracker';
import { isModuleWithProviders } from './jit/util';
/**
 * Generated next to NgModules to monkey-patch directive and pipe references onto a component's
 * definition, when generating a direct reference in the component file would otherwise create an
 * import cycle.
 *
 * See [this explanation](https://hackmd.io/Odw80D0pR6yfsOjg_7XCJg?view) for more details.
 *
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = extractDefListOrFactory(directives, /* pipeDef */ false);
    def.pipeDefs = extractDefListOrFactory(pipes, /* pipeDef */ true);
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = convertToTypeArray(scope.declarations || EMPTY_ARRAY);
        ngModuleDef.imports = convertToTypeArray(scope.imports || EMPTY_ARRAY);
        ngModuleDef.exports = convertToTypeArray(scope.exports || EMPTY_ARRAY);
        if (scope.bootstrap) {
            // This only happens in local compilation mode.
            ngModuleDef.bootstrap = convertToTypeArray(scope.bootstrap);
        }
        depsTracker.registerNgModule(type, scope);
    });
}
function convertToTypeArray(values) {
    if (typeof values === 'function') {
        return values;
    }
    const flattenValues = flatten(values);
    if (flattenValues.some(isForwardRef)) {
        return () => flattenValues.map(resolveForwardRef).map(maybeUnwrapModuleWithProviders);
    }
    else {
        return flattenValues.map(maybeUnwrapModuleWithProviders);
    }
}
function maybeUnwrapModuleWithProviders(value) {
    return isModuleWithProviders(value) ? value.ngModule : value;
}
//# sourceMappingURL=data:application/json;base64,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