/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { assertDefined } from '../../util/assert';
import { global } from '../../util/global';
import { setupFrameworkInjectorProfiler } from '../debug/framework_injector_profiler';
import { setProfiler } from '../profiler';
import { isSignal } from '../reactivity/api';
import { applyChanges } from './change_detection_utils';
import { getComponent, getContext, getDirectiveMetadata, getDirectives, getHostElement, getInjector, getListeners, getOwningComponent, getRootComponents, } from './discovery_utils';
import { getDependenciesFromInjectable, getInjectorMetadata, getInjectorProviders, getInjectorResolutionPath, } from './injector_discovery_utils';
/**
 * This file introduces series of globally accessible debug tools
 * to allow for the Angular debugging story to function.
 *
 * To see this in action run the following command:
 *
 *   bazel run //packages/core/test/bundling/todo:devserver
 *
 *  Then load `localhost:5432` and start using the console tools.
 */
/**
 * This value reflects the property on the window where the dev
 * tools are patched (window.ng).
 * */
export const GLOBAL_PUBLISH_EXPANDO_KEY = 'ng';
const globalUtilsFunctions = {
    /**
     * Warning: functions that start with `ɵ` are considered *INTERNAL* and should not be relied upon
     * in application's code. The contract of those functions might be changed in any release and/or a
     * function can be removed completely.
     */
    'ɵgetDependenciesFromInjectable': getDependenciesFromInjectable,
    'ɵgetInjectorProviders': getInjectorProviders,
    'ɵgetInjectorResolutionPath': getInjectorResolutionPath,
    'ɵgetInjectorMetadata': getInjectorMetadata,
    'ɵsetProfiler': setProfiler,
    'getDirectiveMetadata': getDirectiveMetadata,
    'getComponent': getComponent,
    'getContext': getContext,
    'getListeners': getListeners,
    'getOwningComponent': getOwningComponent,
    'getHostElement': getHostElement,
    'getInjector': getInjector,
    'getRootComponents': getRootComponents,
    'getDirectives': getDirectives,
    'applyChanges': applyChanges,
    'isSignal': isSignal,
};
let _published = false;
/**
 * Publishes a collection of default debug tools onto`window.ng`.
 *
 * These functions are available globally when Angular is in development
 * mode and are automatically stripped away from prod mode is on.
 */
export function publishDefaultGlobalUtils() {
    if (!_published) {
        _published = true;
        if (typeof window !== 'undefined') {
            // Only configure the injector profiler when running in the browser.
            setupFrameworkInjectorProfiler();
        }
        for (const [methodName, method] of Object.entries(globalUtilsFunctions)) {
            publishGlobalUtil(methodName, method);
        }
    }
}
/**
 * Publishes the given function to `window.ng` so that it can be
 * used from the browser console when an application is not in production.
 */
export function publishGlobalUtil(name, fn) {
    if (typeof COMPILED === 'undefined' || !COMPILED) {
        // Note: we can't export `ng` when using closure enhanced optimization as:
        // - closure declares globals itself for minified names, which sometimes clobber our `ng` global
        // - we can't declare a closure extern as the namespace `ng` is already used within Google
        //   for typings for AngularJS (via `goog.provide('ng....')`).
        const w = global;
        ngDevMode && assertDefined(fn, 'function not defined');
        w[GLOBAL_PUBLISH_EXPANDO_KEY] ??= {};
        w[GLOBAL_PUBLISH_EXPANDO_KEY][name] = fn;
    }
}
//# sourceMappingURL=data:application/json;base64,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