/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { setActiveConsumer } from '@angular/core/primitives/signals';
import { hasInSkipHydrationBlockFlag } from '../hydration/skip_hydration';
import { assertDefined } from '../util/assert';
import { assertLContainer, assertLView, assertTNodeForLView } from './assert';
import { renderView } from './instructions/render';
import { createLView } from './instructions/shared';
import { CONTAINER_HEADER_OFFSET, NATIVE } from './interfaces/container';
import { DECLARATION_LCONTAINER, FLAGS, HYDRATION, QUERIES, RENDERER, T_HOST, TVIEW, } from './interfaces/view';
import { addViewToDOM, destroyLView, detachView, getBeforeNodeForView, insertView, nativeParentNode, } from './node_manipulation';
export function createAndRenderEmbeddedLView(declarationLView, templateTNode, context, options) {
    const prevConsumer = setActiveConsumer(null);
    try {
        const embeddedTView = templateTNode.tView;
        ngDevMode && assertDefined(embeddedTView, 'TView must be defined for a template node.');
        ngDevMode && assertTNodeForLView(templateTNode, declarationLView);
        // Embedded views follow the change detection strategy of the view they're declared in.
        const isSignalView = declarationLView[FLAGS] & 4096 /* LViewFlags.SignalView */;
        const viewFlags = isSignalView ? 4096 /* LViewFlags.SignalView */ : 16 /* LViewFlags.CheckAlways */;
        const embeddedLView = createLView(declarationLView, embeddedTView, context, viewFlags, null, templateTNode, null, null, options?.injector ?? null, options?.embeddedViewInjector ?? null, options?.dehydratedView ?? null);
        const declarationLContainer = declarationLView[templateTNode.index];
        ngDevMode && assertLContainer(declarationLContainer);
        embeddedLView[DECLARATION_LCONTAINER] = declarationLContainer;
        const declarationViewLQueries = declarationLView[QUERIES];
        if (declarationViewLQueries !== null) {
            embeddedLView[QUERIES] = declarationViewLQueries.createEmbeddedView(embeddedTView);
        }
        // execute creation mode of a view
        renderView(embeddedTView, embeddedLView, context);
        return embeddedLView;
    }
    finally {
        setActiveConsumer(prevConsumer);
    }
}
export function getLViewFromLContainer(lContainer, index) {
    const adjustedIndex = CONTAINER_HEADER_OFFSET + index;
    // avoid reading past the array boundaries
    if (adjustedIndex < lContainer.length) {
        const lView = lContainer[adjustedIndex];
        ngDevMode && assertLView(lView);
        return lView;
    }
    return undefined;
}
/**
 * Returns whether an elements that belong to a view should be
 * inserted into the DOM. For client-only cases, DOM elements are
 * always inserted. For hydration cases, we check whether serialized
 * info is available for a view and the view is not in a "skip hydration"
 * block (in which case view contents was re-created, thus needing insertion).
 */
export function shouldAddViewToDom(tNode, dehydratedView) {
    return (!dehydratedView || dehydratedView.firstChild === null || hasInSkipHydrationBlockFlag(tNode));
}
export function addLViewToLContainer(lContainer, lView, index, addToDOM = true) {
    const tView = lView[TVIEW];
    // Insert into the view tree so the new view can be change-detected
    insertView(tView, lView, lContainer, index);
    // Insert elements that belong to this view into the DOM tree
    if (addToDOM) {
        const beforeNode = getBeforeNodeForView(index, lContainer);
        const renderer = lView[RENDERER];
        const parentRNode = nativeParentNode(renderer, lContainer[NATIVE]);
        if (parentRNode !== null) {
            addViewToDOM(tView, lContainer[T_HOST], renderer, lView, parentRNode, beforeNode);
        }
    }
    // When in hydration mode, reset the pointer to the first child in
    // the dehydrated view. This indicates that the view was hydrated and
    // further attaching/detaching should work with this view as normal.
    const hydrationInfo = lView[HYDRATION];
    if (hydrationInfo !== null && hydrationInfo.firstChild !== null) {
        hydrationInfo.firstChild = null;
    }
}
export function removeLViewFromLContainer(lContainer, index) {
    const lView = detachView(lContainer, index);
    if (lView !== undefined) {
        destroyLView(lView[TVIEW], lView);
    }
    return lView;
}
//# sourceMappingURL=data:application/json;base64,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