/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */
import { ɵRuntimeError as RuntimeError } from '@angular/core';
import { formArrayNameExample, formControlNameExample, formGroupNameExample, ngModelGroupExample, } from './error_examples';
export function controlParentException(nameOrIndex) {
    return new RuntimeError(1050 /* RuntimeErrorCode.FORM_CONTROL_NAME_MISSING_PARENT */, `formControlName must be used with a parent formGroup directive. You'll want to add a formGroup
      directive and pass it an existing FormGroup instance (you can create one in your class).

      ${describeFormControl(nameOrIndex)}

    Example:

    ${formControlNameExample}`);
}
function describeFormControl(nameOrIndex) {
    if (nameOrIndex == null || nameOrIndex === '') {
        return '';
    }
    const valueType = typeof nameOrIndex === 'string' ? 'name' : 'index';
    return `Affected Form Control ${valueType}: "${nameOrIndex}"`;
}
export function ngModelGroupException() {
    return new RuntimeError(1051 /* RuntimeErrorCode.FORM_CONTROL_NAME_INSIDE_MODEL_GROUP */, `formControlName cannot be used with an ngModelGroup parent. It is only compatible with parents
      that also have a "form" prefix: formGroupName, formArrayName, or formGroup.

      Option 1:  Update the parent to be formGroupName (reactive form strategy)

      ${formGroupNameExample}

      Option 2: Use ngModel instead of formControlName (template-driven strategy)

      ${ngModelGroupExample}`);
}
export function missingFormException() {
    return new RuntimeError(1052 /* RuntimeErrorCode.FORM_GROUP_MISSING_INSTANCE */, `formGroup expects a FormGroup instance. Please pass one in.

      Example:

      ${formControlNameExample}`);
}
export function groupParentException() {
    return new RuntimeError(1053 /* RuntimeErrorCode.FORM_GROUP_NAME_MISSING_PARENT */, `formGroupName must be used with a parent formGroup directive.  You'll want to add a formGroup
    directive and pass it an existing FormGroup instance (you can create one in your class).

    Example:

    ${formGroupNameExample}`);
}
export function arrayParentException() {
    return new RuntimeError(1054 /* RuntimeErrorCode.FORM_ARRAY_NAME_MISSING_PARENT */, `formArrayName must be used with a parent formGroup directive.  You'll want to add a formGroup
      directive and pass it an existing FormGroup instance (you can create one in your class).

      Example:

      ${formArrayNameExample}`);
}
export const disabledAttrWarning = `
  It looks like you're using the disabled attribute with a reactive form directive. If you set disabled to true
  when you set up this control in your component class, the disabled attribute will actually be set in the DOM for
  you. We recommend using this approach to avoid 'changed after checked' errors.

  Example:
  // Specify the \`disabled\` property at control creation time:
  form = new FormGroup({
    first: new FormControl({value: 'Nancy', disabled: true}, Validators.required),
    last: new FormControl('Drew', Validators.required)
  });

  // Controls can also be enabled/disabled after creation:
  form.get('first')?.enable();
  form.get('last')?.disable();
`;
export const asyncValidatorsDroppedWithOptsWarning = `
  It looks like you're constructing using a FormControl with both an options argument and an
  async validators argument. Mixing these arguments will cause your async validators to be dropped.
  You should either put all your validators in the options object, or in separate validators
  arguments. For example:

  // Using validators arguments
  fc = new FormControl(42, Validators.required, myAsyncValidator);

  // Using AbstractControlOptions
  fc = new FormControl(42, {validators: Validators.required, asyncValidators: myAV});

  // Do NOT mix them: async validators will be dropped!
  fc = new FormControl(42, {validators: Validators.required}, /* Oops! */ myAsyncValidator);
`;
export function ngModelWarning(directiveName) {
    return `
  It looks like you're using ngModel on the same form field as ${directiveName}.
  Support for using the ngModel input property and ngModelChange event with
  reactive form directives has been deprecated in Angular v6 and will be removed
  in a future version of Angular.

  For more information on this, see our API docs here:
  https://angular.io/api/forms/${directiveName === 'formControl' ? 'FormControlDirective' : 'FormControlName'}#use-with-ngmodel
  `;
}
function describeKey(isFormGroup, key) {
    return isFormGroup ? `with name: '${key}'` : `at index: ${key}`;
}
export function noControlsError(isFormGroup) {
    return `
    There are no form controls registered with this ${isFormGroup ? 'group' : 'array'} yet. If you're using ngModel,
    you may want to check next tick (e.g. use setTimeout).
  `;
}
export function missingControlError(isFormGroup, key) {
    return `Cannot find form control ${describeKey(isFormGroup, key)}`;
}
export function missingControlValueError(isFormGroup, key) {
    return `Must supply a value for form control ${describeKey(isFormGroup, key)}`;
}
//# sourceMappingURL=data:application/json;base64,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