import { parseTranslation, translate as _translate, } from './utils';
/**
 * Load translations for use by `$localize`, if doing runtime translation.
 *
 * If the `$localize` tagged strings are not going to be replaced at compiled time, it is possible
 * to load a set of translations that will be applied to the `$localize` tagged strings at runtime,
 * in the browser.
 *
 * Loading a new translation will overwrite a previous translation if it has the same `MessageId`.
 *
 * Note that `$localize` messages are only processed once, when the tagged string is first
 * encountered, and does not provide dynamic language changing without refreshing the browser.
 * Loading new translations later in the application life-cycle will not change the translated text
 * of messages that have already been translated.
 *
 * The message IDs and translations are in the same format as that rendered to "simple JSON"
 * translation files when extracting messages. In particular, placeholders in messages are rendered
 * using the `{$PLACEHOLDER_NAME}` syntax. For example the message from the following template:
 *
 * ```html
 * <div i18n>pre<span>inner-pre<b>bold</b>inner-post</span>post</div>
 * ```
 *
 * would have the following form in the `translations` map:
 *
 * ```ts
 * {
 *   "2932901491976224757":
 *      "pre{$START_TAG_SPAN}inner-pre{$START_BOLD_TEXT}bold{$CLOSE_BOLD_TEXT}inner-post{$CLOSE_TAG_SPAN}post"
 * }
 * ```
 *
 * @param translations A map from message ID to translated message.
 *
 * These messages are processed and added to a lookup based on their `MessageId`.
 *
 * @see {@link clearTranslations} for removing translations loaded using this function.
 * @see {@link $localize} for tagging messages as needing to be translated.
 * @publicApi
 */
export function loadTranslations(translations) {
    // Ensure the translate function exists
    if (!$localize.translate) {
        $localize.translate = translate;
    }
    if (!$localize.TRANSLATIONS) {
        $localize.TRANSLATIONS = {};
    }
    Object.keys(translations).forEach((key) => {
        $localize.TRANSLATIONS[key] = parseTranslation(translations[key]);
    });
}
/**
 * Remove all translations for `$localize`, if doing runtime translation.
 *
 * All translations that had been loading into memory using `loadTranslations()` will be removed.
 *
 * @see {@link loadTranslations} for loading translations at runtime.
 * @see {@link $localize} for tagging messages as needing to be translated.
 *
 * @publicApi
 */
export function clearTranslations() {
    $localize.translate = undefined;
    $localize.TRANSLATIONS = {};
}
/**
 * Translate the text of the given message, using the loaded translations.
 *
 * This function may reorder (or remove) substitutions as indicated in the matching translation.
 */
export function translate(messageParts, substitutions) {
    try {
        return _translate($localize.TRANSLATIONS, messageParts, substitutions);
    }
    catch (e) {
        console.warn(e.message);
        return [messageParts, substitutions];
    }
}
//# sourceMappingURL=data:application/json;base64,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