import { NgbDate } from './ngb-date';
export function isChangedDate(prev, next) {
    return !dateComparator(prev, next);
}
export function isChangedMonth(prev, next) {
    return !prev && !next ? false : !prev || !next ? true : prev.year !== next.year || prev.month !== next.month;
}
export function dateComparator(prev, next) {
    return (!prev && !next) || (!!prev && !!next && prev.equals(next));
}
export function checkMinBeforeMax(minDate, maxDate) {
    if (maxDate && minDate && maxDate.before(minDate)) {
        throw new Error(`'maxDate' ${maxDate} should be greater than 'minDate' ${minDate}`);
    }
}
export function checkDateInRange(date, minDate, maxDate) {
    if (date && minDate && date.before(minDate)) {
        return minDate;
    }
    if (date && maxDate && date.after(maxDate)) {
        return maxDate;
    }
    return date || null;
}
export function isDateSelectable(date, state) {
    const { minDate, maxDate, disabled, markDisabled } = state;
    return !(date === null ||
        date === undefined ||
        disabled ||
        (markDisabled && markDisabled(date, { year: date.year, month: date.month })) ||
        (minDate && date.before(minDate)) ||
        (maxDate && date.after(maxDate)));
}
export function generateSelectBoxMonths(calendar, date, minDate, maxDate) {
    if (!date) {
        return [];
    }
    let months = calendar.getMonths(date.year);
    if (minDate && date.year === minDate.year) {
        const index = months.findIndex((month) => month === minDate.month);
        months = months.slice(index);
    }
    if (maxDate && date.year === maxDate.year) {
        const index = months.findIndex((month) => month === maxDate.month);
        months = months.slice(0, index + 1);
    }
    return months;
}
export function generateSelectBoxYears(date, minDate, maxDate) {
    if (!date) {
        return [];
    }
    const start = minDate ? Math.max(minDate.year, date.year - 500) : date.year - 10;
    const end = maxDate ? Math.min(maxDate.year, date.year + 500) : date.year + 10;
    const length = end - start + 1;
    const numbers = Array(length);
    for (let i = 0; i < length; i++) {
        numbers[i] = start + i;
    }
    return numbers;
}
export function nextMonthDisabled(calendar, date, maxDate) {
    const nextDate = Object.assign(calendar.getNext(date, 'm'), { day: 1 });
    return maxDate != null && nextDate.after(maxDate);
}
export function prevMonthDisabled(calendar, date, minDate) {
    const prevDate = Object.assign(calendar.getPrev(date, 'm'), { day: 1 });
    return (minDate != null &&
        ((prevDate.year === minDate.year && prevDate.month < minDate.month) ||
            (prevDate.year < minDate.year && minDate.month === 1)));
}
export function buildMonths(calendar, date, state, i18n, force) {
    const { displayMonths, months } = state;
    // move old months to a temporary array
    const monthsToReuse = months.splice(0, months.length);
    // generate new first dates, nullify or reuse months
    const firstDates = Array.from({ length: displayMonths }, (_, i) => {
        const firstDate = Object.assign(calendar.getNext(date, 'm', i), { day: 1 });
        months[i] = null;
        if (!force) {
            const reusedIndex = monthsToReuse.findIndex((month) => month.firstDate.equals(firstDate));
            // move reused month back to months
            if (reusedIndex !== -1) {
                months[i] = monthsToReuse.splice(reusedIndex, 1)[0];
            }
        }
        return firstDate;
    });
    // rebuild nullified months
    firstDates.forEach((firstDate, i) => {
        if (months[i] === null) {
            months[i] = buildMonth(calendar, firstDate, state, i18n, monthsToReuse.shift() || {});
        }
    });
    return months;
}
export function buildMonth(calendar, date, state, i18n, month = {}) {
    const { dayTemplateData, minDate, maxDate, firstDayOfWeek, markDisabled, outsideDays, weekdayWidth, weekdaysVisible, } = state;
    const calendarToday = calendar.getToday();
    month.firstDate = null;
    month.lastDate = null;
    month.number = date.month;
    month.year = date.year;
    month.weeks = month.weeks || [];
    month.weekdays = month.weekdays || [];
    date = getFirstViewDate(calendar, date, firstDayOfWeek);
    // clearing weekdays, if not visible
    if (!weekdaysVisible) {
        month.weekdays.length = 0;
    }
    // month has weeks
    for (let week = 0; week < calendar.getWeeksPerMonth(); week++) {
        let weekObject = month.weeks[week];
        if (!weekObject) {
            weekObject = month.weeks[week] = { number: 0, days: [], collapsed: true };
        }
        const days = weekObject.days;
        // week has days
        for (let day = 0; day < calendar.getDaysPerWeek(); day++) {
            if (week === 0 && weekdaysVisible) {
                month.weekdays[day] = i18n.getWeekdayLabel(calendar.getWeekday(date), weekdayWidth);
            }
            const newDate = new NgbDate(date.year, date.month, date.day);
            const nextDate = calendar.getNext(newDate);
            const ariaLabel = i18n.getDayAriaLabel(newDate);
            // marking date as disabled
            let disabled = !!((minDate && newDate.before(minDate)) || (maxDate && newDate.after(maxDate)));
            if (!disabled && markDisabled) {
                disabled = markDisabled(newDate, { month: month.number, year: month.year });
            }
            // today
            let today = newDate.equals(calendarToday);
            // adding user-provided data to the context
            let contextUserData = dayTemplateData
                ? dayTemplateData(newDate, { month: month.number, year: month.year })
                : undefined;
            // saving first date of the month
            if (month.firstDate === null && newDate.month === month.number) {
                month.firstDate = newDate;
            }
            // saving last date of the month
            if (newDate.month === month.number && nextDate.month !== month.number) {
                month.lastDate = newDate;
            }
            let dayObject = days[day];
            if (!dayObject) {
                dayObject = days[day] = {};
            }
            dayObject.date = newDate;
            dayObject.context = Object.assign(dayObject.context || {}, {
                $implicit: newDate,
                date: newDate,
                data: contextUserData,
                currentMonth: month.number,
                currentYear: month.year,
                disabled,
                focused: false,
                selected: false,
                today,
            });
            dayObject.tabindex = -1;
            dayObject.ariaLabel = ariaLabel;
            dayObject.hidden = false;
            date = nextDate;
        }
        weekObject.number = calendar.getWeekNumber(days.map((day) => day.date), firstDayOfWeek);
        // marking week as collapsed
        weekObject.collapsed =
            outsideDays === 'collapsed' &&
                days[0].date.month !== month.number &&
                days[days.length - 1].date.month !== month.number;
    }
    return month;
}
export function getFirstViewDate(calendar, date, firstDayOfWeek) {
    const daysPerWeek = calendar.getDaysPerWeek();
    const firstMonthDate = new NgbDate(date.year, date.month, 1);
    const dayOfWeek = calendar.getWeekday(firstMonthDate) % daysPerWeek;
    return calendar.getPrev(firstMonthDate, 'd', (daysPerWeek + dayOfWeek - firstDayOfWeek) % daysPerWeek);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0ZXBpY2tlci10b29scy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9kYXRlcGlja2VyL2RhdGVwaWNrZXItdG9vbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsT0FBTyxFQUFFLE9BQU8sRUFBRSxNQUFNLFlBQVksQ0FBQztBQUtyQyxNQUFNLFVBQVUsYUFBYSxDQUFDLElBQXFCLEVBQUUsSUFBcUI7SUFDekUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDcEMsQ0FBQztBQUVELE1BQU0sVUFBVSxjQUFjLENBQUMsSUFBcUIsRUFBRSxJQUFxQjtJQUMxRSxPQUFPLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsS0FBSyxLQUFLLElBQUksQ0FBQyxLQUFLLENBQUM7QUFDOUcsQ0FBQztBQUVELE1BQU0sVUFBVSxjQUFjLENBQUMsSUFBcUIsRUFBRSxJQUFxQjtJQUMxRSxPQUFPLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7QUFDcEUsQ0FBQztBQUVELE1BQU0sVUFBVSxpQkFBaUIsQ0FBQyxPQUF3QixFQUFFLE9BQXdCO0lBQ25GLElBQUksT0FBTyxJQUFJLE9BQU8sSUFBSSxPQUFPLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDbkQsTUFBTSxJQUFJLEtBQUssQ0FBQyxhQUFhLE9BQU8scUNBQXFDLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDckYsQ0FBQztBQUNGLENBQUM7QUFFRCxNQUFNLFVBQVUsZ0JBQWdCLENBQy9CLElBQXFCLEVBQ3JCLE9BQXdCLEVBQ3hCLE9BQXdCO0lBRXhCLElBQUksSUFBSSxJQUFJLE9BQU8sSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDN0MsT0FBTyxPQUFPLENBQUM7SUFDaEIsQ0FBQztJQUNELElBQUksSUFBSSxJQUFJLE9BQU8sSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDNUMsT0FBTyxPQUFPLENBQUM7SUFDaEIsQ0FBQztJQUVELE9BQU8sSUFBSSxJQUFJLElBQUksQ0FBQztBQUNyQixDQUFDO0FBRUQsTUFBTSxVQUFVLGdCQUFnQixDQUFDLElBQWdDLEVBQUUsS0FBMEI7SUFDNUYsTUFBTSxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLFlBQVksRUFBRSxHQUFHLEtBQUssQ0FBQztJQUMzRCxPQUFPLENBQUMsQ0FDUCxJQUFJLEtBQUssSUFBSTtRQUNiLElBQUksS0FBSyxTQUFTO1FBQ2xCLFFBQVE7UUFDUixDQUFDLFlBQVksSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQzVFLENBQUMsT0FBTyxJQUFJLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDakMsQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUNoQyxDQUFDO0FBQ0gsQ0FBQztBQUVELE1BQU0sVUFBVSx1QkFBdUIsQ0FDdEMsUUFBcUIsRUFDckIsSUFBYSxFQUNiLE9BQXVCLEVBQ3ZCLE9BQXVCO0lBRXZCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNYLE9BQU8sRUFBRSxDQUFDO0lBQ1gsQ0FBQztJQUVELElBQUksTUFBTSxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBRTNDLElBQUksT0FBTyxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQzNDLE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLEtBQUssS0FBSyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbkUsTUFBTSxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVELElBQUksT0FBTyxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQzNDLE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLEtBQUssS0FBSyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbkUsTUFBTSxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLEtBQUssR0FBRyxDQUFDLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQsT0FBTyxNQUFNLENBQUM7QUFDZixDQUFDO0FBRUQsTUFBTSxVQUFVLHNCQUFzQixDQUFDLElBQWEsRUFBRSxPQUF1QixFQUFFLE9BQXVCO0lBQ3JHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNYLE9BQU8sRUFBRSxDQUFDO0lBQ1gsQ0FBQztJQUVELE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsRUFBRSxDQUFDO0lBQ2pGLE1BQU0sR0FBRyxHQUFHLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsRUFBRSxDQUFDO0lBRS9FLE1BQU0sTUFBTSxHQUFHLEdBQUcsR0FBRyxLQUFLLEdBQUcsQ0FBQyxDQUFDO0lBQy9CLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUM5QixLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDakMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssR0FBRyxDQUFDLENBQUM7SUFDeEIsQ0FBQztJQUVELE9BQU8sT0FBTyxDQUFDO0FBQ2hCLENBQUM7QUFFRCxNQUFNLFVBQVUsaUJBQWlCLENBQUMsUUFBcUIsRUFBRSxJQUFhLEVBQUUsT0FBdUI7SUFDOUYsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQ3hFLE9BQU8sT0FBTyxJQUFJLElBQUksSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0FBQ25ELENBQUM7QUFFRCxNQUFNLFVBQVUsaUJBQWlCLENBQUMsUUFBcUIsRUFBRSxJQUFhLEVBQUUsT0FBdUI7SUFDOUYsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQ3hFLE9BQU8sQ0FDTixPQUFPLElBQUksSUFBSTtRQUNmLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxLQUFLLE9BQU8sQ0FBQyxJQUFJLElBQUksUUFBUSxDQUFDLEtBQUssR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDO1lBQ2xFLENBQUMsUUFBUSxDQUFDLElBQUksR0FBRyxPQUFPLENBQUMsSUFBSSxJQUFJLE9BQU8sQ0FBQyxLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FDdkQsQ0FBQztBQUNILENBQUM7QUFFRCxNQUFNLFVBQVUsV0FBVyxDQUMxQixRQUFxQixFQUNyQixJQUFhLEVBQ2IsS0FBMEIsRUFDMUIsSUFBdUIsRUFDdkIsS0FBYztJQUVkLE1BQU0sRUFBRSxhQUFhLEVBQUUsTUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDO0lBQ3hDLHVDQUF1QztJQUN2QyxNQUFNLGFBQWEsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7SUFFdEQsb0RBQW9EO0lBQ3BELE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsYUFBYSxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUU7UUFDakUsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUM1RSxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQVEsSUFBSSxDQUFDO1FBRXRCLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUNaLE1BQU0sV0FBVyxHQUFHLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7WUFDMUYsbUNBQW1DO1lBQ25DLElBQUksV0FBVyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQ3hCLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxhQUFhLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNyRCxDQUFDO1FBQ0YsQ0FBQztRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ2xCLENBQUMsQ0FBQyxDQUFDO0lBRUgsMkJBQTJCO0lBQzNCLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxFQUFFLEVBQUU7UUFDbkMsSUFBSSxNQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUM7WUFDeEIsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLFVBQVUsQ0FBQyxRQUFRLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsYUFBYSxDQUFDLEtBQUssRUFBRSxJQUFLLEVBQXFCLENBQUMsQ0FBQztRQUMzRyxDQUFDO0lBQ0YsQ0FBQyxDQUFDLENBQUM7SUFFSCxPQUFPLE1BQU0sQ0FBQztBQUNmLENBQUM7QUFFRCxNQUFNLFVBQVUsVUFBVSxDQUN6QixRQUFxQixFQUNyQixJQUFhLEVBQ2IsS0FBMEIsRUFDMUIsSUFBdUIsRUFDdkIsUUFBd0IsRUFBb0I7SUFFNUMsTUFBTSxFQUNMLGVBQWUsRUFDZixPQUFPLEVBQ1AsT0FBTyxFQUNQLGNBQWMsRUFDZCxZQUFZLEVBQ1osV0FBVyxFQUNYLFlBQVksRUFDWixlQUFlLEdBQ2YsR0FBRyxLQUFLLENBQUM7SUFDVixNQUFNLGFBQWEsR0FBRyxRQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7SUFFMUMsS0FBSyxDQUFDLFNBQVMsR0FBUSxJQUFJLENBQUM7SUFDNUIsS0FBSyxDQUFDLFFBQVEsR0FBUSxJQUFJLENBQUM7SUFDM0IsS0FBSyxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQzFCLEtBQUssQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztJQUN2QixLQUFLLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLElBQUksRUFBRSxDQUFDO0lBQ2hDLEtBQUssQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUM7SUFFdEMsSUFBSSxHQUFHLGdCQUFnQixDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsY0FBYyxDQUFDLENBQUM7SUFFeEQsb0NBQW9DO0lBQ3BDLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztRQUN0QixLQUFLLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7SUFDM0IsQ0FBQztJQUVELGtCQUFrQjtJQUNsQixLQUFLLElBQUksSUFBSSxHQUFHLENBQUMsRUFBRSxJQUFJLEdBQUcsUUFBUSxDQUFDLGdCQUFnQixFQUFFLEVBQUUsSUFBSSxFQUFFLEVBQUUsQ0FBQztRQUMvRCxJQUFJLFVBQVUsR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ25DLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNqQixVQUFVLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLE1BQU0sRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7UUFDM0UsQ0FBQztRQUNELE1BQU0sSUFBSSxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUM7UUFFN0IsZ0JBQWdCO1FBQ2hCLEtBQUssSUFBSSxHQUFHLEdBQUcsQ0FBQyxFQUFFLEdBQUcsR0FBRyxRQUFRLENBQUMsY0FBYyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQztZQUMxRCxJQUFJLElBQUksS0FBSyxDQUFDLElBQUksZUFBZSxFQUFFLENBQUM7Z0JBQ25DLEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLFlBQVksQ0FBQyxDQUFDO1lBQ3JGLENBQUM7WUFFRCxNQUFNLE9BQU8sR0FBRyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQzdELE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7WUFFM0MsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUVoRCwyQkFBMkI7WUFDM0IsSUFBSSxRQUFRLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxJQUFJLE9BQU8sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQy9GLElBQUksQ0FBQyxRQUFRLElBQUksWUFBWSxFQUFFLENBQUM7Z0JBQy9CLFFBQVEsR0FBRyxZQUFZLENBQUMsT0FBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQzdFLENBQUM7WUFFRCxRQUFRO1lBQ1IsSUFBSSxLQUFLLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUUxQywyQ0FBMkM7WUFDM0MsSUFBSSxlQUFlLEdBQUcsZUFBZTtnQkFDcEMsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksRUFBRSxDQUFDO2dCQUNyRSxDQUFDLENBQUMsU0FBUyxDQUFDO1lBRWIsaUNBQWlDO1lBQ2pDLElBQUksS0FBSyxDQUFDLFNBQVMsS0FBSyxJQUFJLElBQUksT0FBTyxDQUFDLEtBQUssS0FBSyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7Z0JBQ2hFLEtBQUssQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDO1lBQzNCLENBQUM7WUFFRCxnQ0FBZ0M7WUFDaEMsSUFBSSxPQUFPLENBQUMsS0FBSyxLQUFLLEtBQUssQ0FBQyxNQUFNLElBQUksUUFBUSxDQUFDLEtBQUssS0FBSyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7Z0JBQ3ZFLEtBQUssQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDO1lBQzFCLENBQUM7WUFFRCxJQUFJLFNBQVMsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDMUIsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDO2dCQUNoQixTQUFTLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQWtCLENBQUM7WUFDNUMsQ0FBQztZQUNELFNBQVMsQ0FBQyxJQUFJLEdBQUcsT0FBTyxDQUFDO1lBQ3pCLFNBQVMsQ0FBQyxPQUFPLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsT0FBTyxJQUFJLEVBQUUsRUFBRTtnQkFDMUQsU0FBUyxFQUFFLE9BQU87Z0JBQ2xCLElBQUksRUFBRSxPQUFPO2dCQUNiLElBQUksRUFBRSxlQUFlO2dCQUNyQixZQUFZLEVBQUUsS0FBSyxDQUFDLE1BQU07Z0JBQzFCLFdBQVcsRUFBRSxLQUFLLENBQUMsSUFBSTtnQkFDdkIsUUFBUTtnQkFDUixPQUFPLEVBQUUsS0FBSztnQkFDZCxRQUFRLEVBQUUsS0FBSztnQkFDZixLQUFLO2FBQ0wsQ0FBQyxDQUFDO1lBQ0gsU0FBUyxDQUFDLFFBQVEsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUN4QixTQUFTLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztZQUNoQyxTQUFTLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztZQUV6QixJQUFJLEdBQUcsUUFBUSxDQUFDO1FBQ2pCLENBQUM7UUFFRCxVQUFVLENBQUMsTUFBTSxHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQ3pDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFDM0IsY0FBYyxDQUNkLENBQUM7UUFFRiw0QkFBNEI7UUFDNUIsVUFBVSxDQUFDLFNBQVM7WUFDbkIsV0FBVyxLQUFLLFdBQVc7Z0JBQzNCLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxLQUFLLEtBQUssQ0FBQyxNQUFNO2dCQUNuQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxLQUFLLEtBQUssQ0FBQyxNQUFNLENBQUM7SUFDcEQsQ0FBQztJQUVELE9BQU8sS0FBSyxDQUFDO0FBQ2QsQ0FBQztBQUVELE1BQU0sVUFBVSxnQkFBZ0IsQ0FBQyxRQUFxQixFQUFFLElBQWEsRUFBRSxjQUFzQjtJQUM1RixNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsY0FBYyxFQUFFLENBQUM7SUFDOUMsTUFBTSxjQUFjLEdBQUcsSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQzdELE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxVQUFVLENBQUMsY0FBYyxDQUFDLEdBQUcsV0FBVyxDQUFDO0lBQ3BFLE9BQU8sUUFBUSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsR0FBRyxFQUFFLENBQUMsV0FBVyxHQUFHLFNBQVMsR0FBRyxjQUFjLENBQUMsR0FBRyxXQUFXLENBQUMsQ0FBQztBQUN4RyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgTmdiRGF0ZSB9IGZyb20gJy4vbmdiLWRhdGUnO1xuaW1wb3J0IHsgRGF0ZXBpY2tlclZpZXdNb2RlbCwgRGF5Vmlld01vZGVsLCBNb250aFZpZXdNb2RlbCB9IGZyb20gJy4vZGF0ZXBpY2tlci12aWV3LW1vZGVsJztcbmltcG9ydCB7IE5nYkNhbGVuZGFyIH0gZnJvbSAnLi9uZ2ItY2FsZW5kYXInO1xuaW1wb3J0IHsgTmdiRGF0ZXBpY2tlckkxOG4gfSBmcm9tICcuL2RhdGVwaWNrZXItaTE4bic7XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0NoYW5nZWREYXRlKHByZXY/OiBOZ2JEYXRlIHwgbnVsbCwgbmV4dD86IE5nYkRhdGUgfCBudWxsKTogYm9vbGVhbiB7XG5cdHJldHVybiAhZGF0ZUNvbXBhcmF0b3IocHJldiwgbmV4dCk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0NoYW5nZWRNb250aChwcmV2PzogTmdiRGF0ZSB8IG51bGwsIG5leHQ/OiBOZ2JEYXRlIHwgbnVsbCk6IGJvb2xlYW4ge1xuXHRyZXR1cm4gIXByZXYgJiYgIW5leHQgPyBmYWxzZSA6ICFwcmV2IHx8ICFuZXh0ID8gdHJ1ZSA6IHByZXYueWVhciAhPT0gbmV4dC55ZWFyIHx8IHByZXYubW9udGggIT09IG5leHQubW9udGg7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBkYXRlQ29tcGFyYXRvcihwcmV2PzogTmdiRGF0ZSB8IG51bGwsIG5leHQ/OiBOZ2JEYXRlIHwgbnVsbCk6IGJvb2xlYW4ge1xuXHRyZXR1cm4gKCFwcmV2ICYmICFuZXh0KSB8fCAoISFwcmV2ICYmICEhbmV4dCAmJiBwcmV2LmVxdWFscyhuZXh0KSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjaGVja01pbkJlZm9yZU1heChtaW5EYXRlPzogTmdiRGF0ZSB8IG51bGwsIG1heERhdGU/OiBOZ2JEYXRlIHwgbnVsbCk6IHZvaWQge1xuXHRpZiAobWF4RGF0ZSAmJiBtaW5EYXRlICYmIG1heERhdGUuYmVmb3JlKG1pbkRhdGUpKSB7XG5cdFx0dGhyb3cgbmV3IEVycm9yKGAnbWF4RGF0ZScgJHttYXhEYXRlfSBzaG91bGQgYmUgZ3JlYXRlciB0aGFuICdtaW5EYXRlJyAke21pbkRhdGV9YCk7XG5cdH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNoZWNrRGF0ZUluUmFuZ2UoXG5cdGRhdGU/OiBOZ2JEYXRlIHwgbnVsbCxcblx0bWluRGF0ZT86IE5nYkRhdGUgfCBudWxsLFxuXHRtYXhEYXRlPzogTmdiRGF0ZSB8IG51bGwsXG4pOiBOZ2JEYXRlIHwgbnVsbCB7XG5cdGlmIChkYXRlICYmIG1pbkRhdGUgJiYgZGF0ZS5iZWZvcmUobWluRGF0ZSkpIHtcblx0XHRyZXR1cm4gbWluRGF0ZTtcblx0fVxuXHRpZiAoZGF0ZSAmJiBtYXhEYXRlICYmIGRhdGUuYWZ0ZXIobWF4RGF0ZSkpIHtcblx0XHRyZXR1cm4gbWF4RGF0ZTtcblx0fVxuXG5cdHJldHVybiBkYXRlIHx8IG51bGw7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0RhdGVTZWxlY3RhYmxlKGRhdGU6IE5nYkRhdGUgfCBudWxsIHwgdW5kZWZpbmVkLCBzdGF0ZTogRGF0ZXBpY2tlclZpZXdNb2RlbCkge1xuXHRjb25zdCB7IG1pbkRhdGUsIG1heERhdGUsIGRpc2FibGVkLCBtYXJrRGlzYWJsZWQgfSA9IHN0YXRlO1xuXHRyZXR1cm4gIShcblx0XHRkYXRlID09PSBudWxsIHx8XG5cdFx0ZGF0ZSA9PT0gdW5kZWZpbmVkIHx8XG5cdFx0ZGlzYWJsZWQgfHxcblx0XHQobWFya0Rpc2FibGVkICYmIG1hcmtEaXNhYmxlZChkYXRlLCB7IHllYXI6IGRhdGUueWVhciwgbW9udGg6IGRhdGUubW9udGggfSkpIHx8XG5cdFx0KG1pbkRhdGUgJiYgZGF0ZS5iZWZvcmUobWluRGF0ZSkpIHx8XG5cdFx0KG1heERhdGUgJiYgZGF0ZS5hZnRlcihtYXhEYXRlKSlcblx0KTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdlbmVyYXRlU2VsZWN0Qm94TW9udGhzKFxuXHRjYWxlbmRhcjogTmdiQ2FsZW5kYXIsXG5cdGRhdGU6IE5nYkRhdGUsXG5cdG1pbkRhdGU6IE5nYkRhdGUgfCBudWxsLFxuXHRtYXhEYXRlOiBOZ2JEYXRlIHwgbnVsbCxcbikge1xuXHRpZiAoIWRhdGUpIHtcblx0XHRyZXR1cm4gW107XG5cdH1cblxuXHRsZXQgbW9udGhzID0gY2FsZW5kYXIuZ2V0TW9udGhzKGRhdGUueWVhcik7XG5cblx0aWYgKG1pbkRhdGUgJiYgZGF0ZS55ZWFyID09PSBtaW5EYXRlLnllYXIpIHtcblx0XHRjb25zdCBpbmRleCA9IG1vbnRocy5maW5kSW5kZXgoKG1vbnRoKSA9PiBtb250aCA9PT0gbWluRGF0ZS5tb250aCk7XG5cdFx0bW9udGhzID0gbW9udGhzLnNsaWNlKGluZGV4KTtcblx0fVxuXG5cdGlmIChtYXhEYXRlICYmIGRhdGUueWVhciA9PT0gbWF4RGF0ZS55ZWFyKSB7XG5cdFx0Y29uc3QgaW5kZXggPSBtb250aHMuZmluZEluZGV4KChtb250aCkgPT4gbW9udGggPT09IG1heERhdGUubW9udGgpO1xuXHRcdG1vbnRocyA9IG1vbnRocy5zbGljZSgwLCBpbmRleCArIDEpO1xuXHR9XG5cblx0cmV0dXJuIG1vbnRocztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdlbmVyYXRlU2VsZWN0Qm94WWVhcnMoZGF0ZTogTmdiRGF0ZSwgbWluRGF0ZTogTmdiRGF0ZSB8IG51bGwsIG1heERhdGU6IE5nYkRhdGUgfCBudWxsKSB7XG5cdGlmICghZGF0ZSkge1xuXHRcdHJldHVybiBbXTtcblx0fVxuXG5cdGNvbnN0IHN0YXJ0ID0gbWluRGF0ZSA/IE1hdGgubWF4KG1pbkRhdGUueWVhciwgZGF0ZS55ZWFyIC0gNTAwKSA6IGRhdGUueWVhciAtIDEwO1xuXHRjb25zdCBlbmQgPSBtYXhEYXRlID8gTWF0aC5taW4obWF4RGF0ZS55ZWFyLCBkYXRlLnllYXIgKyA1MDApIDogZGF0ZS55ZWFyICsgMTA7XG5cblx0Y29uc3QgbGVuZ3RoID0gZW5kIC0gc3RhcnQgKyAxO1xuXHRjb25zdCBudW1iZXJzID0gQXJyYXkobGVuZ3RoKTtcblx0Zm9yIChsZXQgaSA9IDA7IGkgPCBsZW5ndGg7IGkrKykge1xuXHRcdG51bWJlcnNbaV0gPSBzdGFydCArIGk7XG5cdH1cblxuXHRyZXR1cm4gbnVtYmVycztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIG5leHRNb250aERpc2FibGVkKGNhbGVuZGFyOiBOZ2JDYWxlbmRhciwgZGF0ZTogTmdiRGF0ZSwgbWF4RGF0ZTogTmdiRGF0ZSB8IG51bGwpIHtcblx0Y29uc3QgbmV4dERhdGUgPSBPYmplY3QuYXNzaWduKGNhbGVuZGFyLmdldE5leHQoZGF0ZSwgJ20nKSwgeyBkYXk6IDEgfSk7XG5cdHJldHVybiBtYXhEYXRlICE9IG51bGwgJiYgbmV4dERhdGUuYWZ0ZXIobWF4RGF0ZSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBwcmV2TW9udGhEaXNhYmxlZChjYWxlbmRhcjogTmdiQ2FsZW5kYXIsIGRhdGU6IE5nYkRhdGUsIG1pbkRhdGU6IE5nYkRhdGUgfCBudWxsKSB7XG5cdGNvbnN0IHByZXZEYXRlID0gT2JqZWN0LmFzc2lnbihjYWxlbmRhci5nZXRQcmV2KGRhdGUsICdtJyksIHsgZGF5OiAxIH0pO1xuXHRyZXR1cm4gKFxuXHRcdG1pbkRhdGUgIT0gbnVsbCAmJlxuXHRcdCgocHJldkRhdGUueWVhciA9PT0gbWluRGF0ZS55ZWFyICYmIHByZXZEYXRlLm1vbnRoIDwgbWluRGF0ZS5tb250aCkgfHxcblx0XHRcdChwcmV2RGF0ZS55ZWFyIDwgbWluRGF0ZS55ZWFyICYmIG1pbkRhdGUubW9udGggPT09IDEpKVxuXHQpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRNb250aHMoXG5cdGNhbGVuZGFyOiBOZ2JDYWxlbmRhcixcblx0ZGF0ZTogTmdiRGF0ZSxcblx0c3RhdGU6IERhdGVwaWNrZXJWaWV3TW9kZWwsXG5cdGkxOG46IE5nYkRhdGVwaWNrZXJJMThuLFxuXHRmb3JjZTogYm9vbGVhbixcbik6IE1vbnRoVmlld01vZGVsW10ge1xuXHRjb25zdCB7IGRpc3BsYXlNb250aHMsIG1vbnRocyB9ID0gc3RhdGU7XG5cdC8vIG1vdmUgb2xkIG1vbnRocyB0byBhIHRlbXBvcmFyeSBhcnJheVxuXHRjb25zdCBtb250aHNUb1JldXNlID0gbW9udGhzLnNwbGljZSgwLCBtb250aHMubGVuZ3RoKTtcblxuXHQvLyBnZW5lcmF0ZSBuZXcgZmlyc3QgZGF0ZXMsIG51bGxpZnkgb3IgcmV1c2UgbW9udGhzXG5cdGNvbnN0IGZpcnN0RGF0ZXMgPSBBcnJheS5mcm9tKHsgbGVuZ3RoOiBkaXNwbGF5TW9udGhzIH0sIChfLCBpKSA9PiB7XG5cdFx0Y29uc3QgZmlyc3REYXRlID0gT2JqZWN0LmFzc2lnbihjYWxlbmRhci5nZXROZXh0KGRhdGUsICdtJywgaSksIHsgZGF5OiAxIH0pO1xuXHRcdG1vbnRoc1tpXSA9IDxhbnk+bnVsbDtcblxuXHRcdGlmICghZm9yY2UpIHtcblx0XHRcdGNvbnN0IHJldXNlZEluZGV4ID0gbW9udGhzVG9SZXVzZS5maW5kSW5kZXgoKG1vbnRoKSA9PiBtb250aC5maXJzdERhdGUuZXF1YWxzKGZpcnN0RGF0ZSkpO1xuXHRcdFx0Ly8gbW92ZSByZXVzZWQgbW9udGggYmFjayB0byBtb250aHNcblx0XHRcdGlmIChyZXVzZWRJbmRleCAhPT0gLTEpIHtcblx0XHRcdFx0bW9udGhzW2ldID0gbW9udGhzVG9SZXVzZS5zcGxpY2UocmV1c2VkSW5kZXgsIDEpWzBdO1xuXHRcdFx0fVxuXHRcdH1cblxuXHRcdHJldHVybiBmaXJzdERhdGU7XG5cdH0pO1xuXG5cdC8vIHJlYnVpbGQgbnVsbGlmaWVkIG1vbnRoc1xuXHRmaXJzdERhdGVzLmZvckVhY2goKGZpcnN0RGF0ZSwgaSkgPT4ge1xuXHRcdGlmIChtb250aHNbaV0gPT09IG51bGwpIHtcblx0XHRcdG1vbnRoc1tpXSA9IGJ1aWxkTW9udGgoY2FsZW5kYXIsIGZpcnN0RGF0ZSwgc3RhdGUsIGkxOG4sIG1vbnRoc1RvUmV1c2Uuc2hpZnQoKSB8fCAoe30gYXMgTW9udGhWaWV3TW9kZWwpKTtcblx0XHR9XG5cdH0pO1xuXG5cdHJldHVybiBtb250aHM7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBidWlsZE1vbnRoKFxuXHRjYWxlbmRhcjogTmdiQ2FsZW5kYXIsXG5cdGRhdGU6IE5nYkRhdGUsXG5cdHN0YXRlOiBEYXRlcGlja2VyVmlld01vZGVsLFxuXHRpMThuOiBOZ2JEYXRlcGlja2VySTE4bixcblx0bW9udGg6IE1vbnRoVmlld01vZGVsID0ge30gYXMgTW9udGhWaWV3TW9kZWwsXG4pOiBNb250aFZpZXdNb2RlbCB7XG5cdGNvbnN0IHtcblx0XHRkYXlUZW1wbGF0ZURhdGEsXG5cdFx0bWluRGF0ZSxcblx0XHRtYXhEYXRlLFxuXHRcdGZpcnN0RGF5T2ZXZWVrLFxuXHRcdG1hcmtEaXNhYmxlZCxcblx0XHRvdXRzaWRlRGF5cyxcblx0XHR3ZWVrZGF5V2lkdGgsXG5cdFx0d2Vla2RheXNWaXNpYmxlLFxuXHR9ID0gc3RhdGU7XG5cdGNvbnN0IGNhbGVuZGFyVG9kYXkgPSBjYWxlbmRhci5nZXRUb2RheSgpO1xuXG5cdG1vbnRoLmZpcnN0RGF0ZSA9IDxhbnk+bnVsbDtcblx0bW9udGgubGFzdERhdGUgPSA8YW55Pm51bGw7XG5cdG1vbnRoLm51bWJlciA9IGRhdGUubW9udGg7XG5cdG1vbnRoLnllYXIgPSBkYXRlLnllYXI7XG5cdG1vbnRoLndlZWtzID0gbW9udGgud2Vla3MgfHwgW107XG5cdG1vbnRoLndlZWtkYXlzID0gbW9udGgud2Vla2RheXMgfHwgW107XG5cblx0ZGF0ZSA9IGdldEZpcnN0Vmlld0RhdGUoY2FsZW5kYXIsIGRhdGUsIGZpcnN0RGF5T2ZXZWVrKTtcblxuXHQvLyBjbGVhcmluZyB3ZWVrZGF5cywgaWYgbm90IHZpc2libGVcblx0aWYgKCF3ZWVrZGF5c1Zpc2libGUpIHtcblx0XHRtb250aC53ZWVrZGF5cy5sZW5ndGggPSAwO1xuXHR9XG5cblx0Ly8gbW9udGggaGFzIHdlZWtzXG5cdGZvciAobGV0IHdlZWsgPSAwOyB3ZWVrIDwgY2FsZW5kYXIuZ2V0V2Vla3NQZXJNb250aCgpOyB3ZWVrKyspIHtcblx0XHRsZXQgd2Vla09iamVjdCA9IG1vbnRoLndlZWtzW3dlZWtdO1xuXHRcdGlmICghd2Vla09iamVjdCkge1xuXHRcdFx0d2Vla09iamVjdCA9IG1vbnRoLndlZWtzW3dlZWtdID0geyBudW1iZXI6IDAsIGRheXM6IFtdLCBjb2xsYXBzZWQ6IHRydWUgfTtcblx0XHR9XG5cdFx0Y29uc3QgZGF5cyA9IHdlZWtPYmplY3QuZGF5cztcblxuXHRcdC8vIHdlZWsgaGFzIGRheXNcblx0XHRmb3IgKGxldCBkYXkgPSAwOyBkYXkgPCBjYWxlbmRhci5nZXREYXlzUGVyV2VlaygpOyBkYXkrKykge1xuXHRcdFx0aWYgKHdlZWsgPT09IDAgJiYgd2Vla2RheXNWaXNpYmxlKSB7XG5cdFx0XHRcdG1vbnRoLndlZWtkYXlzW2RheV0gPSBpMThuLmdldFdlZWtkYXlMYWJlbChjYWxlbmRhci5nZXRXZWVrZGF5KGRhdGUpLCB3ZWVrZGF5V2lkdGgpO1xuXHRcdFx0fVxuXG5cdFx0XHRjb25zdCBuZXdEYXRlID0gbmV3IE5nYkRhdGUoZGF0ZS55ZWFyLCBkYXRlLm1vbnRoLCBkYXRlLmRheSk7XG5cdFx0XHRjb25zdCBuZXh0RGF0ZSA9IGNhbGVuZGFyLmdldE5leHQobmV3RGF0ZSk7XG5cblx0XHRcdGNvbnN0IGFyaWFMYWJlbCA9IGkxOG4uZ2V0RGF5QXJpYUxhYmVsKG5ld0RhdGUpO1xuXG5cdFx0XHQvLyBtYXJraW5nIGRhdGUgYXMgZGlzYWJsZWRcblx0XHRcdGxldCBkaXNhYmxlZCA9ICEhKChtaW5EYXRlICYmIG5ld0RhdGUuYmVmb3JlKG1pbkRhdGUpKSB8fCAobWF4RGF0ZSAmJiBuZXdEYXRlLmFmdGVyKG1heERhdGUpKSk7XG5cdFx0XHRpZiAoIWRpc2FibGVkICYmIG1hcmtEaXNhYmxlZCkge1xuXHRcdFx0XHRkaXNhYmxlZCA9IG1hcmtEaXNhYmxlZChuZXdEYXRlLCB7IG1vbnRoOiBtb250aC5udW1iZXIsIHllYXI6IG1vbnRoLnllYXIgfSk7XG5cdFx0XHR9XG5cblx0XHRcdC8vIHRvZGF5XG5cdFx0XHRsZXQgdG9kYXkgPSBuZXdEYXRlLmVxdWFscyhjYWxlbmRhclRvZGF5KTtcblxuXHRcdFx0Ly8gYWRkaW5nIHVzZXItcHJvdmlkZWQgZGF0YSB0byB0aGUgY29udGV4dFxuXHRcdFx0bGV0IGNvbnRleHRVc2VyRGF0YSA9IGRheVRlbXBsYXRlRGF0YVxuXHRcdFx0XHQ/IGRheVRlbXBsYXRlRGF0YShuZXdEYXRlLCB7IG1vbnRoOiBtb250aC5udW1iZXIsIHllYXI6IG1vbnRoLnllYXIgfSlcblx0XHRcdFx0OiB1bmRlZmluZWQ7XG5cblx0XHRcdC8vIHNhdmluZyBmaXJzdCBkYXRlIG9mIHRoZSBtb250aFxuXHRcdFx0aWYgKG1vbnRoLmZpcnN0RGF0ZSA9PT0gbnVsbCAmJiBuZXdEYXRlLm1vbnRoID09PSBtb250aC5udW1iZXIpIHtcblx0XHRcdFx0bW9udGguZmlyc3REYXRlID0gbmV3RGF0ZTtcblx0XHRcdH1cblxuXHRcdFx0Ly8gc2F2aW5nIGxhc3QgZGF0ZSBvZiB0aGUgbW9udGhcblx0XHRcdGlmIChuZXdEYXRlLm1vbnRoID09PSBtb250aC5udW1iZXIgJiYgbmV4dERhdGUubW9udGggIT09IG1vbnRoLm51bWJlcikge1xuXHRcdFx0XHRtb250aC5sYXN0RGF0ZSA9IG5ld0RhdGU7XG5cdFx0XHR9XG5cblx0XHRcdGxldCBkYXlPYmplY3QgPSBkYXlzW2RheV07XG5cdFx0XHRpZiAoIWRheU9iamVjdCkge1xuXHRcdFx0XHRkYXlPYmplY3QgPSBkYXlzW2RheV0gPSB7fSBhcyBEYXlWaWV3TW9kZWw7XG5cdFx0XHR9XG5cdFx0XHRkYXlPYmplY3QuZGF0ZSA9IG5ld0RhdGU7XG5cdFx0XHRkYXlPYmplY3QuY29udGV4dCA9IE9iamVjdC5hc3NpZ24oZGF5T2JqZWN0LmNvbnRleHQgfHwge30sIHtcblx0XHRcdFx0JGltcGxpY2l0OiBuZXdEYXRlLFxuXHRcdFx0XHRkYXRlOiBuZXdEYXRlLFxuXHRcdFx0XHRkYXRhOiBjb250ZXh0VXNlckRhdGEsXG5cdFx0XHRcdGN1cnJlbnRNb250aDogbW9udGgubnVtYmVyLFxuXHRcdFx0XHRjdXJyZW50WWVhcjogbW9udGgueWVhcixcblx0XHRcdFx0ZGlzYWJsZWQsXG5cdFx0XHRcdGZvY3VzZWQ6IGZhbHNlLFxuXHRcdFx0XHRzZWxlY3RlZDogZmFsc2UsXG5cdFx0XHRcdHRvZGF5LFxuXHRcdFx0fSk7XG5cdFx0XHRkYXlPYmplY3QudGFiaW5kZXggPSAtMTtcblx0XHRcdGRheU9iamVjdC5hcmlhTGFiZWwgPSBhcmlhTGFiZWw7XG5cdFx0XHRkYXlPYmplY3QuaGlkZGVuID0gZmFsc2U7XG5cblx0XHRcdGRhdGUgPSBuZXh0RGF0ZTtcblx0XHR9XG5cblx0XHR3ZWVrT2JqZWN0Lm51bWJlciA9IGNhbGVuZGFyLmdldFdlZWtOdW1iZXIoXG5cdFx0XHRkYXlzLm1hcCgoZGF5KSA9PiBkYXkuZGF0ZSksXG5cdFx0XHRmaXJzdERheU9mV2Vlayxcblx0XHQpO1xuXG5cdFx0Ly8gbWFya2luZyB3ZWVrIGFzIGNvbGxhcHNlZFxuXHRcdHdlZWtPYmplY3QuY29sbGFwc2VkID1cblx0XHRcdG91dHNpZGVEYXlzID09PSAnY29sbGFwc2VkJyAmJlxuXHRcdFx0ZGF5c1swXS5kYXRlLm1vbnRoICE9PSBtb250aC5udW1iZXIgJiZcblx0XHRcdGRheXNbZGF5cy5sZW5ndGggLSAxXS5kYXRlLm1vbnRoICE9PSBtb250aC5udW1iZXI7XG5cdH1cblxuXHRyZXR1cm4gbW9udGg7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRGaXJzdFZpZXdEYXRlKGNhbGVuZGFyOiBOZ2JDYWxlbmRhciwgZGF0ZTogTmdiRGF0ZSwgZmlyc3REYXlPZldlZWs6IG51bWJlcik6IE5nYkRhdGUge1xuXHRjb25zdCBkYXlzUGVyV2VlayA9IGNhbGVuZGFyLmdldERheXNQZXJXZWVrKCk7XG5cdGNvbnN0IGZpcnN0TW9udGhEYXRlID0gbmV3IE5nYkRhdGUoZGF0ZS55ZWFyLCBkYXRlLm1vbnRoLCAxKTtcblx0Y29uc3QgZGF5T2ZXZWVrID0gY2FsZW5kYXIuZ2V0V2Vla2RheShmaXJzdE1vbnRoRGF0ZSkgJSBkYXlzUGVyV2Vlaztcblx0cmV0dXJuIGNhbGVuZGFyLmdldFByZXYoZmlyc3RNb250aERhdGUsICdkJywgKGRheXNQZXJXZWVrICsgZGF5T2ZXZWVrIC0gZmlyc3REYXlPZldlZWspICUgZGF5c1BlcldlZWspO1xufVxuIl19