import { NgbDate } from '../ngb-date';
const PARTS_PER_HOUR = 1080;
const PARTS_PER_DAY = 24 * PARTS_PER_HOUR;
const PARTS_FRACTIONAL_MONTH = 12 * PARTS_PER_HOUR + 793;
const PARTS_PER_MONTH = 29 * PARTS_PER_DAY + PARTS_FRACTIONAL_MONTH;
const BAHARAD = 11 * PARTS_PER_HOUR + 204;
const HEBREW_DAY_ON_JAN_1_1970 = 2092591;
const GREGORIAN_EPOCH = 1721425.5;
function isGregorianLeapYear(year) {
    return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
}
function numberOfFirstDayInYear(year) {
    let monthsBeforeYear = Math.floor((235 * year - 234) / 19);
    let fractionalMonthsBeforeYear = monthsBeforeYear * PARTS_FRACTIONAL_MONTH + BAHARAD;
    let dayNumber = monthsBeforeYear * 29 + Math.floor(fractionalMonthsBeforeYear / PARTS_PER_DAY);
    let timeOfDay = fractionalMonthsBeforeYear % PARTS_PER_DAY;
    let dayOfWeek = dayNumber % 7; // 0 == Monday
    if (dayOfWeek === 2 || dayOfWeek === 4 || dayOfWeek === 6) {
        dayNumber++;
        dayOfWeek = dayNumber % 7;
    }
    if (dayOfWeek === 1 && timeOfDay > 15 * PARTS_PER_HOUR + 204 && !isHebrewLeapYear(year)) {
        dayNumber += 2;
    }
    else if (dayOfWeek === 0 && timeOfDay > 21 * PARTS_PER_HOUR + 589 && isHebrewLeapYear(year - 1)) {
        dayNumber++;
    }
    return dayNumber;
}
function getDaysInGregorianMonth(month, year) {
    let days = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    if (isGregorianLeapYear(year)) {
        days[1]++;
    }
    return days[month - 1];
}
function getHebrewMonths(year) {
    return isHebrewLeapYear(year) ? 13 : 12;
}
/**
 * Returns the number of days in a specific Hebrew year.
 * `year` is any Hebrew year.
 */
function getDaysInHebrewYear(year) {
    return numberOfFirstDayInYear(year + 1) - numberOfFirstDayInYear(year);
}
export function isHebrewLeapYear(year) {
    if (year != null) {
        let b = (year * 12 + 17) % 19;
        return b >= (b < 0 ? -7 : 12);
    }
    return false;
}
/**
 * Returns the number of days in a specific Hebrew month.
 * `month` is 1 for Nisan, 2 for Iyar etc. Note: Hebrew leap year contains 13 months.
 * `year` is any Hebrew year.
 */
export function getDaysInHebrewMonth(month, year) {
    let yearLength = numberOfFirstDayInYear(year + 1) - numberOfFirstDayInYear(year);
    let yearType = (yearLength <= 380 ? yearLength : yearLength - 30) - 353;
    let leapYear = isHebrewLeapYear(year);
    let daysInMonth = leapYear
        ? [30, 29, 29, 29, 30, 30, 29, 30, 29, 30, 29, 30, 29]
        : [30, 29, 29, 29, 30, 29, 30, 29, 30, 29, 30, 29];
    if (yearType > 0) {
        daysInMonth[2]++; // Kislev gets an extra day in normal or complete years.
    }
    if (yearType > 1) {
        daysInMonth[1]++; // Heshvan gets an extra day in complete years only.
    }
    return daysInMonth[month - 1];
}
export function getDayNumberInHebrewYear(date) {
    let numberOfDay = 0;
    for (let i = 1; i < date.month; i++) {
        numberOfDay += getDaysInHebrewMonth(i, date.year);
    }
    return numberOfDay + date.day;
}
export function setHebrewMonth(date, val) {
    let after = val >= 0;
    if (!after) {
        val = -val;
    }
    while (val > 0) {
        if (after) {
            if (val > getHebrewMonths(date.year) - date.month) {
                val -= getHebrewMonths(date.year) - date.month + 1;
                date.year++;
                date.month = 1;
            }
            else {
                date.month += val;
                val = 0;
            }
        }
        else {
            if (val >= date.month) {
                date.year--;
                val -= date.month;
                date.month = getHebrewMonths(date.year);
            }
            else {
                date.month -= val;
                val = 0;
            }
        }
    }
    return date;
}
export function setHebrewDay(date, val) {
    let after = val >= 0;
    if (!after) {
        val = -val;
    }
    while (val > 0) {
        if (after) {
            if (val > getDaysInHebrewYear(date.year) - getDayNumberInHebrewYear(date)) {
                val -= getDaysInHebrewYear(date.year) - getDayNumberInHebrewYear(date) + 1;
                date.year++;
                date.month = 1;
                date.day = 1;
            }
            else if (val > getDaysInHebrewMonth(date.month, date.year) - date.day) {
                val -= getDaysInHebrewMonth(date.month, date.year) - date.day + 1;
                date.month++;
                date.day = 1;
            }
            else {
                date.day += val;
                val = 0;
            }
        }
        else {
            if (val >= date.day) {
                val -= date.day;
                date.month--;
                if (date.month === 0) {
                    date.year--;
                    date.month = getHebrewMonths(date.year);
                }
                date.day = getDaysInHebrewMonth(date.month, date.year);
            }
            else {
                date.day -= val;
                val = 0;
            }
        }
    }
    return date;
}
/**
 * Returns the equivalent Hebrew date value for a give input Gregorian date.
 * `gdate` is a JS Date to be converted to Hebrew date.
 */
export function fromGregorian(gdate) {
    const date = new Date(gdate);
    const gYear = date.getFullYear(), gMonth = date.getMonth(), gDay = date.getDate();
    let julianDay = GREGORIAN_EPOCH -
        1 +
        365 * (gYear - 1) +
        Math.floor((gYear - 1) / 4) -
        Math.floor((gYear - 1) / 100) +
        Math.floor((gYear - 1) / 400) +
        Math.floor((367 * (gMonth + 1) - 362) / 12 + (gMonth + 1 <= 2 ? 0 : isGregorianLeapYear(gYear) ? -1 : -2) + gDay);
    julianDay = Math.floor(julianDay + 0.5);
    let daysSinceHebEpoch = julianDay - 347997;
    let monthsSinceHebEpoch = Math.floor((daysSinceHebEpoch * PARTS_PER_DAY) / PARTS_PER_MONTH);
    let hYear = Math.floor((monthsSinceHebEpoch * 19 + 234) / 235) + 1;
    let firstDayOfThisYear = numberOfFirstDayInYear(hYear);
    let dayOfYear = daysSinceHebEpoch - firstDayOfThisYear;
    while (dayOfYear < 1) {
        hYear--;
        firstDayOfThisYear = numberOfFirstDayInYear(hYear);
        dayOfYear = daysSinceHebEpoch - firstDayOfThisYear;
    }
    let hMonth = 1;
    let hDay = dayOfYear;
    while (hDay > getDaysInHebrewMonth(hMonth, hYear)) {
        hDay -= getDaysInHebrewMonth(hMonth, hYear);
        hMonth++;
    }
    return new NgbDate(hYear, hMonth, hDay);
}
/**
 * Returns the equivalent JS date value for a given Hebrew date.
 * `hebrewDate` is an Hebrew date to be converted to Gregorian.
 */
export function toGregorian(hebrewDate) {
    const hYear = hebrewDate.year;
    const hMonth = hebrewDate.month;
    const hDay = hebrewDate.day;
    let days = numberOfFirstDayInYear(hYear);
    for (let i = 1; i < hMonth; i++) {
        days += getDaysInHebrewMonth(i, hYear);
    }
    days += hDay;
    let diffDays = days - HEBREW_DAY_ON_JAN_1_1970;
    let after = diffDays >= 0;
    if (!after) {
        diffDays = -diffDays;
    }
    let gYear = 1970;
    let gMonth = 1;
    let gDay = 1;
    while (diffDays > 0) {
        if (after) {
            if (diffDays >= (isGregorianLeapYear(gYear) ? 366 : 365)) {
                diffDays -= isGregorianLeapYear(gYear) ? 366 : 365;
                gYear++;
            }
            else if (diffDays >= getDaysInGregorianMonth(gMonth, gYear)) {
                diffDays -= getDaysInGregorianMonth(gMonth, gYear);
                gMonth++;
            }
            else {
                gDay += diffDays;
                diffDays = 0;
            }
        }
        else {
            if (diffDays >= (isGregorianLeapYear(gYear - 1) ? 366 : 365)) {
                diffDays -= isGregorianLeapYear(gYear - 1) ? 366 : 365;
                gYear--;
            }
            else {
                if (gMonth > 1) {
                    gMonth--;
                }
                else {
                    gMonth = 12;
                    gYear--;
                }
                if (diffDays >= getDaysInGregorianMonth(gMonth, gYear)) {
                    diffDays -= getDaysInGregorianMonth(gMonth, gYear);
                }
                else {
                    gDay = getDaysInGregorianMonth(gMonth, gYear) - diffDays + 1;
                    diffDays = 0;
                }
            }
        }
    }
    return new Date(gYear, gMonth - 1, gDay);
}
export function hebrewNumerals(numerals) {
    if (!numerals) {
        return '';
    }
    const hArray0_9 = ['', '\u05d0', '\u05d1', '\u05d2', '\u05d3', '\u05d4', '\u05d5', '\u05d6', '\u05d7', '\u05d8'];
    const hArray10_19 = [
        '\u05d9',
        '\u05d9\u05d0',
        '\u05d9\u05d1',
        '\u05d9\u05d2',
        '\u05d9\u05d3',
        '\u05d8\u05d5',
        '\u05d8\u05d6',
        '\u05d9\u05d6',
        '\u05d9\u05d7',
        '\u05d9\u05d8',
    ];
    const hArray20_90 = ['', '', '\u05db', '\u05dc', '\u05de', '\u05e0', '\u05e1', '\u05e2', '\u05e4', '\u05e6'];
    const hArray100_900 = [
        '',
        '\u05e7',
        '\u05e8',
        '\u05e9',
        '\u05ea',
        '\u05ea\u05e7',
        '\u05ea\u05e8',
        '\u05ea\u05e9',
        '\u05ea\u05ea',
        '\u05ea\u05ea\u05e7',
    ];
    const hArray1000_9000 = [
        '',
        '\u05d0',
        '\u05d1',
        '\u05d1\u05d0',
        '\u05d1\u05d1',
        '\u05d4',
        '\u05d4\u05d0',
        '\u05d4\u05d1',
        '\u05d4\u05d1\u05d0',
        '\u05d4\u05d1\u05d1',
    ];
    const geresh = '\u05f3', gershaim = '\u05f4';
    let mem = 0;
    let result = [];
    let step = 0;
    while (numerals > 0) {
        let m = numerals % 10;
        if (step === 0) {
            mem = m;
        }
        else if (step === 1) {
            if (m !== 1) {
                result.unshift(hArray20_90[m], hArray0_9[mem]);
            }
            else {
                result.unshift(hArray10_19[mem]);
            }
        }
        else if (step === 2) {
            result.unshift(hArray100_900[m]);
        }
        else {
            if (m !== 5) {
                result.unshift(hArray1000_9000[m], geresh, ' ');
            }
            break;
        }
        numerals = Math.floor(numerals / 10);
        if (step === 0 && numerals === 0) {
            result.unshift(hArray0_9[m]);
        }
        step++;
    }
    result = result.join('').split('');
    if (result.length === 1) {
        result.push(geresh);
    }
    else if (result.length > 1) {
        result.splice(result.length - 1, 0, gershaim);
    }
    return result.join('');
}
//# sourceMappingURL=data:application/json;base64,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