import { NgbCalendarIslamicCivil } from './ngb-calendar-islamic-civil';
import { NgbDate } from '../ngb-date';
import { Injectable } from '@angular/core';
import * as i0 from "@angular/core";
/**
 * Umalqura calendar is one type of Hijri calendars used in islamic countries.
 * This Calendar is used by Saudi Arabia for administrative purpose.
 * Unlike tabular calendars, the algorithm involves astronomical calculation, but it's still deterministic.
 * http://cldr.unicode.org/development/development-process/design-proposals/islamic-calendar-types
 */
const GREGORIAN_FIRST_DATE = new Date(1882, 10, 12);
const GREGORIAN_LAST_DATE = new Date(2174, 10, 25);
const HIJRI_BEGIN = 1300;
const HIJRI_END = 1600;
const ONE_DAY = 1000 * 60 * 60 * 24;
const MONTH_LENGTH = [
    // 1300-1304
    '101010101010',
    '110101010100',
    '111011001001',
    '011011010100',
    '011011101010',
    // 1305-1309
    '001101101100',
    '101010101101',
    '010101010101',
    '011010101001',
    '011110010010',
    // 1310-1314
    '101110101001',
    '010111010100',
    '101011011010',
    '010101011100',
    '110100101101',
    // 1315-1319
    '011010010101',
    '011101001010',
    '101101010100',
    '101101101010',
    '010110101101',
    // 1320-1324
    '010010101110',
    '101001001111',
    '010100010111',
    '011010001011',
    '011010100101',
    // 1325-1329
    '101011010101',
    '001011010110',
    '100101011011',
    '010010011101',
    '101001001101',
    // 1330-1334
    '110100100110',
    '110110010101',
    '010110101100',
    '100110110110',
    '001010111010',
    // 1335-1339
    '101001011011',
    '010100101011',
    '101010010101',
    '011011001010',
    '101011101001',
    // 1340-1344
    '001011110100',
    '100101110110',
    '001010110110',
    '100101010110',
    '101011001010',
    // 1345-1349
    '101110100100',
    '101111010010',
    '010111011001',
    '001011011100',
    '100101101101',
    // 1350-1354
    '010101001101',
    '101010100101',
    '101101010010',
    '101110100101',
    '010110110100',
    // 1355-1359
    '100110110110',
    '010101010111',
    '001010010111',
    '010101001011',
    '011010100011',
    // 1360-1364
    '011101010010',
    '101101100101',
    '010101101010',
    '101010101011',
    '010100101011',
    // 1365-1369
    '110010010101',
    '110101001010',
    '110110100101',
    '010111001010',
    '101011010110',
    // 1370-1374
    '100101010111',
    '010010101011',
    '100101001011',
    '101010100101',
    '101101010010',
    // 1375-1379
    '101101101010',
    '010101110101',
    '001001110110',
    '100010110111',
    '010001011011',
    // 1380-1384
    '010101010101',
    '010110101001',
    '010110110100',
    '100111011010',
    '010011011101',
    // 1385-1389
    '001001101110',
    '100100110110',
    '101010101010',
    '110101010100',
    '110110110010',
    // 1390-1394
    '010111010101',
    '001011011010',
    '100101011011',
    '010010101011',
    '101001010101',
    // 1395-1399
    '101101001001',
    '101101100100',
    '101101110001',
    '010110110100',
    '101010110101',
    // 1400-1404
    '101001010101',
    '110100100101',
    '111010010010',
    '111011001001',
    '011011010100',
    // 1405-1409
    '101011101001',
    '100101101011',
    '010010101011',
    '101010010011',
    '110101001001',
    // 1410-1414
    '110110100100',
    '110110110010',
    '101010111001',
    '010010111010',
    '101001011011',
    // 1415-1419
    '010100101011',
    '101010010101',
    '101100101010',
    '101101010101',
    '010101011100',
    // 1420-1424
    '010010111101',
    '001000111101',
    '100100011101',
    '101010010101',
    '101101001010',
    // 1425-1429
    '101101011010',
    '010101101101',
    '001010110110',
    '100100111011',
    '010010011011',
    // 1430-1434
    '011001010101',
    '011010101001',
    '011101010100',
    '101101101010',
    '010101101100',
    // 1435-1439
    '101010101101',
    '010101010101',
    '101100101001',
    '101110010010',
    '101110101001',
    // 1440-1444
    '010111010100',
    '101011011010',
    '010101011010',
    '101010101011',
    '010110010101',
    // 1445-1449
    '011101001001',
    '011101100100',
    '101110101010',
    '010110110101',
    '001010110110',
    // 1450-1454
    '101001010110',
    '111001001101',
    '101100100101',
    '101101010010',
    '101101101010',
    // 1455-1459
    '010110101101',
    '001010101110',
    '100100101111',
    '010010010111',
    '011001001011',
    // 1460-1464
    '011010100101',
    '011010101100',
    '101011010110',
    '010101011101',
    '010010011101',
    // 1465-1469
    '101001001101',
    '110100010110',
    '110110010101',
    '010110101010',
    '010110110101',
    // 1470-1474
    '001011011010',
    '100101011011',
    '010010101101',
    '010110010101',
    '011011001010',
    // 1475-1479
    '011011100100',
    '101011101010',
    '010011110101',
    '001010110110',
    '100101010110',
    // 1480-1484
    '101010101010',
    '101101010100',
    '101111010010',
    '010111011001',
    '001011101010',
    // 1485-1489
    '100101101101',
    '010010101101',
    '101010010101',
    '101101001010',
    '101110100101',
    // 1490-1494
    '010110110010',
    '100110110101',
    '010011010110',
    '101010010111',
    '010101000111',
    // 1495-1499
    '011010010011',
    '011101001001',
    '101101010101',
    '010101101010',
    '101001101011',
    // 1500-1504
    '010100101011',
    '101010001011',
    '110101000110',
    '110110100011',
    '010111001010',
    // 1505-1509
    '101011010110',
    '010011011011',
    '001001101011',
    '100101001011',
    '101010100101',
    // 1510-1514
    '101101010010',
    '101101101001',
    '010101110101',
    '000101110110',
    '100010110111',
    // 1515-1519
    '001001011011',
    '010100101011',
    '010101100101',
    '010110110100',
    '100111011010',
    // 1520-1524
    '010011101101',
    '000101101101',
    '100010110110',
    '101010100110',
    '110101010010',
    // 1525-1529
    '110110101001',
    '010111010100',
    '101011011010',
    '100101011011',
    '010010101011',
    // 1530-1534
    '011001010011',
    '011100101001',
    '011101100010',
    '101110101001',
    '010110110010',
    // 1535-1539
    '101010110101',
    '010101010101',
    '101100100101',
    '110110010010',
    '111011001001',
    // 1540-1544
    '011011010010',
    '101011101001',
    '010101101011',
    '010010101011',
    '101001010101',
    // 1545-1549
    '110100101001',
    '110101010100',
    '110110101010',
    '100110110101',
    '010010111010',
    // 1550-1554
    '101000111011',
    '010010011011',
    '101001001101',
    '101010101010',
    '101011010101',
    // 1555-1559
    '001011011010',
    '100101011101',
    '010001011110',
    '101000101110',
    '110010011010',
    // 1560-1564
    '110101010101',
    '011010110010',
    '011010111001',
    '010010111010',
    '101001011101',
    // 1565-1569
    '010100101101',
    '101010010101',
    '101101010010',
    '101110101000',
    '101110110100',
    // 1570-1574
    '010110111001',
    '001011011010',
    '100101011010',
    '101101001010',
    '110110100100',
    // 1575-1579
    '111011010001',
    '011011101000',
    '101101101010',
    '010101101101',
    '010100110101',
    // 1580-1584
    '011010010101',
    '110101001010',
    '110110101000',
    '110111010100',
    '011011011010',
    // 1585-1589
    '010101011011',
    '001010011101',
    '011000101011',
    '101100010101',
    '101101001010',
    // 1590-1594
    '101110010101',
    '010110101010',
    '101010101110',
    '100100101110',
    '110010001111',
    // 1595-1599
    '010100100111',
    '011010010101',
    '011010101010',
    '101011010110',
    '010101011101',
    // 1600
    '001010011101',
];
function getDaysDiff(date1, date2) {
    // Ignores the time part in date1 and date2:
    const time1 = Date.UTC(date1.getFullYear(), date1.getMonth(), date1.getDate());
    const time2 = Date.UTC(date2.getFullYear(), date2.getMonth(), date2.getDate());
    const diff = Math.abs(time1 - time2);
    return Math.round(diff / ONE_DAY);
}
export class NgbCalendarIslamicUmalqura extends NgbCalendarIslamicCivil {
    /**
     * Returns the equivalent islamic(Umalqura) date value for a give input Gregorian date.
     * `gdate` is s JS Date to be converted to Hijri.
     */
    fromGregorian(gDate) {
        let hDay = 1, hMonth = 0, hYear = 1300;
        let daysDiff = getDaysDiff(gDate, GREGORIAN_FIRST_DATE);
        if (gDate.getTime() - GREGORIAN_FIRST_DATE.getTime() >= 0 && gDate.getTime() - GREGORIAN_LAST_DATE.getTime() <= 0) {
            let year = 1300;
            for (let i = 0; i < MONTH_LENGTH.length; i++, year++) {
                for (let j = 0; j < 12; j++) {
                    let numOfDays = +MONTH_LENGTH[i][j] + 29;
                    if (daysDiff <= numOfDays) {
                        hDay = daysDiff + 1;
                        if (hDay > numOfDays) {
                            hDay = 1;
                            j++;
                        }
                        if (j > 11) {
                            j = 0;
                            year++;
                        }
                        hMonth = j;
                        hYear = year;
                        return new NgbDate(hYear, hMonth + 1, hDay);
                    }
                    daysDiff = daysDiff - numOfDays;
                }
            }
            return null;
        }
        else {
            return super.fromGregorian(gDate);
        }
    }
    /**
     * Converts the current Hijri date to Gregorian.
     */
    toGregorian(hDate) {
        const hYear = hDate.year;
        const hMonth = hDate.month - 1;
        const hDay = hDate.day;
        let gDate = new Date(GREGORIAN_FIRST_DATE);
        let dayDiff = hDay - 1;
        if (hYear >= HIJRI_BEGIN && hYear <= HIJRI_END) {
            for (let y = 0; y < hYear - HIJRI_BEGIN; y++) {
                for (let m = 0; m < 12; m++) {
                    dayDiff += +MONTH_LENGTH[y][m] + 29;
                }
            }
            for (let m = 0; m < hMonth; m++) {
                dayDiff += +MONTH_LENGTH[hYear - HIJRI_BEGIN][m] + 29;
            }
            gDate.setDate(GREGORIAN_FIRST_DATE.getDate() + dayDiff);
        }
        else {
            gDate = super.toGregorian(hDate);
        }
        return gDate;
    }
    /**
     * Returns the number of days in a specific Hijri hMonth.
     * `hMonth` is 1 for Muharram, 2 for Safar, etc.
     * `hYear` is any Hijri hYear.
     */
    getDaysPerMonth(hMonth, hYear) {
        if (hYear >= HIJRI_BEGIN && hYear <= HIJRI_END) {
            const pos = hYear - HIJRI_BEGIN;
            return +MONTH_LENGTH[pos][hMonth - 1] + 29;
        }
        return super.getDaysPerMonth(hMonth, hYear);
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.0.2", ngImport: i0, type: NgbCalendarIslamicUmalqura, deps: null, target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.0.2", ngImport: i0, type: NgbCalendarIslamicUmalqura }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.0.2", ngImport: i0, type: NgbCalendarIslamicUmalqura, decorators: [{
            type: Injectable
        }] });
//# sourceMappingURL=data:application/json;base64,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